package com.liveperson.infra.preferences

import android.content.Context
import android.content.SharedPreferences
import androidx.annotation.VisibleForTesting
import com.liveperson.infra.auth.LPAuthenticationParams
import com.liveperson.infra.auth.LPAuthenticationType
import com.liveperson.infra.controller.DBEncryptionHelper
import com.liveperson.infra.log.LPLog.d
import com.liveperson.infra.model.Consumer
import com.liveperson.infra.utils.EncryptionVersion
import java.util.*
import kotlin.collections.HashSet

/**
 * Handles persistent storage of authentication details on a per-brand basis. Each brand's values
 * get a different key, tied to the brandId, that allow us to store and retrieve multiple different
 * brands' logged-in Consumers.
 */
class AuthPreferences private constructor(applicationContext: Context) {

	private var preferences: SharedPreferences? = applicationContext.getSharedPreferences(
			LP_AUTH_SHARED_PREF_FILENAME, Context.MODE_PRIVATE)

	companion object {
		private const val TAG = "AuthPreferences"

		private const val LP_AUTH_SHARED_PREF_FILENAME = "lp_auth_shared_pref"

		private const val KEY_IDP_DOMAIN = "idp_domain"
		private const val OLD_KEY_IDP_DOMAIN = "idp"

		private const val KEY_ACCOUNT_UN_AUTH_TOKEN = "account_un_auth_token"

		private const val KEY_CONSUMER_ID = "consumer_id"
		private const val KEY_ORIGINAL_CONSUMER_ID = "original_consumer_id"
		private const val KEY_LP_TOKEN = "lp_token"
		private const val KEY_AUTH_TYPE = "auth_type"
		private const val KEY_AUTH_KEY = "auth_key"
		private const val KEY_HOST_APP_JWT = "host_app_jwt"
		private const val KEY_HOST_APP_REDIRECT_URI = "host_app_redirect_uri"
		private const val KEY_PINNING_KEYS = "pinning_keys"

		private var instance: AuthPreferences? = null

		// Create singleton instance
		fun getInstance(context: Context): AuthPreferences {
			if (instance == null) {
				instance = AuthPreferences(context)
			}
			return instance as AuthPreferences
		}
	}

	/**
	 * Set mocked SharedPreferences object. Used only for testing.
	 */
	@VisibleForTesting
	fun setPreferenceDelegate(prefs: SharedPreferences?) {
		preferences = prefs
	}

	/**
	 * Store a Consumer's Un-Auth token (encrypted at rest)
	 */
	fun setUnAuthToken(brandId: String?, value: String?) {
		preferences?.edit()?.putString(buildKey(KEY_ACCOUNT_UN_AUTH_TOKEN, brandId),
				DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, value))?.apply()
	}

	/**
	 * Retrieve a Consumer's Un-Auth token (automatically decrypted)
	 */
	fun getUnAuthToken(brandId: String?, defaultValue: String?): String? {
		return DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
				preferences?.getString(buildKey(KEY_ACCOUNT_UN_AUTH_TOKEN, brandId), defaultValue))
	}

	/**
	 * Stores a Consumer object for a Brand in Preferences (encrypted at rest)
	 */
	fun setCachedConsumer(brandId: String, consumer: Consumer?) {
		val editor = preferences?.edit()
		if (consumer == null) {

			editor?.remove(buildKey(KEY_CONSUMER_ID, brandId))
					?.remove(buildKey(KEY_ORIGINAL_CONSUMER_ID, brandId))
					?.remove(buildKey(KEY_LP_TOKEN, brandId))
					?.remove(buildKey(KEY_AUTH_TYPE, brandId))
					?.remove(buildKey(KEY_AUTH_KEY, brandId))
					?.remove(buildKey(KEY_HOST_APP_JWT, brandId))
					?.remove(buildKey(KEY_HOST_APP_REDIRECT_URI, brandId))
					?.remove(buildKey(KEY_PINNING_KEYS, brandId))

		} else {
			editor?.putString(buildKey(KEY_CONSUMER_ID, brandId),
					DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, consumer.consumerId))
					?.putString(buildKey(KEY_ORIGINAL_CONSUMER_ID, brandId),
							DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, consumer.originalConsumerId))
					?.putString(buildKey(KEY_LP_TOKEN, brandId),
							DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, consumer.lpToken))

			if (consumer.lpAuthenticationParams != null) {

				editor?.putInt(buildKey(KEY_AUTH_TYPE, brandId),
						consumer.lpAuthenticationParams.authType.storageVal)

						?.putString(buildKey(KEY_AUTH_KEY, brandId),
								DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1,
										consumer.lpAuthenticationParams.authKey))

						?.putString(buildKey(KEY_HOST_APP_JWT, brandId),
								DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1,
										consumer.lpAuthenticationParams.hostAppJWT))

						?.putString(buildKey(KEY_HOST_APP_REDIRECT_URI, brandId),
								DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1,
										consumer.lpAuthenticationParams.hostAppRedirectUri))

				val keys = HashSet<String>()
				for (key in consumer.lpAuthenticationParams.certificatePinningKeys) {
					keys.add(DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, key))
				}
				editor?.putStringSet(buildKey(KEY_PINNING_KEYS, brandId), keys)
			}
		}

		editor?.apply()
	}

	/**
	 * Retrieves the Consumer object for a Brand (automatically decrypted)
	 */
	fun getCachedConsumer(brandId: String?): Consumer? {
		if (brandId.isNullOrBlank()) return null

		var authParams: LPAuthenticationParams? = null
		var consumer: Consumer? = null

		if (preferences?.contains(buildKey(KEY_AUTH_TYPE, brandId)) == true) {
			val authType = LPAuthenticationType.fromStorageVal(
					preferences?.getInt(buildKey(KEY_AUTH_TYPE, brandId),
							LPAuthenticationType.SIGN_UP.storageVal)
							?: LPAuthenticationType.SIGN_UP.storageVal)
			// null-safe defaulting

			authParams = LPAuthenticationParams(authType)

			authParams.authKey = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
					preferences?.getString(buildKey(KEY_AUTH_KEY, brandId), ""))
			authParams.hostAppJWT = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
					preferences?.getString(buildKey(KEY_HOST_APP_JWT, brandId), ""))
			authParams.hostAppRedirectUri = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
					preferences?.getString(buildKey(KEY_HOST_APP_REDIRECT_URI, brandId), ""))

			for (key in preferences?.getStringSet(
					buildKey(KEY_PINNING_KEYS, brandId), Collections.emptySet())
					?: Collections.emptySet()) {

				authParams.addCertificatePinningKey(
						DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1, key))
			}
		}

		val consumerId = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
				preferences?.getString(buildKey(KEY_CONSUMER_ID, brandId), "")) ?: ""
		val originalConsumerId = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
				preferences?.getString(buildKey(KEY_ORIGINAL_CONSUMER_ID, brandId), "")) ?: ""
		val lpToken = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
				preferences?.getString(buildKey(KEY_LP_TOKEN, brandId), "")) ?: ""

		if (consumerId.isNotEmpty() && lpToken.isNotEmpty()) {
			consumer = Consumer(authParams, brandId, consumerId, originalConsumerId, lpToken)
		}

		return consumer
	}

	private fun buildKey(key: String?, brandId: String?): String? {
		return "$key$$$brandId"
	}

	/**
	 * Clear all entries from Auth preferences
	 */
	fun clearAll() {
		d(TAG, "clearAll: Clearing all data of Auth preferences")
		preferences?.edit()?.clear()?.apply()
	}
}