package com.liveperson.infra.utils;

import android.content.Context;
import android.net.Uri;
import android.os.AsyncTask;
import android.os.ParcelFileDescriptor;
import androidx.annotation.Nullable;

import com.liveperson.infra.log.LPLog;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;

import static com.liveperson.infra.errors.ErrorCode.ERR_00000012;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000013;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000014;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000142;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000143;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000144;

public class FileUtils {
	private static final String TAG = "FileUtils";
	private static final String DELETE_COMMAND = "rm -rf";
	private static final String[] FILE_DIRS = {"/images", "/voice", "/documents"};

    /**
     * Write the contents of internal file to the external file created.
     * @param context application context
     * @param privateDirectory app's private directory from where you need to copy the file
     * @param publicDirectory device's public directory (downloads/drive) where file is being copied to
     * @return
     */
    public static boolean writeFileContent(Context context, File privateDirectory, Uri publicDirectory)
    {
        FileInputStream inputStream = null;
        ParcelFileDescriptor parcelFileDescriptor = null;
        FileOutputStream fileOutputStream = null;
        try {
            inputStream = new FileInputStream(privateDirectory);
            parcelFileDescriptor = context.getContentResolver().openFileDescriptor(publicDirectory, "w");
            fileOutputStream = new FileOutputStream(parcelFileDescriptor.getFileDescriptor());

            byte[] b = new byte[1024 * 5];
            int len;

            while ((len = inputStream.read(b)) > 0) {
                fileOutputStream.write(b, 0, len);
            }
        } catch (Exception e) {
			LPLog.INSTANCE.e(TAG, ERR_00000012, "Failed to copy file from privateDir to publicDir. ", e);
            return false;
        } finally {
            try {
                if (inputStream != null) {
                    inputStream.close();
                }
                if (fileOutputStream != null) {
                    fileOutputStream.close();
                }
                if (parcelFileDescriptor != null) {
                    parcelFileDescriptor.close();
                }
            } catch (IOException exception) {
				LPLog.INSTANCE.e(TAG, ERR_00000013, "Failed to close fileOutputStream/parcelFileDescriptor ", exception);
            }
        }
        return true;
    }

    /**
     * Create new directory inside app's private directory and file if does not exists
     * @param context application context
     * @param path location of file to be created
     * @param filename Name of file to be created
     */
    @Nullable
    public static File getFilePath(Context context, String path, String filename) {
        // Define directory path and file path
        File privateDir = new File(context.getApplicationContext().getFilesDir() + path);
        File filePath = new File(privateDir, filename);

        // Create directory for the current brandId
        if (!privateDir.exists()) {
            if (!privateDir.mkdirs()) {
                LPLog.INSTANCE.e(TAG, ERR_00000014, "getFilePath: folder could not be created");
                return null;
            }
        }
        return filePath;
    }

    /**
     * Delete all downloaded files.
     */
    public static void deleteAllSharedFiles(Context context) {
        LPLog.INSTANCE.d(TAG, "deleting all shared files with agent");
        String basePath = context.getFilesDir().getPath();
        new DeleteFileAsyncTask(basePath).execute();
    }

    /**
     * Delete files in background.
     */
    private static class DeleteFileAsyncTask extends AsyncTask<Void, Void, Void> {

        private String basePath;

        public DeleteFileAsyncTask(String path) {
            this.basePath = path;
        }

        @Override
        protected Void doInBackground(Void... voids) {
            try {
                StringBuilder command = new StringBuilder(DELETE_COMMAND);
                for (String fileDir : FILE_DIRS) {
                    String fullPath = basePath + fileDir;
                    command.append(" ").append(fullPath);
                }
                LPLog.INSTANCE.d(TAG, "deleting command: " + command.toString());
                Runtime.getRuntime().exec(command.toString());
            } catch (IOException e) {
                LPLog.INSTANCE.e(TAG, ERR_00000142, "Failed to delete files using command line.", e);
                for (String fileDir : FILE_DIRS) {
                    String fullPath = basePath + fileDir;
                    deleteRecursively(new File(fullPath));
                }
            }
            return null;
        }

        private void deleteRecursively(File fileOrDirectory) {
            if (fileOrDirectory == null) {
                LPLog.INSTANCE.e(TAG, ERR_00000143, "file path is null");
                return;
            }
            if (fileOrDirectory.isDirectory()) {
                LPLog.INSTANCE.d(TAG, "deleteRecursive: deleting directory: " + fileOrDirectory.getAbsolutePath());
                File[] listFiles = fileOrDirectory.listFiles();
                if (listFiles == null) {
                    LPLog.INSTANCE.e(TAG, ERR_00000144, "File list is null");
                    return;
                }
                for (File child : fileOrDirectory.listFiles()) {
                    LPLog.INSTANCE.d(TAG, "deleteRecursive: deleting file: " + child.getAbsolutePath());
                    deleteRecursively(child);
                }
            }
            //noinspection ResultOfMethodCallIgnored
            fileOrDirectory.delete();
        }
    }
}
