package com.liveperson.infra.network.http.requests;

import android.text.TextUtils;

import com.liveperson.infra.Command;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.model.LptagData;
import com.liveperson.infra.network.http.HttpHandler;
import com.liveperson.infra.network.http.request.HttpGetRequest;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

import javax.net.ssl.SSLPeerUnverifiedException;

/**
 * Created by nirni on 9/11/17.
 * <p/>
 * A request to fetch data from LPTag
 *
 * Since this request should NOT block the connection process, we always return onSuccess()
 */
public class LptagRequest implements Command {

	private static final String TAG = "LptagRequest";
    private static final String LPTAG_URL = "https://%1$s/lptag/api/account/%2$s/configuration/applications/taglets/?v=2.0&scp=mb";
    private static final int LPTAG_REQUEST_TIMEOUT = 30000;
	private static final String JSON_KEY_FEATURES = "features";
	private static final String JSON_KEY_RETRY = "retry";
	private static final String JSON_KEY_RETRY_TIMEOUT = "timeout";
	private static final String JSON_KEY_MAX_RETRIES = "maxRetries";
	private static final String JSON_KEY_MESSAGING_AUTO_MESSAGES = "Messaging.Auto_Messages";

	private ICallback<LptagData, Exception> mCallback;
    private String mBrandId;
    private String mDomain;
    private List<String> mCertificates;

    public LptagRequest(String domain, String brandId, List<String> certificates, ICallback<LptagData, Exception> callback) {
        mCallback = callback;
        mBrandId = brandId;
        mDomain = domain;
		mCertificates = certificates;
    }


    /**
	 *	If return a Retry structure such as:
	 *	{
	 *		retry: {
	 *		timeout: 1000,
	 *		maxRetries: 5
	 *		}
	 *	}
	 *
	 *	In this case we return LptagData object with retry in formation so the caller can retry
	 */
	@Override
    public void execute() {

        final String lptagRequestUrl = String.format(LPTAG_URL, mDomain, mBrandId);
        HttpGetRequest httpGetRequest = new HttpGetRequest(lptagRequestUrl);
        httpGetRequest.setTimeout(LPTAG_REQUEST_TIMEOUT);
        httpGetRequest.setCertificatePinningKeys(mCertificates);
        httpGetRequest.setCallback(new ICallback<String, Exception>() {
            @Override
            public void onSuccess(String lptagString) {

				LPLog.INSTANCE.d(TAG, "onSuccess: received response: " + lptagString);
				LptagData lptagData = new LptagData(); // LptagData is initiated with AutoMessage = false

				if(!TextUtils.isEmpty(lptagString)){

                    try{
                        JSONObject lptagJson = new JSONObject(lptagString);
						// Try to get Features object from JSON
                        JSONObject featuresJsonObject = lptagJson.optJSONObject(JSON_KEY_FEATURES);

						// If we don't have the 'features' key
                        if (featuresJsonObject == null) {

							// Try to get the 'retry' block and return LptagData with a retry info
							JSONObject retry = lptagJson.optJSONObject(JSON_KEY_RETRY);
							if (retry != null) {
								LPLog.INSTANCE.d(TAG, "onSuccess: Received Retry json. Parse and return");
								extractRetryFromJson(lptagData, retry);
							}
							mCallback.onSuccess(lptagData);
						}
						else{ // We got features info. Parse and return it.

							LPLog.INSTANCE.d(TAG, "onSuccess: Received Features json. Parse and return");
							extractFeaturesFromJson(lptagData, featuresJsonObject);
							mCallback.onSuccess(lptagData);
						}

					} catch (JSONException je) {

						LPLog.INSTANCE.w(TAG, "Error parsing LPTag data. AutoMessage is Off. Return success", je);
						mCallback.onSuccess(lptagData);

					}
                }
                else {
					LPLog.INSTANCE.w(TAG, "LPTag failed! lptagString is empty. AutoMessage is Off. Return success");
					mCallback.onSuccess(lptagData);
				}
            }

            @Override
            public void onError(Exception exception) {
				LPLog.INSTANCE.w(TAG, "LPTag failed! Received error. AutoMessage is Off. Return success", exception);
				if(exception instanceof SSLPeerUnverifiedException){
					mCallback.onError(exception);
					return;
				}
				mCallback.onSuccess(new LptagData());
            }
        });
        HttpHandler.execute(httpGetRequest);
    }

	/**
	 * Extract Retry info from the given JSON
	 */
	private void extractRetryFromJson(LptagData lptagData, JSONObject retry) {
		lptagData.setRetryTimeout(retry.optInt(JSON_KEY_RETRY_TIMEOUT, 0));
		lptagData.setMaxRetries(retry.optInt(JSON_KEY_MAX_RETRIES, 0));

		// Rest AutoMessages param
		lptagData.setAutoMessagesFeatureEnabled(false);
	}

	/**
	 * Extract required features from the Features JSON
	 */
	private void extractFeaturesFromJson(LptagData lptagData, JSONObject featuresJsonObject) {

		lptagData.setAutoMessagesFeatureEnabled(featuresJsonObject.optBoolean(JSON_KEY_MESSAGING_AUTO_MESSAGES));

		// Reset retry params
		lptagData.setRetryTimeout(0);
		lptagData.setMaxRetries(0);

	}

}
