package com.liveperson.lpdatepicker

import android.os.Bundle
import androidx.appcompat.app.AppCompatActivity
import android.view.Gravity
import android.view.View
import android.widget.AdapterView
import android.widget.ArrayAdapter
import com.liveperson.lpdatepicker.calendar.models.LPICalendarStyleAttributes
import com.liveperson.lpdatepicker.calendar.views.LPICalendarListener
import com.liveperson.lpdatepicker.calendar.views.LPICalendarPageChangedListener
import com.liveperson.lpdatepicker.databinding.ActivityDatePickerBinding
import java.util.*

class DatePickerActivity : AppCompatActivity() {

	private lateinit var binding: ActivityDatePickerBinding
	private val yearList: MutableList<Int> = ArrayList()
	private val monthList: MutableList<String> = ArrayList()
	private lateinit var currentDate: Calendar
	private var selectedStartDate: Long? = null
	private var selectedEndDate: Long? = null
	private var dateFormat: String = ""

	override fun onCreate(savedInstanceState: Bundle?) {
		super.onCreate(savedInstanceState)
		binding = ActivityDatePickerBinding.inflate(layoutInflater)
		setContentView(binding.root)

		if (intent.hasExtra(CALENDAR_INIT_INFO)) {
			val initInfo: CalendarInitInfo? = intent.getParcelableExtra(CALENDAR_INIT_INFO)
			if (initInfo == null) {
				closeDatePicker()
			}
			initViews(initInfo!!)
		} else {
			closeDatePicker()
		}
	}

	private fun initViews(initInfo: CalendarInitInfo) {
		dateFormat = initInfo.dateFormat
		binding.apply {
			tvHeader.text = initInfo.title
			val minDate: Calendar = CalendarUtils.getCalendarDate(initInfo.minDate)
			val maxDate: Calendar = CalendarUtils.getCalendarDate(initInfo.maxDate)
			currentDate = minDate.clone() as Calendar
			val allMonthNames = CalendarUtils.getMonthNameList(this@DatePickerActivity)

			monthList.addAll(CalendarUtils.getMonthRange(minDate, maxDate, currentDate, this@DatePickerActivity))
			val monthAdapter = ArrayAdapter(
				this@DatePickerActivity,
				R.layout.lp_spinner_item,
				monthList
			)
			spMonth.adapter = monthAdapter
			spMonth.onItemSelectedListener = object : AdapterView.OnItemSelectedListener {
				override fun onItemSelected(
					parent: AdapterView<*>?,
					view: View?,
					position: Int,
					id: Long
				) {
					val selectedMonth = allMonthNames.indexOf(monthList[position])
					currentDate[Calendar.MONTH] = selectedMonth
					calendar.setCurrentMonth(currentDate)
				}

				override fun onNothingSelected(parent: AdapterView<*>?) {

				}

			}

			yearList.addAll(CalendarUtils.getYearRange(minDate, maxDate))
			val yearAdapter = ArrayAdapter(
				this@DatePickerActivity,
				R.layout.lp_spinner_item,
				yearList
			)
			spYear.adapter = yearAdapter
			spYear.onItemSelectedListener = object : AdapterView.OnItemSelectedListener {
				override fun onItemSelected(
					parent: AdapterView<*>?,
					view: View?,
					position: Int,
					id: Long
				) {
					currentDate[Calendar.YEAR] = yearList[position]
					var scrollTo = 0
					if (currentDate.before(minDate)) {
						currentDate = minDate.clone() as Calendar
						scrollTo = 1
					} else if (currentDate.after(maxDate)) {
						currentDate = maxDate.clone() as Calendar
						scrollTo = 2
					}
					monthList.clear()
					monthList.addAll(CalendarUtils.getMonthRange(minDate, maxDate, currentDate, this@DatePickerActivity))
					calendar.setCurrentMonth(currentDate)
					when (scrollTo) {
						1 -> spMonth.setSelection(0)
						2 -> spMonth.setSelection(monthList.lastIndex)
						else -> spMonth.setSelection(
							monthList.indexOf(
								CalendarUtils.getMonthText(currentDate, this@DatePickerActivity)
							)
						)
					}
					monthAdapter.notifyDataSetChanged()
				}

				override fun onNothingSelected(parent: AdapterView<*>?) {

				}

			}

			calendar.apply {

				setVisibleMonthRange(minDate, maxDate)
				setSelectableDateRange(minDate, maxDate)

				calendar.setCurrentMonth(minDate)
				calendar.setSelectMode(LPICalendarStyleAttributes.DateSelectionMode.FREE_RANGE)
				calendar.setSelectMode(
					when (initInfo.type) {
						Type.RANGE -> LPICalendarStyleAttributes.DateSelectionMode.FREE_RANGE
						else -> LPICalendarStyleAttributes.DateSelectionMode.SINGLE
					}
				)
				setCalendarListener(object : LPICalendarListener {
					override fun onFirstDateSelected(startDate: Calendar) {
					}

					override fun onDateRangeSelected(startDate: Calendar, endDate: Calendar) {
						bottomWrapper.visibility = View.VISIBLE
						selectedStartDate = startDate.timeInMillis / 1000
						selectedEndDate = endDate.timeInMillis / 1000

						if (initInfo.type == Type.RANGE) {
							tvEndDate.visibility = View.VISIBLE
							tvEndDayOfWeek.visibility = View.VISIBLE
							imgArrow.visibility = View.VISIBLE

							tvStartDate.text = CalendarUtils.getDateText(startDate, getString(R.string.lp_datepicker_date_text_format), this@DatePickerActivity)
							tvStartDayOfWeek.text = CalendarUtils.getDayOfWeek(startDate, this@DatePickerActivity)
							tvEndDate.text = CalendarUtils.getDateText(endDate, getString(R.string.lp_datepicker_date_text_format), this@DatePickerActivity)
							tvEndDayOfWeek.text = CalendarUtils.getDayOfWeek(endDate, this@DatePickerActivity)
						} else {
							tvStartDate.gravity = Gravity.CENTER_HORIZONTAL
							tvStartDayOfWeek.gravity = Gravity.CENTER_HORIZONTAL
							tvStartDate.text = CalendarUtils.getDateText(startDate, getString(R.string.lp_datepicker_date_text_format), this@DatePickerActivity)
							tvStartDayOfWeek.text = CalendarUtils.getDayOfWeek(startDate, this@DatePickerActivity)
						}
					}
				})
				setOnPageChangeListener(object : LPICalendarPageChangedListener {
					override fun onCalendarChanged(calendar: Calendar) {
						val yearChanged = calendar[Calendar.YEAR] != currentDate[Calendar.YEAR]
						currentDate = calendar.clone() as Calendar
						if (yearChanged) {
							monthList.clear()
							monthList.addAll(
								CalendarUtils.getMonthRange(
									minDate,
									maxDate,
									currentDate,
									this@DatePickerActivity
								)
							)
						}
						spMonth.setSelection(
							monthList.indexOf(
								CalendarUtils.getMonthText(
									currentDate,
									this@DatePickerActivity
								)
							)
						)
						monthAdapter.notifyDataSetChanged()
						spYear.setSelection(yearList.indexOf(currentDate[Calendar.YEAR]))
						yearAdapter.notifyDataSetChanged()
					}
				})
			}

			imgClose.setOnClickListener {
				closeDatePicker()
			}

			btnConfirm.setOnClickListener {
				if (calendar.startDate == null || calendar.endDate == null) {
					setResult(RESULT_CANCELED)
				} else {
					intent.putExtra(RESULT_START_DATE_IN_SECONDS, selectedStartDate)
					intent.putExtra(RESULT_END_DATE_IN_SECONDS, selectedEndDate)
					setResult(RESULT_OK, intent)
				}
				finish()
			}
		}
	}

	private fun closeDatePicker() {
		setResult(RESULT_CANCELED)
		finish()
	}

	companion object {
		const val CALENDAR_INIT_INFO = "CALENDAR_INIT_INFO"
		const val RESULT_START_DATE_IN_SECONDS = "RESULT_START_DATE_IN_SECONDS"
		const val RESULT_END_DATE_IN_SECONDS = "RESULT_END_DATE_IN_SECONDS"
	}
}
