package com.liveperson.lpdatepicker.calendar.views

import android.content.Context
import android.graphics.Color
import android.graphics.Typeface
import androidx.core.content.ContextCompat
import androidx.appcompat.widget.AppCompatTextView
import android.util.AttributeSet
import android.util.TypedValue
import android.view.LayoutInflater
import android.view.View
import android.view.View.OnClickListener
import android.widget.FrameLayout
import com.liveperson.lpdatepicker.CalendarUtils
import com.liveperson.lpdatepicker.R
import com.liveperson.lpdatepicker.R.*
import com.liveperson.lpdatepicker.calendar.models.LPCalendarStyleAttrImpl
import com.liveperson.lpdatepicker.calendar.models.LPICalendarStyleAttributes
import com.liveperson.lpdatepicker.calendar.views.LPIDateView.DateState
import com.liveperson.lpdatepicker.calendar.views.LPIDateView.DateState.*
import com.liveperson.lpdatepicker.calendar.views.LPIDateView.OnDateClickListener
import java.text.ParseException
import java.text.SimpleDateFormat
import java.util.*


class LPCustomDateView @JvmOverloads constructor(
    context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0
) : FrameLayout(context, attrs, defStyleAttr), LPIDateView {

    private val tvDate: AppCompatTextView
    private val strip: View
    private val simpleDateFormat =
        SimpleDateFormat(LPICalendarDateRangeManager.DATE_FORMAT, CalendarUtils.getLocale(context))

    private var onDateClickListener: OnDateClickListener? = null
    private var mDateState: DateState
    private val isRightToLeft = resources.getBoolean(bool.lp_datepicker_is_right_to_left)

    init {
        val inflater = context.getSystemService(Context.LAYOUT_INFLATER_SERVICE) as LayoutInflater
        inflater.inflate(layout.lp_layout_calendar_day, this, true)
        tvDate = findViewById(R.id.dayOfMonthText)
        strip = findViewById(R.id.viewStrip)
        mDateState = SELECTABLE
        if (!isInEditMode) {
            setDateStyleAttributes(LPCalendarStyleAttrImpl.getDefAttributes(context))
            updateDateBackground(mDateState)
        }
    }

    private val defCalendarStyleAttr: LPCalendarStyleAttrImpl =
        LPCalendarStyleAttrImpl.getDefAttributes(context)
    override var dateTextSize: Float = defCalendarStyleAttr.textSizeDate
    override var defaultDateColor: Int = defCalendarStyleAttr.defaultDateColor
    override var disableDateColor: Int = defCalendarStyleAttr.disableDateColor
    override var selectedDateCircleColor: Int = defCalendarStyleAttr.selectedDateCircleColor
    override var selectedDateColor: Int = defCalendarStyleAttr.selectedDateColor
    override var rangeDateColor: Int = defCalendarStyleAttr.rangeDateColor
    override var stripColor: Int = defCalendarStyleAttr.rangeStripColor

    private val mViewClickListener = OnClickListener {
        val key = it.tag as Long
        if (onDateClickListener != null) {
            val selectedCal = Calendar.getInstance()
            var date = Date()
            try {
                date = simpleDateFormat.parse(key.toString())
            } catch (e: ParseException) {
                e.printStackTrace()
            }
            selectedCal.time = date
            onDateClickListener?.onDateClicked(it, selectedCal)
        }
    }

    override fun setDateText(date: String) {
        tvDate.text = date
    }

    override fun setDateStyleAttributes(attr: LPICalendarStyleAttributes) {
        disableDateColor = attr.disableDateColor
        defaultDateColor = attr.defaultDateColor
        selectedDateCircleColor = attr.selectedDateCircleColor
        selectedDateColor = attr.selectedDateColor
        stripColor = attr.rangeStripColor
        rangeDateColor = attr.rangeDateColor
        tvDate.textSize = attr.textSizeDate
        refreshLayout()
    }

    override fun setTypeface(typeface: Typeface) {
        tvDate.typeface = typeface
    }

    override fun setDateTag(date: Calendar) {
        tag = LPIDateView.getContainerKey(date)
    }

    override fun updateDateBackground(dateState: DateState) {
        mDateState = dateState
        when (dateState) {
            START, END, START_END_SAME -> makeAsSelectedDate(dateState)
            HIDDEN -> hideDayContainer()
            SELECTABLE -> enabledDayContainer()
            DISABLE -> disableDayContainer()
            MIDDLE -> makeAsRangeDate()
        }
    }

    override fun refreshLayout() {
        tvDate.setTextSize(TypedValue.COMPLEX_UNIT_PX, dateTextSize)
    }

    override fun setDateClickListener(listener: OnDateClickListener) {
        onDateClickListener = listener
    }

    /**
     * To hide date if date is from previous month.
     */
    private fun hideDayContainer() {
        tvDate.text = ""
        tvDate.setBackgroundColor(Color.TRANSPARENT)
        strip.setBackgroundColor(Color.TRANSPARENT)
        setBackgroundColor(Color.TRANSPARENT)
        visibility = View.INVISIBLE
        setOnClickListener(null)
    }

    /**
     * To disable past date. Click listener will be removed.
     */
    private fun disableDayContainer() {
        tvDate.setBackgroundColor(Color.TRANSPARENT)
        strip.setBackgroundColor(Color.TRANSPARENT)
        setBackgroundColor(Color.TRANSPARENT)
        tvDate.setTextColor(disableDateColor)
        visibility = View.VISIBLE
        setOnClickListener(null)
    }

    /**
     * To enable date by enabling click listeners.
     */
    private fun enabledDayContainer() {
        tvDate.setBackgroundColor(Color.TRANSPARENT)
        strip.setBackgroundColor(Color.TRANSPARENT)
        setBackgroundColor(Color.TRANSPARENT)
        tvDate.setTextColor(defaultDateColor)
        visibility = View.VISIBLE
        setOnClickListener(mViewClickListener)
    }

    /**
     * To draw date container as selected as end selection or middle selection.
     *
     * @param state - DateState
     */
    private fun makeAsSelectedDate(state: DateState) {
        when (state) {
            START_END_SAME -> {
                val layoutParams = strip.layoutParams as LayoutParams
                strip.setBackgroundColor(Color.TRANSPARENT)
                layoutParams.setMargins(0, 0, 0, 0)
                strip.layoutParams = layoutParams
            }
            START -> {
                if (isRightToLeft) {
                    setRightFacedSelectedDate()
                } else {
                    setLeftFacedSelectedDate()
                }
            }
            END -> {
                if (isRightToLeft) {
                    setLeftFacedSelectedDate()
                } else {
                    setRightFacedSelectedDate()
                }
            }
            else -> {
                throw IllegalArgumentException("$state is an invalid state.")
            }
        }
        val mDrawable = ContextCompat.getDrawable(context, drawable.lp_selected_date_bg)
        tvDate.background = mDrawable
        setBackgroundColor(Color.TRANSPARENT)
        tvDate.setTextColor(selectedDateColor)
        visibility = View.VISIBLE
        setOnClickListener(mViewClickListener)
    }

    private fun setLeftFacedSelectedDate() {
        val layoutParams = strip.layoutParams as LayoutParams
        val drawable = ContextCompat.getDrawable(context, drawable.lp_range_bg_left)
        strip.background = drawable
        layoutParams.setMargins(70, 0, 0, 0)
        strip.layoutParams = layoutParams
    }

    private fun setRightFacedSelectedDate() {
        val layoutParams = strip.layoutParams as LayoutParams
        val drawable = ContextCompat.getDrawable(context, drawable.lp_range_bg_right)
        strip.background = drawable
        layoutParams.setMargins(0, 0, 70, 0)
        strip.layoutParams = layoutParams
    }

    /**
     * To draw date as middle date
     */
    private fun makeAsRangeDate() {
        tvDate.setBackgroundColor(Color.TRANSPARENT)
        val mDrawable = ContextCompat.getDrawable(context, drawable.lp_range_bg)
        strip.background = mDrawable
        setBackgroundColor(Color.TRANSPARENT)
        tvDate.setTextColor(rangeDateColor)
        visibility = View.VISIBLE
        val layoutParams = strip.layoutParams as LayoutParams
        layoutParams.setMargins(0, 0, 0, 0)
        strip.layoutParams = layoutParams
        setOnClickListener(mViewClickListener)
    }
}