package com.liveperson.lpdatepicker

import android.content.Context
import android.os.Build
import java.text.DateFormatSymbols
import java.text.SimpleDateFormat
import java.util.*
import kotlin.collections.ArrayList

object CalendarUtils {

	private const val DAY_OF_WEEK_FORMAT = "EEEE" //Sunday, Monday
	private const val DAY_OF_WEEK_TITLE_FORMAT = "EEEEE" //S, M, T
	private const val MONTH_NAME_FORMAT = "MMMM" //January, February

	/**
	 * return the text in [DAY_OF_WEEK_FORMAT]
	 */
	fun getDayOfWeek(calendar: Calendar, context: Context): String {
		val dateFormat = SimpleDateFormat(DAY_OF_WEEK_FORMAT, getLocale(context))
		return dateFormat.format(calendar.time)
	}

	/**
	 * @param calendar The calendar object
	 * @param format The format to return
	 *
	 * return formatted date text
	 */
	fun getDateText(calendar: Calendar, format: String, context: Context): String {
		val dateFormat = SimpleDateFormat(format, getLocale(context))
		return dateFormat.format(calendar.time)
	}

	/**
	 * return the text in [MONTH_NAME_FORMAT]
	 */
	fun getMonthText(calendar: Calendar, context: Context): String {
		val dateFormat = SimpleDateFormat(MONTH_NAME_FORMAT, getLocale(context))
		return dateFormat.format(calendar.time)
	}

	/**
	 * return the Calendar object of time.
	 */
	fun getCalendarDate(time: Long): Calendar {
		val calendar = Calendar.getInstance()
		calendar.timeInMillis = time
		return calendar
	}

	/**
	 * @param minDate Minimum valid date.
	 * @param maxDate Maximum valid date.
	 * @param currentDate currentVisible date in the date picker calendar.
	 *
	 * return available month range. It is used in the month spinner.
	 */
	fun getMonthRange(minDate: Calendar, maxDate: Calendar, currentDate: Calendar, context: Context): List<String> {
		val monthRange = ArrayList<Int>()
		if (minDate[Calendar.YEAR] == maxDate[Calendar.YEAR]) {
			var month = minDate[Calendar.MONTH]
			while (month <= maxDate[Calendar.MONTH]) {
				monthRange.add(month++)
			}
		} else {
			when {
				minDate[Calendar.YEAR] == currentDate[Calendar.YEAR] -> {
					var month = minDate[Calendar.MONTH]
					while (month <= Calendar.DECEMBER) {
						monthRange.add(month++)
					}
				}
				maxDate[Calendar.YEAR] == currentDate[Calendar.YEAR] -> {
					var month = Calendar.JANUARY
					while (month <= maxDate[Calendar.MONTH]) {
						monthRange.add(month++)
					}
				}
				else -> {
					return getMonthNameList(context)
				}
			}
		}

		val monthNameRange = ArrayList<String>()
		val monthNames = getMonthNameList(context)
		for (i in monthRange) {
			monthNameRange.add(monthNames[i])
		}
		return monthNameRange
	}

	/**
	 * @param minDate Minimum valid date.
	 * @param maxDate Maximum valid date.
	 *
	 * return the year list. It is used in year spinner.
	 */
	fun getYearRange(minDate: Calendar, maxDate: Calendar): List<Int> {
		val yearRange = ArrayList<Int>()
		var year = minDate[Calendar.YEAR]
		while (year <= maxDate[Calendar.YEAR]) {
			yearRange.add(year++)
		}
		return yearRange
	}

	/**
	 * Return month names. It's formatted by SimpleDateFormat. It should support all languages in Android.
	 */
	fun getMonthNameList(context: Context): List<String> {
		val monthNames: MutableList<String> = ArrayList()
		val calendar = Calendar.getInstance()
		calendar.set(Calendar.MONTH, Calendar.JANUARY)
		for (i in 0..11) {
			monthNames.add(getMonthText(calendar, context))
			calendar.add(Calendar.MONTH, 1)
		}
		return monthNames
	}

	/**
	 * Return week names starts from Sunday. It's formatted by SimpleDateFormat. It should support all languages in Android.
	 */
	fun getDayOfWeekNameList(context: Context): List<String> {
		val dayOfWeekNameList: MutableList<String> = ArrayList()
		val dateFormat = SimpleDateFormat(DAY_OF_WEEK_TITLE_FORMAT, getLocale(context))
		val dayOfWeeks = arrayOf(
			Calendar.SUNDAY,
			Calendar.MONDAY,
			Calendar.TUESDAY,
			Calendar.WEDNESDAY,
			Calendar.THURSDAY,
			Calendar.FRIDAY,
			Calendar.SATURDAY
		)
		val calendar = Calendar.getInstance()
		for (i in dayOfWeeks) {
			calendar.set(Calendar.DAY_OF_WEEK, i)
			dayOfWeekNameList.add(dateFormat.format(calendar.time))
		}
		return dayOfWeekNameList
	}

	/**
	 * Return week names starts from Sunday. It's formatted by SimpleDateFormat. It should support all languages in Android.
	 */
	fun getDayOfWeekFullNameList(context: Context): Array<String> {
		return DateFormatSymbols.getInstance(getLocale(context)).weekdays
	}
	fun getLocale(context: Context) : Locale {
		return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
			context.resources.configuration.locales.get(0)
		} else {
			context.resources.configuration.locale
		}
	}
}
