package com.liveperson.lpdatepicker.calendar.views

import com.liveperson.lpdatepicker.calendar.views.LPICalendarDateRangeManager.DateSelectionState
import com.liveperson.lpdatepicker.calendar.views.LPICalendarDateRangeManager.DateSelectionState.IN_SELECTED_RANGE
import com.liveperson.lpdatepicker.calendar.views.LPICalendarDateRangeManager.DateSelectionState.LAST_DATE
import com.liveperson.lpdatepicker.calendar.views.LPICalendarDateRangeManager.DateSelectionState.START_DATE
import com.liveperson.lpdatepicker.calendar.views.LPICalendarDateRangeManager.DateSelectionState.START_END_SAME
import com.liveperson.lpdatepicker.calendar.views.LPICalendarDateRangeManager.DateSelectionState.UNKNOWN
import com.liveperson.lpdatepicker.calendar.views.DateTiming.END
import com.liveperson.lpdatepicker.calendar.views.DateTiming.START
import com.liveperson.lpdatepicker.calendar.models.LPICalendarStyleAttributes
import com.liveperson.lpdatepicker.calendar.models.LPICalendarStyleAttributes.DateSelectionMode
import com.liveperson.lpdatepicker.calendar.models.LPICalendarStyleAttributes.DateSelectionMode.FIXED_RANGE
import com.liveperson.lpdatepicker.calendar.models.LPICalendarStyleAttributes.DateSelectionMode.FREE_RANGE
import com.liveperson.lpdatepicker.calendar.models.LPICalendarStyleAttributes.DateSelectionMode.SINGLE
import java.util.Calendar

internal class LPCalendarDateRangeManagerImpl(
    startMonthDate: Calendar,
    endMonthDate: Calendar,
    private val calendarStyleAttributes: LPICalendarStyleAttributes
) : LPICalendarDateRangeManager {
    private lateinit var mStartVisibleMonth: Calendar
    private lateinit var mEndVisibleMonth: Calendar
    private lateinit var mStartSelectableDate: Calendar
    private lateinit var mEndSelectableDate: Calendar
    private var mMinSelectedDate: Calendar? = null
    private var mMaxSelectedDate: Calendar? = null
    private val mVisibleMonths = mutableListOf<Calendar>()

    companion object {
        private const val TAG = "CDRManagerImpl"
    }

    init {
        setVisibleMonths(startMonthDate, endMonthDate)
    }

    override fun getMaxSelectedDate(): Calendar? {
        return mMaxSelectedDate
    }

    override fun getMinSelectedDate(): Calendar? {
        return mMinSelectedDate
    }

    override fun getVisibleMonthDataList(): List<Calendar> {
        return mVisibleMonths
    }

    override fun getMonthIndex(month: Calendar): Int {
        for (i in mVisibleMonths.indices) {
            val item: Calendar = mVisibleMonths[i]
            if (month[Calendar.YEAR] == item.get(Calendar.YEAR)) {
                if (month[Calendar.MONTH] == item.get(Calendar.MONTH)) {
                    return i
                }
            }
        }
        throw RuntimeException("Month(" + month.time.toString() + ") is not available in the given month range.")
    }

    override fun setVisibleMonths(startMonth: Calendar, endMonth: Calendar) {
        validateDatesOrder(startMonth, endMonth)
        val startMonthDate = startMonth.clone() as Calendar
        val endMonthDate = endMonth.clone() as Calendar

        startMonthDate[Calendar.DAY_OF_MONTH] = 1
        resetTime(startMonthDate, START)

        endMonthDate[Calendar.DAY_OF_MONTH] = endMonthDate.getActualMaximum(Calendar.DAY_OF_MONTH)
        resetTime(endMonthDate, END)

        mStartVisibleMonth = startMonthDate.clone() as Calendar
        resetTime(mStartVisibleMonth, START)
        mEndVisibleMonth = endMonthDate.clone() as Calendar
        resetTime(mEndVisibleMonth, END)

        // Creating visible months data list
        mVisibleMonths.clear()
        val temp = mStartVisibleMonth.clone() as Calendar
        while (!isMonthSame(temp, mEndVisibleMonth)) {
            mVisibleMonths.add(temp.clone() as Calendar)
            temp.add(Calendar.MONTH, 1)
        }
        mVisibleMonths.add(temp.clone() as Calendar)
        setSelectableDateRange(mStartVisibleMonth, mEndVisibleMonth)
    }

    override fun getStartVisibleMonth() = mStartVisibleMonth

    override fun getEndVisibleMonth() = mEndVisibleMonth

    override fun setSelectableDateRange(startDate: Calendar, endDate: Calendar) {
        validateDatesOrder(startDate, endDate)
        mStartSelectableDate = startDate.clone() as Calendar
        resetTime(mStartSelectableDate, START)
        mEndSelectableDate = endDate.clone() as Calendar
        resetTime(mEndSelectableDate, END)
        if (mStartSelectableDate.before(mStartVisibleMonth)) {
            throw LPInvalidDateException(
                "Selectable start date ${printDate(startDate)} is out of visible months" +
                        "(${printDate(mStartVisibleMonth)} " +
                        "- ${printDate(mEndVisibleMonth)})."
            )
        }
        if (mEndSelectableDate.after(mEndVisibleMonth)) {
            throw LPInvalidDateException(
                "Selectable end date ${printDate(endDate)} is out of visible months" +
                        "(${printDate(mStartVisibleMonth)} " +
                        "- ${printDate(mEndVisibleMonth)})."
            )
        }
        resetSelectedDateRange()
    }

    override fun resetSelectedDateRange() {
        this.mMinSelectedDate = null
        this.mMaxSelectedDate = null
    }

    override fun setSelectedDateRange(startDate: Calendar, endDate: Calendar?) {
        validateDatesOrder(startDate, endDate)
        if (startDate.before(mStartSelectableDate)) {
            throw LPInvalidDateException("Start date(${printDate(startDate)}) is out of selectable date range.")
        }
        if (endDate?.after(mEndSelectableDate) == true) {
            throw LPInvalidDateException("End date(${printDate(endDate)}) is out of selectable date range.")
        }
        val selectionMode: DateSelectionMode = calendarStyleAttributes.dateSelectionMode
        val finalEndDate: Calendar?
        when (selectionMode) {
            SINGLE -> {
                finalEndDate = startDate.clone() as Calendar
            }
            FIXED_RANGE -> {
                finalEndDate = startDate.clone() as Calendar
                finalEndDate.add(Calendar.DATE, calendarStyleAttributes.fixedDaysSelectionNumber)
            }
            FREE_RANGE -> finalEndDate = endDate
        }
        this.mMinSelectedDate = startDate.clone() as Calendar
        this.mMaxSelectedDate = finalEndDate?.clone() as Calendar?
    }

    /**
     * To check whether date belongs to selected range.
     *
     * @return DateSelectionState state
     */
    override fun checkDateRange(selectedDate: Calendar): DateSelectionState {

        if (mMinSelectedDate != null && mMaxSelectedDate != null) {

            val selectedDateVal = LPIDateView.getContainerKey(selectedDate)
            val minDateVal = LPIDateView.getContainerKey(mMinSelectedDate!!)
            val maxDateVal = LPIDateView.getContainerKey(mMaxSelectedDate!!)

            if (isDateSame(selectedDate, mMinSelectedDate!!) && isDateSame(
                    selectedDate,
                    mMaxSelectedDate!!
                )
            ) {
                return START_END_SAME
            } else if (isDateSame(selectedDate, mMinSelectedDate!!)) {
                return START_DATE
            } else if (isDateSame(selectedDate, mMaxSelectedDate!!)) {
                return LAST_DATE
            } else if (selectedDateVal in minDateVal until maxDateVal) {
                return IN_SELECTED_RANGE
            }
        } else if (mMinSelectedDate != null) {
            // When only single date is selected
            if (isDateSame(selectedDate, mMinSelectedDate!!)) {
                return START_END_SAME
            }
        }
        return UNKNOWN
    }

    override fun isSelectableDate(date: Calendar): Boolean {
        // It would work even if date is exactly equal to one of the end cases
        val isSelectable = !(date.before(mStartSelectableDate) || date.after(mEndSelectableDate))
        if (!(!isSelectable && checkDateRange(date) !== UNKNOWN)) {
            "Selected date can not be out of Selectable Date range." +
                    " Date: ${printDate(date)}" +
                    " Min: ${printDate(mMinSelectedDate)}" +
                    " Max: ${printDate(mMaxSelectedDate)}"
        }
        return isSelectable
    }

    private fun validateDatesOrder(start: Calendar, end: Calendar?) {
        if (start.after(end)) {
            throw LPInvalidDateException(
                "Start date(${printDate(start)}) can not be after end date(${
                    printDate(
                        end
                    )
                })."
            )
        }
    }
}