package com.liveperson.messaging.commands;

import android.support.annotation.NonNull;

import com.liveperson.api.request.BaseAMSSocketRequest;
import com.liveperson.api.response.types.ConversationState;
import com.liveperson.api.response.model.ContentType;
import com.liveperson.api.response.types.DialogState;
import com.liveperson.infra.database.DBUtilities;
import com.liveperson.infra.database.DataBaseCommand;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.network.socket.BaseSocketRequest;
import com.liveperson.infra.network.socket.SocketManager;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.infra.utils.MaskedMessage;
import com.liveperson.infra.utils.UniqueID;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.MessagingFactory;
import com.liveperson.messaging.model.AmsMessages;
import com.liveperson.messaging.model.Conversation;
import com.liveperson.messaging.model.ConversationData;
import com.liveperson.messaging.model.Dialog;
import com.liveperson.messaging.model.FileMessage;
import com.liveperson.messaging.model.MessagingChatMessage;
import com.liveperson.messaging.network.http.RestRequestParams;
import com.liveperson.messaging.network.http.SendFileRequestRest;
import com.liveperson.messaging.network.socket.requests.NewConversationRequest;
import com.liveperson.messaging.network.socket.requests.SendFileRequest;
import com.liveperson.messaging.network.socket.requests.SendMessageRequest;

/**
 * A command for sending message
 */
public abstract class SendFileCommand extends SendMessageCommand {

	private static final String TAG = "SendFileCommand";
	private final String mFileContentType;
	private final String mFileTypeExtension;
	private String mThumbnailPath;
	private String mLocalFilePath;
	protected String mCaption;
	private String mRelativePath;
	private String mFileType;
	protected String mPreview;

	protected SendFileCommandListener mCallback;
	protected long mMessageRowId = -1;
	private boolean sendViaRest;
	private RestRequestParams restDomain;
	// Holds the idle (queued) conversation ID
	private String mTempConversationId;
	private String mTempDialogId;

	protected abstract MessagingChatMessage.MessageType getMessageType(MaskedMessage message);

	/**
	 * Creates a command for sending messages
	 *
	 * @param controller
	 * @param message
	 */
	public SendFileCommand(Messaging controller, String targetId, String brandId, String contentType, String thumbnailPath, String filePath, String fileTypeExtension, MaskedMessage message) {
		super(controller, targetId, brandId, message);
		this.mFileContentType = contentType;
		mThumbnailPath = thumbnailPath;
		mLocalFilePath = filePath;
		mMessage = message;
		this.mCaption = mMessage.getServerMessage();
		mFileTypeExtension = fileTypeExtension;
	}

	public void setCallback(SendFileCommandListener mCallback) {
		this.mCallback = mCallback;
	}

	public void setFileDetails(String relativePath, String fileType, String previewBase64) {

		this.mRelativePath = relativePath;
		this.mFileType = fileType;
		this.mPreview = "data:" + mFileContentType + ";base64," + previewBase64;
	}

	/**
	 * Adding file message to db under the current available conversation and dialog.
	 * we'll update it again after connecting to the server
	 */
	public void addMessageToDB() {
		try {
			LPMobileLog.d(TAG, "addMessageToDB");
			String newDialogId = updateMessageWithCurrentDialog();

			LPMobileLog.d(TAG, "addMessageToDB File! - dialogId = " + newDialogId);
			addMessageToDBAndSend(newDialogId, mMessage);

		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * Create conversation and dialog if they are not open yet.
	 *
	 * @return new Temp dialog id in case of opening new dialog or the current dialog id otherwise
	 */
	private String updateMessageWithCurrentDialog() {
		//Conversation conversation = mController.amsConversations.getConversation(mTargetId);
		Dialog dialog = mController.amsDialogs.getActiveDialog();

		mConsumerId = mController.amsUsers.getConsumerId(mTargetId);

		String newConversationId;
		String newDialogId;

		if (dialog == null) {
			LPMobileLog.i(TAG, "updateMessageWithCurrentDialog: creating new temp dialog and conversation, there is no open one.");
			newConversationId = Conversation.createTempConversationId();
			newDialogId = Dialog.createTempDialogId();

			createQueuedConversationAndDialog(newConversationId, newDialogId);

		} else {
			switch (dialog.getState()) {
				case CLOSE:
					LPMobileLog.i(TAG, "updateMessageWithCurrentDialog: creating new temp dialog and conversation, they are closed.");
					newConversationId = Conversation.createTempConversationId();
					newDialogId = Dialog.createTempDialogId();

					BaseAMSSocketRequest newConversationRequest = createPendingConversationAndDialog(newConversationId, newDialogId);
					sendCreateConversationRequest(newConversationRequest);
					break;
				case OPEN:
				case PENDING:
				case QUEUED:
				default:
					newDialogId = dialog.getDialogId();
					LPMobileLog.i(TAG, "updateMessageWithCurrentDialog: getting current dialog ID: " + newDialogId + ", state: " + dialog.getState());
					break;
			}
		}
		return newDialogId;
	}

	/**
	 * This method is used to add a message to DB. If regexPattern and maskCharacter are provided, the message is checked
	 * and masked if the regex pattern exist in the message. Otherwise it is sent as is.
	 *
	 * @param dialogId
	 * @param message
	 */
	protected void addMessageToDBAndSend(String dialogId, MaskedMessage message) {
		mEventId = UniqueID.createUniqueMessageEventId();
		LPMobileLog.d(TAG, "addMessageToDBAndSend, createNewChatMessage, ContentType: " + mFileContentType);
		MessagingChatMessage chatMessage = createNewChatMessage(dialogId, message);

		chatMessage.setContentType(mFileContentType);

		// Add the message to DB
		mController.amsMessages.addMessage(mTargetId, chatMessage, true)
				.setPostQueryOnBackground(new DataBaseCommand.QueryCallback<Long>() {

					@Override
					public void onResult(Long rowId) {

						if (rowId == DBUtilities.ROW_UPDATED) {
							LPMobileLog.d(TAG, "onResult: message was updated on DB (and not inserted). No need to add the file to DB");
						} else {
							LPMobileLog.d(TAG, "addMessageToDBAndSend,  MessagingChatMessage was added. row id: " + rowId + ". Adding fileMessage to db.");

							// Add the file the DB
							FileMessage fileMessage = new FileMessage(mThumbnailPath, mFileTypeExtension, mLocalFilePath, null, rowId);
							long fileRowId = MessagingFactory.getInstance().getController().amsFiles.addFile(rowId, fileMessage).executeSynchronously();

							mMessageRowId = rowId;
							LPMobileLog.d(TAG, "addMessageToDBAndSend, fileMessage was added to db. Thumbnail path: " + mThumbnailPath + ", local file path: " + mLocalFilePath);

							mController.amsMessages.updateFileMessageByRowId(rowId, fileRowId).executeSynchronously();
							if (mCallback != null) {
								mCallback.onFileAddedToDB(mMessageRowId, fileRowId);
							}
						}
					}
				}).execute();


		// If messages are different it means the message was masked. We add a system message for the user
		if (mMessage.isMasked()) {
			MessagingChatMessage warning = new MessagingChatMessage(chatMessage.getOriginatorId(),
					message.getMaskedSystemMessage(),
					chatMessage.getTimeStamp() + 1, chatMessage.getDialogId(), UniqueID.createUniqueMessageEventId(),
					MessagingChatMessage.MessageType.SYSTEM_MASKED, MessagingChatMessage.MessageState.RECEIVED,
					AmsMessages.MASKED_CC_MSG_SEQUENCE_NUMBER, ContentType.text_plain.getText(), EncryptionVersion.NONE);

			mController.amsMessages.addMessage(mTargetId, warning, true).execute();
		}

	}


	/**
	 * After connected to UMS we updating message's dialog id according to the new one (if there is)
	 */
	public void updateMessageConversationId() {
		LPMobileLog.i(TAG, "update Message dialog ID");
		String newDialogId = updateMessageWithCurrentDialog();

		LPMobileLog.i(TAG, "update Message dialog ID - updating file message: " + mMessageRowId + " with new dialog id = " + newDialogId);

		//updating the message with new dialog id (if needed)
		mController.amsMessages.updateMessageDialogServerIdAndTime(mMessageRowId, newDialogId, getMessageTime())
				.setPostQueryOnBackground(new DataBaseCommand.QueryCallback<Void>() {
					@Override
					public void onResult(Void data) {
						LPMobileLog.i(TAG, "Finished updating file message ( dialog ID and new time ) ");
						if (mCallback != null) {
							mCallback.onMessageUpdatedInDB();
						}
					}
				}).execute();
	}

	/**
	 * @return time of the message, this is for resend to override.
	 * DO NOT DELETE
	 */
	protected long getMessageTime() {
		return System.currentTimeMillis();
	}

	/**
	 * Sending the message or creating conversation and adding the message to current conversation.
	 */
	@Override
	public void execute() {
		LPMobileLog.i(TAG, "Sending file message. creating new conversation if there is no open one.");
		//Conversation conversation = mController.amsConversations.getConversation(mTargetId);
		mConsumerId = mController.amsUsers.getConsumerId(mTargetId);

		Dialog dialog = mController.amsDialogs.getActiveDialog();

		mConsumerId = mController.amsUsers.getConsumerId(mTargetId);

		if (dialog == null) {
			String tempConversationId = Conversation.createTempConversationId();
			String tempDialogId = Dialog.createTempDialogId();
			LPMobileLog.e(TAG, "SHOULD NEVER HAPPEN!!");
			addMessageToDBAndSend(tempConversationId, mMessage);
			BaseAMSSocketRequest newConversationRequest = createPendingConversationAndDialog(tempConversationId, tempDialogId);
			sendCreateConversationRequest(newConversationRequest);
		} else {
			LPMobileLog.i(TAG, "checking current dialog, state - " + dialog.getState());
			switch (dialog.getState()) {
				case CLOSE:
					// If the conversation is closed during file upload, we fail the message
					LPMobileLog.i(TAG, "Dialog is closed. Fail the file message");
					markMessageAsError();
					break;
				case QUEUED:
					LPMobileLog.i(TAG, "Dialog is queued and waiting to be created...");
					markMessageAsPending();
					sendNewConversationRequest(dialog.getConversationId(), dialog.getDialogId(), dialog.getRequestId());
					break;
				case OPEN:
				case PENDING:
					LPMobileLog.i(TAG, "Dialog is open/pending. Sending message");
					markMessageAsPending();
				default:
					break;
			}
		}
		sendMessageIfDialogIsOpen();
	}

	private void markMessageAsPending() {
		LPMobileLog.i(TAG, "Changing message state to pending.. waiting to be send...");
		mController.amsMessages.updateMessageState(mMessageRowId, MessagingChatMessage.MessageState.PENDING);
	}

	private void markMessageAsError() {
		LPMobileLog.i(TAG, "Changing message state to Error.");
		mController.amsMessages.updateMessageState(mMessageRowId, MessagingChatMessage.MessageState.ERROR);
	}

	private void sendNewConversationRequest(String newConversationId, String newDialogId, long requestId) {
		LPMobileLog.i(TAG, "Sending request to create new conversation. conversation id = " + newConversationId);
		NewConversationRequest newConversationRequest = createNewConversationRequest(newConversationId, newDialogId, requestId);
		mController.amsConversations.updateConversationState(mTargetId, newConversationId, ConversationState.PENDING);
		mController.amsDialogs.updateDialogState(mTargetId, newDialogId, DialogState.PENDING);
		SocketManager.getInstance().send(newConversationRequest);
	}

	private void createQueuedConversationAndDialog(String tempDialogId, String tempConversationId){
		long requestId = BaseSocketRequest.createRequestId();
		createQueuedConversation(tempConversationId, requestId);
		createQueuedDialog(tempDialogId, tempConversationId, requestId);
	}

	private void createQueuedConversation(String tempConversationID, long requestId) {
		mTempConversationId = tempConversationID;
		mController.amsConversations.createQueuedConversation(mTargetId, mBrandId, tempConversationID, requestId);
	}

	private void createQueuedDialog(String tempDialogId, String tempConversationId, long requestId){
		mTempDialogId = tempDialogId;
		mController.amsDialogs.createQueuedDialog(mTargetId, mBrandId, tempDialogId, tempConversationId, requestId);
	}

	@NonNull
	protected MessagingChatMessage createNewChatMessage(String dialogId, MaskedMessage message) { // TODO: 5/7/18 make abstract and implement on image and voice
		return new MessagingChatMessage(mController.getOriginatorId(mTargetId),
				message.getDbMessage(), System.currentTimeMillis(),
				dialogId,
				mEventId,
				getMessageType(message),
				MessagingChatMessage.MessageState.QUEUED,
				EncryptionVersion.NONE);
	}

    @NonNull
    protected SendMessageRequest createMessageRequest(Messaging mController, String eventId, String mTargetId, String mBrandId, String dialogId, String conversationId) {
		SendFileRequest sendFileRequest = new SendFileRequest(mController, eventId, mTargetId, mBrandId, dialogId, conversationId);
		sendFileRequest.setFileContent(mCaption, mRelativePath, mFileType, mPreview);
		return sendFileRequest;
	}

	@Override
	protected void sendMessage(String conversationServerId, SendMessageRequest sendMessageRequest) {
		if (sendViaRest) {
			new SendFileRequestRest(restDomain, sendMessageRequest).execute();
		} else {
			super.sendMessage(conversationServerId, sendMessageRequest);
		}
	}

	public void failMessage() {

		LPMobileLog.w(TAG, "failMessage: upload file failed");

		if (mMessageRowId != -1) {
			LPMobileLog.d(TAG, "failMessage: setting message (rowId: " + mMessageRowId + ") to error");
			mController.amsMessages.updateMessageState(mMessageRowId, MessagingChatMessage.MessageState.ERROR);
		}

		// If conversation is in state QUEUED close it
		final Conversation conversation = mController.amsConversations.getConversation(mTargetId);
		if (conversation.getState() == ConversationState.QUEUED) {
			LPMobileLog.d(TAG, "failMessage: conversation " + conversation.getConversationId() + " is queued. Close it");
			ConversationData conversationData = new ConversationData();
			conversationData.conversationId = mTempConversationId;
			conversationData.brandId = mBrandId;
			conversationData.targetId = mTargetId;

			mController.amsConversations.updateClosedConversation(conversationData, false).execute();

		}

        Dialog dialog = mController.amsDialogs.getActiveDialog();

        // If dialog is in state QUEUED close it
        if (dialog != null && dialog.getState() == DialogState.QUEUED) {
            ConversationData conversationData = new ConversationData();
            conversationData.conversationId = mTempConversationId;
            conversationData.brandId = mBrandId;
            conversationData.targetId = mTargetId;
            mController.amsDialogs.updateClosedDialog(conversationData, dialog, false).execute();
        }
	}

	public void setSendViaRest(boolean sendViaRest, RestRequestParams restDomain) {
		this.sendViaRest = sendViaRest;
		this.restDomain = restDomain;
	}

	public interface SendFileCommandListener {
		void onFileAddedToDB(long messageRowId, long fileRowId);

		void onMessageUpdatedInDB();
	}
}