package com.liveperson.messaging.model;

import android.content.ContentValues;
import android.database.Cursor;
import android.net.Uri;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.text.format.DateUtils;
import android.webkit.URLUtil;

import com.liveperson.api.request.message.BasePublishMessage;
import com.liveperson.api.request.message.FilePublishMessage;
import com.liveperson.api.request.message.FormPublishMessage;
import com.liveperson.api.request.message.FormSubmissionPublishMessage;
import com.liveperson.api.response.events.ContentEventNotification;
import com.liveperson.api.response.model.ContentType;
import com.liveperson.api.response.model.Participants;
import com.liveperson.api.response.types.ConversationState;
import com.liveperson.api.response.types.DeliveryStatus;
import com.liveperson.api.response.types.DialogState;
import com.liveperson.infra.Clearable;
import com.liveperson.infra.ConversationViewParams;
import com.liveperson.infra.Infra;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.controller.DBEncryptionHelper;
import com.liveperson.infra.controller.DBEncryptionKeyHelper;
import com.liveperson.infra.database.BaseDBRepository;
import com.liveperson.infra.database.DBUtilities;
import com.liveperson.infra.database.DataBaseCommand;
import com.liveperson.infra.database.DataBaseExecutor;
import com.liveperson.infra.database.tables.BaseTable;
import com.liveperson.infra.database.tables.DialogsTable;
import com.liveperson.infra.database.tables.FilesTable;
import com.liveperson.infra.database.tables.MessagesTable;
import com.liveperson.infra.database.tables.UsersTable;
import com.liveperson.infra.database.transaction_helper.InsertOrUpdateSQLCommand;
import com.liveperson.infra.database.transaction_helper.InsertSQLCommand;
import com.liveperson.infra.database.transaction_helper.SQLiteCommand;
import com.liveperson.infra.database.transaction_helper.UpdateSQLCommand;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.messaging.R;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDown;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.infra.utils.ImageUtils;
import com.liveperson.infra.utils.UniqueID;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.MessagingFactory;
import com.liveperson.messaging.commands.DeliveryStatusUpdateCommand;
import com.liveperson.messaging.network.MessageTimeoutQueue;
import com.liveperson.messaging.network.http.MessageTimeoutListener;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * Created by Ilya Gazman on 11/10/2015.
 * <p/>
 * Messages model
 */
public class AmsMessages extends BaseDBRepository implements ShutDown, Clearable, AmsMessagesLoaderProvider {

	public enum MessagesSortedBy {
		TargetId, ConversationId
	}

	public interface MessagesListener {

		void initMessages(ArrayList<FullMessageRow> searchedMessageList);

		void onQueryMessagesResult(long firstTimestamp, long lastTimestamp);

		void onUpdateMessages(long firstTimestamp, long lastTimestamp);

		void onNewMessage(FullMessageRow fullMessageRow);

		void onUpdateMessage(FullMessageRow fullMessageRow);

		void removeAll(String targetId);

		void onHistoryFetched();

		void onExConversationHandled(boolean emptyNotification);

		void onHistoryFetchedFailed();
	}

	private static final String TAG = AmsMessages.class.getSimpleName();
    private static final int MAX_SQL_VARIABLES = 997;

	public static final int PENDING_MSG_SEQUENCE_NUMBER = -1;
	public static final int RESOLVE_MSG_SEQUENCE_NUMBER = -2;
	public static final int MASKED_CC_MSG_SEQUENCE_NUMBER = -3;
	public static final int WELCOME_MSG_SEQUENCE_NUMBER = -4;
	public static final String STRUCTURED_CONTENT_PREFIX = "lpsc:";

    private final Messaging mController;
	public final FormsManager mFormsManager;

	private MessagesListener mMessagesListener = null;
	private MessagesListener mNullMessagesListener = new NullMessagesListener();

	// Holds the most recent agent message quick replies data
	private QuickRepliesMessageHolder mQuickRepliesMessageHolder = null;

	public AmsMessages(Messaging controller) {
		super(MessagesTable.MESSAGES_TABLE);

		mController = controller;
		MessageTimeoutListener mMessageTimeoutListener = new MessageTimeoutListener() {
			@Override
			public void onMessageTimeout(String brandId) {
				mController.onMessageTimeout(brandId);
				LPMobileLog.e(TAG, "on message timeout received");
			}

			@Override
			public void onPublishMessageTimeout(String brandId, String eventId, String dialogId) {
				updateMessageState(eventId,
						brandId,
						dialogId,
						MessagingChatMessage.MessageState.ERROR);
				LPMobileLog.e(TAG, "on update message timeout");
			}
		};
		mMessageTimeoutQueue = new MessageTimeoutQueue(mMessageTimeoutListener);
		mFormsManager = new FormsManager();
	}

	// A queue for storing messages for a specific timeout so we can mark them as error if not getting ack on them
	public final MessageTimeoutQueue mMessageTimeoutQueue;

	public Cursor getMessages() {
		return null;
	}

	static Uri getContentURI(String uri) {
		return Uri.parse("content://" + uri);
	}

	/**
	 * Notify of data change from change of agent profile
	 */
	public DataBaseCommand<Void> updateOnCommand(final String targetId, final String conversationID) {

		return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<Void>() {
			@Override
			public Void query() {
				return null;
			}
		});
	}

	/**
	 * MUST RUN ON DB THREAD!!
	 *
	 * @param brandID
	 * @param limitSize          max num of messages required.  0 or negative value for no limit
	 * @param olderThanTimestamp negative value if there is no need to use this value
	 * @param newerThanTimestamp negative value if there is no need to use this value
	 * @return cursor with messages according to params
	 */
	Cursor messagesByTarget(String brandID, int limitSize, long olderThanTimestamp, long newerThanTimestamp) {

		StringBuilder sql = getMessagesForTargetQuery().append(" WHERE ").append(DialogsTable.TABLE_NAME)
				.append(".").append(DialogsTable.Key.BRAND_ID).append(" = \"").append(brandID).append("\"");

		if (olderThanTimestamp > -1) {
			sql.append(" AND ").append(MessagesTable.KEY_TIMESTAMP).append(" <= ").append(olderThanTimestamp);
		}
		if (newerThanTimestamp > -1) {
			sql.append(" AND ").append(MessagesTable.KEY_TIMESTAMP).append(" >= ").append(newerThanTimestamp);
		}

		// LE-79838 [Android] History Control API :

		ConversationViewParams conversationViewParams = mController.getConversationViewParams();

		//bring only conversations that started/ended x days ago.
		if (conversationViewParams.getHistoryConversationsMaxDays() > -1){

			long historyConversationsMaxDays = conversationViewParams.getHistoryConversationsMaxDays() * DateUtils.DAY_IN_MILLIS;
			long daysAgo = System.currentTimeMillis() - historyConversationsMaxDays;
			switch (conversationViewParams.getHistoryConversationMaxDaysType()){
				case endConversationDate:
					sql.append(" AND ").append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.END_TIMESTAMP).append(" >= ").append(daysAgo);
					break;
				case startConversationDate:
					sql.append(" AND ").append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.START_TIMESTAMP).append(" >= ").append(daysAgo);
					break;
			}
		}
		//bring only conversations under specific state
		switch (conversationViewParams.getHistoryConversationsStateToDisplay()){
			case OPEN:
				sql.append(" AND ").append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.STATE).append(" = ").append(ConversationState.OPEN.ordinal());
				break;
			case CLOSE:
				sql.append(" AND ").append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.STATE).append(" = ").append(ConversationState.CLOSE.ordinal());
				break;
			case ALL:
				break;
		}


		//end of LE-79838 [Android] History Control API


		if (limitSize > 0) {
			sql.append(" ORDER BY ").append(MessagesTable.KEY_TIMESTAMP).append(" DESC ");
			sql.append(" LIMIT ").append(limitSize);
			sql = new StringBuilder("Select * FROM ( ").append(sql).append(" ) ORDER BY ").append(MessagesTable.KEY_TIMESTAMP).append(" ASC ");
		} else {
			sql.append(" ORDER BY ").append(MessagesTable.KEY_TIMESTAMP).append(" ASC ");
		}


		return getDB().rawQuery(sql.toString());
	}

	/**
	 * @param conversationID
	 * @param limitSize      -1 for no limit
	 * @return
	 */
	Cursor messagesByConversationID(String conversationID, int limitSize) {
		StringBuilder sql = getBasicMessagesQuery().append(" WHERE ").append(DialogsTable.TABLE_NAME)
				.append(".").append(DialogsTable.Key.DIALOG_ID).append(" = \"").append(conversationID).append("\"").append(" ORDER BY ")
				.append(MessagesTable.KEY_TIMESTAMP);

		if (limitSize != -1) {
			sql.append(" LIMIT ").append(limitSize);
		}
		return getDB().rawQuery(sql.toString());
	}

	private StringBuilder getMessagesForTargetQuery() {
		return new StringBuilder()
				.append("select ")
				.append(MessagesTable.MESSAGES_TABLE).append(".").append(BaseTable.KEY_ID).append(",")
				.append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_EVENT_ID).append(",")
				.append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_ORIGINATOR_ID).append(",")
				.append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_ENCRYPTION_VERSION).append(" AS ").append(MessagesTable.ENCRYPTION_VERSION_CURSOR_AS_VALUE).append(",")
				.append(UsersTable.USERS_TABLE).append(".").append(UsersTable.KEY_ENCRYPTION_VERSION).append(" AS ").append(UsersTable.ENCRYPTION_VERSION_CURSOR_AS_VALUE).append(",")
				.append(MessagesTable.KEY_SERVER_SEQUENCE).append(",")
				.append(MessagesTable.KEY_DIALOG_ID).append(",")
				.append(MessagesTable.KEY_TEXT).append(",")
				.append(MessagesTable.KEY_CONTENT_TYPE).append(",")
				.append(MessagesTable.KEY_MESSAGE_TYPE).append(",")
				.append(MessagesTable.KEY_STATUS).append(",")
				.append(MessagesTable.KEY_TIMESTAMP).append(",")
				.append(UsersTable.KEY_PROFILE_IMAGE).append(",")
				.append(UsersTable.KEY_NICKNAME).append(",")
				.append(FilesTable.FILES_TABLE).append(".").append(FilesTable.KEY_ID).append(" AS ").append(FilesTable.KEY_ID_AS_VALUE).append(",")
				.append(FilesTable.KEY_FILE_TYPE).append(",")
				.append(FilesTable.KEY_LOCAL_URL).append(",")
				.append(FilesTable.KEY_PREVIEW).append(",")
				.append(FilesTable.KEY_LOAD_STATUS).append(",")
				.append(FilesTable.KEY_RELATED_MESSAGE_ROW_ID).append(",")
				.append(FilesTable.KEY_SWIFT_PATH)
				.append(" from ")
				.append(MessagesTable.MESSAGES_TABLE)
				.append(" left join ").append(DialogsTable.TABLE_NAME).append(" on ").append(MessagesTable.MESSAGES_TABLE).append(".")
				.append(MessagesTable.KEY_DIALOG_ID).append("=").append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.DIALOG_ID)
				.append(" left join ").append(UsersTable.USERS_TABLE).append(" on ").append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_ORIGINATOR_ID)
				.append("=").append(UsersTable.USERS_TABLE).append(".").append(UsersTable.KEY_ORIGINATOR_ID)
				.append(" left join ").append(FilesTable.FILES_TABLE).append(" on ").append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_ID)
				.append("=").append(FilesTable.FILES_TABLE).append(".").append(FilesTable.KEY_RELATED_MESSAGE_ROW_ID);
	}

	@NonNull
	private StringBuilder getBasicMessagesQuery() {
		return new StringBuilder().append("select ").append(MessagesTable.MESSAGES_TABLE).append("._id, serverSequence,convID,text,contentType,type,status,")
				.append(MessagesTable.MESSAGES_TABLE).append(".eventId,").append(MessagesTable.MESSAGES_TABLE).append(".originatorId,timeStamp,encryptVer,")
				.append("description,firstName,lastName,phoneNumber,userType,email,profileImage,coverImage from ").append(MessagesTable.MESSAGES_TABLE)
				.append(" left join ").append(DialogsTable.TABLE_NAME).append(" on ").append(MessagesTable.MESSAGES_TABLE).append(".")
				.append(MessagesTable.KEY_DIALOG_ID).append("=").append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.DIALOG_ID)
				.append(" left join ").append(UsersTable.USERS_TABLE).append(" on ").append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_ORIGINATOR_ID)
				.append("=").append(UsersTable.USERS_TABLE).append(".").append(UsersTable.KEY_ORIGINATOR_ID);
	}

	/**
	 * Add a message, even if the conversation is in pending state
	 */
	public DataBaseCommand<Long> addMessage(final String targetId, final MessagingChatMessage message, final boolean updateUI) {
		return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<Long>() {
			@Override
			public Long query() {
				long rowId;
				boolean isEmptyEventId = TextUtils.isEmpty(message.getEventId());
				if (isEmptyEventId) {
					LPMobileLog.i(TAG, "Received new message without event id, generating new one.. ");
					message.setEventId(UniqueID.createUniqueMessageEventId());
					//in case there is no event id we need to check if such message exists
					//in the db by the conversation id and sequence. is exists - update it,
					//if not, insert this message.
					StringBuilder whereBuilder = new StringBuilder();
					whereBuilder.append(MessagesTable.KEY_DIALOG_ID).append(" = ? AND ")
							.append(MessagesTable.KEY_SERVER_SEQUENCE).append(" = ?");
					rowId = getDB().insertOrUpdate(getContentValuesForMessage(message),
							getContentValuesForMessageUpdate(message),
							whereBuilder.toString(),
							new String[]{message.getDialogId(), String.valueOf(message.getServerSequence())});
					LPMobileLog.d(TAG, "Insert or Update message: " + message + " rowId = " + rowId);
				} else {
					//Check if the message exist
					Cursor cursor = null;
					cursor = getDB().query(getProjection(), MessagesTable.KEY_EVENT_ID + " = ?", new String[]{message.getEventId()}, null, null, null);
					if (cursor != null && cursor.getCount() > 0) {
						ContentValues messageValues = getContentValuesForMessageUpdate(message, cursor);

						//If needed update the message in DB
						if (messageValues.size() > 0) {
							rowId = getDB().update(messageValues, MessagesTable.KEY_EVENT_ID + "=?",
									new String[]{String.valueOf(message.getEventId())});
							LPMobileLog.d(TAG, "Adding message: This message was update with message: " + message + " rowId = " + rowId);
						} else {
							rowId = -1;
							LPMobileLog.d(TAG, "Adding message: Skip add\\update this message since its already exist" + message + " rowId = " + rowId);
						}
					} else {
						// Add the recieved message to DB
						rowId = getDB().insertWithOnConflict(getContentValuesForMessage(message));
						LPMobileLog.d(TAG, "Adding message: " + message + " rowId = " + rowId);
					}
				}

				if (updateUI) {
                    if (rowId != -1) {
                        //new Item
                        getMessagesListener().onNewMessage(createFullMessageRow(rowId, message, -1));
                    } else {
                        //updated item - need to read again from db cause 'message' might contain different data than db
                        //we don;t have the row id on updated message - need to update by event id.
                        //updateMessageByRowIdOnDbThread(rowId);
                        String eventIdToUpdate = message.getEventId();
                        if (isEmptyEventId) {
                            LPMobileLog.d(TAG, "Updating message that originally didn't have event id. ");
                            eventIdToUpdate = getEventIdForMessage(message.getDialogId(), message.getServerSequence());
                        }
                        if (!TextUtils.isEmpty(eventIdToUpdate)) {
                            updateMessageByEventId(eventIdToUpdate);
                        }
                    }
				}

				return rowId;
			}
		});
	}


	/**
	 * Update a message, even if the conversation is in pending state
	 * Don't update the message if the serverSequence was meanwhile received from the server
	 */
	public DataBaseCommand<Integer> updateMessageOnRetry(final String targetId, final MessagingChatMessage message) {
		return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<Integer>() {
			@Override
			public Integer query() {
				LPMobileLog.d(TAG, "Updating message: " + message);
				int numOfRowsAffected = getDB().update(getContentValuesForMessage(message),
						MessagesTable.KEY_EVENT_ID + "=? and " + MessagesTable.KEY_SERVER_SEQUENCE + " =?",
						new String[]{String.valueOf(message.getEventId()), String.valueOf(-1)});

				return numOfRowsAffected;
			}
		});
	}

	public DataBaseCommand<Void> addMultipleMessages(final ArrayList<ContentEventNotification> responseMessages, final String originatorId, final String brandId,
													 final String targetId, final String dialogId, final String conversationId, final long clockDiff, final boolean firstNotification, final boolean mShouldUpdateUI) {

		return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<Void>() {

			public String[] extractLinks(String text) {
				List<String> links = new ArrayList<>();
				String[] l = text.split("\\s+");
				for (int i = 0; i < l.length; i++) {
					if (URLUtil.isValidUrl(l[i])) {
						links.add(l[i]);
					}
				}
				return links.toArray(new String[links.size()]);
			}

			private MessagingChatMessage.MessageType checkIfMessageContainsURLandChangeType(MessagingChatMessage.MessageType type, String msg) {
				String[] urls = extractLinks(msg);
				if (urls.length > 0) {
					if (type == MessagingChatMessage.MessageType.CONSUMER) {
						return MessagingChatMessage.MessageType.CONSUMER_URL;
					}
					if (type == MessagingChatMessage.MessageType.CONSUMER_MASKED) {
						return MessagingChatMessage.MessageType.CONSUMER_URL_MASKED;
					}
					if (type == MessagingChatMessage.MessageType.AGENT) {
						return MessagingChatMessage.MessageType.AGENT_URL;
					}
				}
				return type;
			}

			@Override
			public Void query() {

				ArrayList<SQLiteCommand> commands;
				if (responseMessages != null) {
					LPMobileLog.d(TAG, "Start addMultipleMessages. num of commands = " + responseMessages.size());

					commands = new ArrayList<>(responseMessages.size());

					MessagingChatMessage.MessageState messageState;
					MessagingChatMessage.MessageType messageType;
					MessagingChatMessage message = null;
					int[] sequenceList;
					ContentValues contentValues;
					StringBuilder whereBuilder;
					String[] whereArgs;

					int firstSequence = -1;
					int lastSequence = -1;

					int maxAcceptStatusSequence = -1;
					int maxReadStatusSequence = -1;

					int lastAgentMessageSequence = -1;

					for (ContentEventNotification notification : responseMessages) {
				   /* if (TextUtils.equals(originatorId, notification.originatorId)) {
						// ignore messages from consumer
                        continue;
                    }*/

				   		if (notification.event == null){
							LPMobileLog.e(TAG, "received message with empty event! continuing to next message.. " );
							continue;
						}

						switch (notification.event.type) {
							case ContentEvent: {
								BasePublishMessage publishMessage = null;

								if (notification.event.message != null) {
									publishMessage = notification.event.message;
								}

								// If the publish message is a text and the message itself is empty we ignore this
								if ((publishMessage == null)
										|| ((publishMessage.getType() == BasePublishMessage.PublishMessageType.TEXT)
										&& TextUtils.isEmpty(publishMessage.getMessageText()))) {

									LPMobileLog.e(TAG, "Text message received in query messages is empty :| shouldn't happen! " +
											"dialogId = " + dialogId + " , sequence = " + notification.sequence);
									// it's some protocol error, lets ignore it
									continue;
								}

								// Get the content type from the event and set the message type and state
								ContentType contentType = ContentType.fromString(notification.event.contentType);


								if (TextUtils.equals(originatorId, notification.originatorId)) {
									//consumer message - make sure it's not double
									messageType = MessagingChatMessage.MessageType.getMessageContentTypeForConsumer(notification, contentType);
									messageState = MessagingChatMessage.MessageState.SENT;
								} else {
									// If this message is not from myself

									// If message from Controller
									if(notification.originatorMetadata != null && notification.originatorMetadata.mRole == Participants.ParticipantRole.CONTROLLER){
										messageType = MessagingChatMessage.MessageType.CONTROLLER_SYSTEM;
									}
									else { // Message from Agent
										messageType = MessagingChatMessage.MessageType.getMessageContentTypeForAgent(notification,contentType);
										mController.amsConversations.resetEffectiveTTR(targetId);
										mController.amsDialogs.resetEffectiveTTR(targetId);
										//saving the sequence number of the last message sent by an agent
										lastAgentMessageSequence = notification.sequence;
									}
									messageState = MessagingChatMessage.MessageState.RECEIVED;
								}

								// check if we have a link than we need to set the message type to be with URL
								if ((messageType == MessagingChatMessage.MessageType.CONSUMER) ||
										(messageType == MessagingChatMessage.MessageType.CONSUMER_MASKED) ||
										(messageType == MessagingChatMessage.MessageType.AGENT)) {

									messageType = checkIfMessageContainsURLandChangeType(messageType, publishMessage.getMessageText());
								}

								// Create the relevant message and add to DB
								message = createMessageInDB(commands, messageState, messageType, notification, publishMessage, contentType);

								// Get the QuickReplies JSON from the current notification
								getQuickRepliesFromEvent(brandId, notification, messageType, dialogId);

								if (firstSequence == -1) {
									firstSequence = notification.sequence;
								}
								lastSequence = notification.sequence;
							}
							break;

							case RichContentEvent: {
								BasePublishMessage publishMessage = null;

								if (notification.event.message != null) {
									publishMessage = notification.event.message;
								}

								// If the publish message is a text and the message itself is empty we ignore this
								if ((publishMessage == null)
										|| ((publishMessage.getType() == BasePublishMessage.PublishMessageType.TEXT)
										&& TextUtils.isEmpty(publishMessage.getMessageText()))) {

									LPMobileLog.e(TAG, "Text message received in query messages is empty :| shouldn't happen! " +
											"dialogId = " + dialogId + " , sequence = " + notification.sequence);
									// it's some protocol error, lets ignore it
									continue;
								}

								// Get the content type from the event and set the message type and state
								ContentType contentType = ContentType.text_structured_content;
								messageType = MessagingChatMessage.MessageType.AGENT_STRUCTURED_CONTENT;


								if (TextUtils.equals(originatorId, notification.originatorId)) {
									//consumer message - make sure it's not double
									messageState = MessagingChatMessage.MessageState.SENT;
								} else {
									//message from agent
									messageState = MessagingChatMessage.MessageState.RECEIVED;
									mController.amsConversations.resetEffectiveTTR(targetId);
                                    mController.amsDialogs.resetEffectiveTTR(targetId);
									//saving the sequence number of the last message sent by an agent
									lastAgentMessageSequence = notification.sequence;
								}

								// Create the relevant message and add to DB
								createMessageInDB(commands, messageState, messageType, notification, publishMessage, contentType);

								// Get the QuickReplies JSON from the current notification
								getQuickRepliesFromEvent(brandId, notification, messageType, dialogId);

								if (firstSequence == -1) {
									firstSequence = notification.sequence;
								}
								lastSequence = notification.sequence;
							}
							break;
							case AcceptStatusEvent:

								messageState = getReceivedMessageState(notification.event.status);
								sequenceList = notification.event.sequenceList;

								if (messageState == null) {
									LPMobileLog.e(TAG, "messageState is null :| shouldn't happen! original status: " + notification.event.status +
											", dialogId = " + dialogId + " , sequence = " + Arrays.toString(sequenceList));

									continue;
								}
								//check who the is the origin of this AcceptStatus Event
								if (TextUtils.equals(originatorId, notification.originatorId)) {
									//the AcceptStatus is from the consumer
								} else {
									//the AcceptStatus is from an Agent
									if(sequenceList[0] == lastAgentMessageSequence){
										LPMobileLog.d(TAG, "AcceptStatusEvent recieved from agent for agent message. we ignore this event. lastAgentMsgSequence = " +lastAgentMessageSequence);
										break;
									}
								}

								if ((sequenceList != null) && sequenceList.length > 0) {

									int length = sequenceList.length;
                                    int lastStatusSequence = sequenceList[length - 1];

									//saving max sequence to update all messages at the end.
									if (messageState == MessagingChatMessage.MessageState.READ  &&
											maxReadStatusSequence < lastStatusSequence){

										maxReadStatusSequence = lastStatusSequence;

									}else if (messageState == MessagingChatMessage.MessageState.RECEIVED&&
											maxAcceptStatusSequence < lastStatusSequence){

										maxAcceptStatusSequence = lastStatusSequence;

									}else{
										//for viewed STATUS / SUBMITTED / EXPIRED etc..
										for (int i = 0; i < length; i += (MAX_SQL_VARIABLES - 1)) {
											int[] tempArray;
											int size = (length - i > (MAX_SQL_VARIABLES - 1)) ? (MAX_SQL_VARIABLES - 1) : length - i;
											if (size == length) {
												tempArray = sequenceList;
											} else {
												tempArray = new int[MAX_SQL_VARIABLES];
												System.arraycopy(sequenceList, i, tempArray, 0, size);
											}

											contentValues = new ContentValues();
											whereBuilder = new StringBuilder();
											// Create where args array for the conversation ID and all the sequences from the list
											whereArgs = new String[size + 2];

											createStatementForUpdateMessagesState(dialogId, tempArray, messageState, size, contentValues, whereBuilder, whereArgs);

											commands.add(new UpdateSQLCommand(contentValues, whereBuilder.toString(), whereArgs));
										}
									}

									//for updating UI later..
									if (firstSequence == -1) {
										firstSequence = sequenceList[0];
									}

                                    //Update UI: only if the last sequence is less than the status sequence - update.
                                    //if not, it will be in the range between firstSequence and lastSequence
                                    if (lastSequence < lastStatusSequence) {
                                        lastSequence = lastStatusSequence;
                                    }
                                }

								break;
						}


					}

					LPMobileLog.d(TAG, "dialogId = " + dialogId + ", responseMessages.size()  = " + responseMessages.size() );

					// Save the last quick replies (if exist and if conversation is not closed)
					if (mQuickRepliesMessageHolder != null && !MessagingFactory.getInstance().getController().isDialogClosed(dialogId)) {
						LPMobileLog.d(TAG, "QuickReplies exist in the received message, write to SharedPrefs");
						mQuickRepliesMessageHolder.writeToSharedPreferences();
					}

					// First we update all the read than all the Accept to minimize the updates requests!
					if (maxReadStatusSequence > -1){
						LPMobileLog.d(TAG, "dialogId = " + dialogId + ", maxReadStatusSequence = " + maxReadStatusSequence );
						commands.add(createStatementForUpdateMaxMessagesState(dialogId, MessagingChatMessage.MessageState.READ, maxReadStatusSequence));
					}
					//if accept is less or equals to read, no need to update
					if (maxAcceptStatusSequence > -1 && maxAcceptStatusSequence > maxReadStatusSequence){
						LPMobileLog.d(TAG, "dialogId = " + dialogId + ", maxAcceptStatusSequence = " + maxAcceptStatusSequence );
						commands.add(createStatementForUpdateMaxMessagesState(dialogId, MessagingChatMessage.MessageState.RECEIVED, maxAcceptStatusSequence));
					}

					getDB().runTransaction(commands);

					if (mShouldUpdateUI) {
						updateMessages(firstNotification, dialogId, firstSequence, lastSequence);
					}

					//sending update on messages status.
					sendReadAckOnMessages(brandId, targetId, originatorId);
				}
				return null;
			}

			@NonNull
			private MessagingChatMessage createMessageInDB(ArrayList<SQLiteCommand> commands, MessagingChatMessage.MessageState messageState, MessagingChatMessage.MessageType messageType, ContentEventNotification notification, BasePublishMessage publishMessage, ContentType contentType) {
				StringBuilder whereBuilder;
				MessagingChatMessage message;
				String eventId = notification.eventId;
				if (TextUtils.isEmpty(eventId)) {
					eventId = UniqueID.createUniqueMessageEventId();

					LPMobileLog.d(TAG, "no event id for message: " + publishMessage.getMessageText() + " creating event id: " + eventId);

					//in case there is no event id we need to check if such message exists
					//in the db by the conversation id and sequence. is exists - update it,
					//if not, insert this message.
					whereBuilder = new StringBuilder();
					whereBuilder.append(MessagesTable.KEY_DIALOG_ID).append(" = ? AND ")
							.append(MessagesTable.KEY_SERVER_SEQUENCE).append(" = ?");

					message = createMessage(publishMessage.getMessageText(), messageState, messageType, notification, eventId, contentType.getText());
					InsertOrUpdateSQLCommand insertOrUpdateSQLCommand = new InsertOrUpdateSQLCommand(
							getContentValuesForMessage(message),
							getContentValuesForMessageUpdate(message), whereBuilder.toString(),
							new String[]{dialogId, String.valueOf(notification.sequence)});

					// Add a mMessagesListener on the SQL command in order to add the image file when finished (if requierd)
					AddSqliteCommandListenerForFile(publishMessage, insertOrUpdateSQLCommand);

					// Add a treatment for form invitation and submission
					AddCommandForForm(publishMessage, message, brandId, commands);

					commands.add(insertOrUpdateSQLCommand);

				} else {

					//we got message with eventId - we insert, if it's exists it will replace.
					message = createMessage(publishMessage.getMessageText(), messageState, messageType, notification, eventId, contentType.getText());

					//Check if the message exist
					Cursor cursor = null;
					cursor = getDB().query(getProjection(), MessagesTable.KEY_EVENT_ID + " = ?", new String[]{message.getEventId()}, null, null, null);
					if (cursor != null && cursor.getCount() > 0) {
						ContentValues messageValues = getContentValuesForMessageUpdate(message, cursor);

						//If needed update the message in DB
						if (messageValues.size() > 0) {
							LPMobileLog.d(TAG, "Updating message: This message need to be update with message: " + message);
							UpdateSQLCommand updateSQLCommand = new UpdateSQLCommand(messageValues, MessagesTable.KEY_EVENT_ID + "=?",
									new String[]{String.valueOf(message.getEventId())});

							// Add a mMessagesListener on the SQL command in order to add the image file when finished (if requierd)
							AddSqliteCommandListenerForFile(publishMessage, updateSQLCommand);

							// Add a treatment for form invitation and submission
							AddCommandForForm(publishMessage, message, brandId, commands);

							commands.add(updateSQLCommand);

						} else {
							LPMobileLog.d(TAG, "Updating message: Skip updating this message since its already exist" + message);
						}
					} else {
						//we got message with eventId - we insert, if it's exists it will replace.
						InsertSQLCommand insertSQLCommand = new InsertSQLCommand(getContentValuesForMessage(message));

						// Add a mMessagesListener on the SQL command in order to add the image file when finished (if requierd)
						AddSqliteCommandListenerForFile(publishMessage, insertSQLCommand);

						// Add a treatment for form invitation and submission
						AddCommandForForm(publishMessage, message, targetId, commands);

						commands.add(insertSQLCommand);
					}

				}
				return message;
			}

			@NonNull
			private MessagingChatMessage createMessage(String textMessage, MessagingChatMessage.MessageState messageState, MessagingChatMessage.MessageType messageType, ContentEventNotification notification, String eventId, String contentType) {
				MessagingChatMessage message;

				// If the message text starts with STRUCTURED_CONTENT_PREFIX than this is a structured content message.
				// We set the type of the message accordingly and remove the prefix from the message text
				// Note: this is a temporary implementation until the server side implements the new event for structured content
 // TODO: This is left here just for debug purposes until UMS is fully ready with structured content
/*
				if (textMessage.startsWith(STRUCTURED_CONTENT_PREFIX)) {
					messageType = MessagingChatMessage.MessageType.AGENT_STRUCTURED_CONTENT;
					textMessage = textMessage.substring(STRUCTURED_CONTENT_PREFIX.length(), textMessage.length());
				}
*/


				message = new MessagingChatMessage(
						notification.originatorId,
						textMessage,
						notification.serverTimestamp + clockDiff,
						dialogId,
						eventId,
						messageType,
						messageState,
						notification.sequence,
						contentType,
						EncryptionVersion.NONE);

                LPMobileLog.d(TAG, "creating message '" + textMessage + "', seq: " + notification.sequence + ", at time: " + notification.serverTimestamp
						+ ", dialogId: " + dialogId + ", clock diff: " + clockDiff + " = " + (notification.serverTimestamp + clockDiff));
                return message;
            }

			private void AddCommandForForm(BasePublishMessage publishMessage, MessagingChatMessage msg, String brandId, ArrayList<SQLiteCommand> commands) {
				if (publishMessage.getType() == BasePublishMessage.PublishMessageType.FORM_INVITATION) {
					FormPublishMessage formPublishMessage = (FormPublishMessage) publishMessage;
					LPMobileLog.d(TAG, "onResult: new form obj to DB getMessage " + formPublishMessage.getMessage());
					mController.amsMessages.mFormsManager.addForm(formPublishMessage.getInvitationId(),
							new Form(
									msg.getDialogId(),
									conversationId,
									formPublishMessage.getInvitationId(),
									formPublishMessage.getFormId(),
									formPublishMessage.getFormTitle(),
									mController.mAccountsController.getTokenizerUrl(brandId),
									brandId,
									msg.getServerSequence(),
									msg.getEventId()
							));
				}
				if (publishMessage.getType() == BasePublishMessage.PublishMessageType.FORM_SUBMISSION) {
					FormSubmissionPublishMessage formPublishMessage = (FormSubmissionPublishMessage) publishMessage;
					Form form = mController.amsMessages.mFormsManager.getForm(formPublishMessage.getInvitationId());
					if (form != null) {
						mController.amsMessages.mFormsManager.updateForm(formPublishMessage.getInvitationId(), formPublishMessage.getmSubmissionId());
						LPMobileLog.d(TAG, "Updating message: This message need to be update with message: ");
						ContentValues cv = new ContentValues();
						cv.put(MessagesTable.KEY_STATUS, getReceivedMessageState(DeliveryStatus.SUBMITTED).ordinal());
						UpdateSQLCommand updateSQLCommand = new UpdateSQLCommand(cv, MessagesTable.KEY_EVENT_ID + "=?",
								new String[]{String.valueOf(form.getEventId())});
						commands.add(updateSQLCommand);
						//mFormsManager.removeForm(invitationId);
					}


				}
			}

			/**
			 * Add a mMessagesListener to the given sqLiteCommand to add the image file after the command is complete
			 * @param publishMessage
			 * @param sqLiteCommand
			 */
			private void AddSqliteCommandListenerForFile(BasePublishMessage publishMessage, SQLiteCommand sqLiteCommand) {

				// Validate the given publishMessage is of type file
				if (publishMessage.getType() == BasePublishMessage.PublishMessageType.FILE) {

					final BasePublishMessage finalPublishMessage = publishMessage;
					// Listener
					sqLiteCommand.setListener(new SQLiteCommand.SQLiteCommandListener() {
						@Override
						public void onInsertComplete(long rowId) {
							String tag = "onInsertComplete";

							if (rowId == DBUtilities.ROW_UPDATED) {
								LPMobileLog.d(TAG, tag + ": message was updated on DB (and not inserted). No need to add the file to DB");
							} else {
								addFileFromPublishMessageToDB(rowId, tag, (FilePublishMessage) finalPublishMessage, targetId);
							}
						}
					});
				}
			}

		});
	}

	/**
	 * Get the QuickReplies JSON from the given notification. This only happen for message from the agent
	 * @param brandId
	 * @param notification
	 * @param messageType
	 * @param dialogId
	 */
	private void getQuickRepliesFromEvent(String brandId, ContentEventNotification notification, MessagingChatMessage.MessageType messageType, String dialogId) {

		// If conversation is closed don't add QuickReplies
		if(MessagingFactory.getInstance().getController().isDialogClosed(dialogId)) {
			LPMobileLog.d(TAG, "getQuickRepliesFromEvent: conversation is closed, not adding QuickReplies message");
			return;
		}

		// Save the message only if from agent
		if ((messageType == MessagingChatMessage.MessageType.AGENT) ||
				(messageType == MessagingChatMessage.MessageType.AGENT_STRUCTURED_CONTENT) ||
				(messageType == MessagingChatMessage.MessageType.AGENT_URL)) {


			QuickRepliesMessageHolder currentQuickRepliesMessageHolder = QuickRepliesMessageHolder.fromContentEventNotification(brandId, notification);

			LPMobileLog.d(TAG, "getQuickRepliesFromEvent: Message is from agent, try to get QuickReplies string from event");

			// Store the current QR message only if newer
			if (mQuickRepliesMessageHolder == null ||
					((currentQuickRepliesMessageHolder != null) && currentQuickRepliesMessageHolder.newerThan(mQuickRepliesMessageHolder))) {

				LPMobileLog.d(TAG, "QuickReplies", "QuickReplies message is newer than the current one. New one: " + ((currentQuickRepliesMessageHolder != null) ? currentQuickRepliesMessageHolder.toString() : "null"));

				mQuickRepliesMessageHolder = currentQuickRepliesMessageHolder;
			}
		}
	}

	/**
	 * Get the {@link QuickRepliesMessageHolder} stored in memory. If not exist in memory load from SharedPreferences
	 * @param brandId
	 * @return
	 */
	@Override
	public QuickRepliesMessageHolder getQuickRepliesMessageHolder(String brandId) {

		// If does not exist in memory load from SharedPreferences
		if (mQuickRepliesMessageHolder == null) {
			mQuickRepliesMessageHolder = QuickRepliesMessageHolder.loadFromSharedPreferences(brandId);
		}

		return mQuickRepliesMessageHolder;
	}

	@Override
	public void resetQuickRepliesMessageHolder() {
		LPMobileLog.d(TAG, "resetQuickRepliesMessageHolder: resetting QuickRepliesMessageHolder");
		mQuickRepliesMessageHolder = null;
	}

	//Check if the message state and server sequence need to be updated in the existing message
	@NonNull
	private ContentValues getContentValuesForMessageUpdate(MessagingChatMessage message, Cursor cursor) {
		MessagingChatMessage existingMessage = getSingleMessageFromCursor(cursor);
		ContentValues messageValues = new ContentValues();

		//Check if we need to update the existing message
		if ((message.getMessageState().ordinal() != existingMessage.getMessageState().ordinal()) &&
				(MessagingChatMessage.MessageState.validChange(existingMessage.getMessageState(), message.getMessageState()))) {
			messageValues.put(MessagesTable.KEY_STATUS, message.getMessageState().ordinal());
		} else {
			LPMobileLog.d(TAG, "Skip update message state, old val: " + message.getMessageState() + " , new val: " + existingMessage.getMessageState());
		}
		if (message.getServerSequence() != existingMessage.getServerSequence()) {
			messageValues.put(MessagesTable.KEY_SERVER_SEQUENCE, message.getServerSequence());
		} else {
			LPMobileLog.d(TAG, "Skip update message server sequence, old val: " + message.getServerSequence() + " , new val: " + existingMessage.getServerSequence());
		}
		return messageValues;
	}

    /**
     * MUST BE CALLED FROM DB Thread!
     *  @param messageRowId
     * @param tag
	 * @param finalPublishMessage
	 * @param targetId
	 */
    public void addFileFromPublishMessageToDB(long messageRowId, String tag, FilePublishMessage finalPublishMessage, String targetId) {

		LPMobileLog.d(TAG, tag + ": MessagingChatMessage was added. row id: " + messageRowId + ". Adding fileMessage to db.");

        // We first save the base64 as bitmap to disk
        String previewImagePath = ImageUtils.saveBase64ToDisk(Infra.instance.getApplicationContext(), finalPublishMessage.getPreview(), targetId);

		LPMobileLog.d(TAG, tag + ": preview image saved to location: " + previewImagePath);

		if (previewImagePath != null) {
            // Add the file the DB
            FileMessage fileMessage = new FileMessage(previewImagePath, finalPublishMessage.getFileType(), null, finalPublishMessage.getRelativePath(), messageRowId);
            long fileRowId = MessagingFactory.getInstance().getController().amsFiles.addFile(messageRowId, fileMessage).executeSynchronously();
            LPMobileLog.d(TAG, tag + ": fileMessage was added to db. fileRowId = " + fileRowId);

		}
	}

	/**
	 * Update the status of the message with the given statuses
	 *
	 * @param targetId
	 * @param dialogId
	 * @param sequenceList
	 * @param status
	 */
	public void updateMessageReceivedState(final String targetId, final String dialogId, final int[] sequenceList, final DeliveryStatus status) {

		DataBaseExecutor.execute(new Runnable() {
			@Override
			public void run() {
				updateReceivedState(status, targetId, dialogId, sequenceList);
			}
		});
	}

	private void updateReceivedState(DeliveryStatus status, String targetId, String dialogId, int[] sequenceList) {
		MessagingChatMessage.MessageState messageState = getReceivedMessageState(status);

		if (messageState == null) {
			return;
		}

		updateMessagesState(targetId, dialogId, sequenceList, messageState);
	}

	@NonNull
	private MessagingChatMessage.MessageState getReceivedMessageState(DeliveryStatus status) {
		MessagingChatMessage.MessageState messageState = null;

		switch (status) {
			case ACCEPT:
				messageState = MessagingChatMessage.MessageState.RECEIVED;
				break;
			case READ:
			case ACTION:
				messageState = MessagingChatMessage.MessageState.READ;
				break;
			case VIEWED:
				messageState = MessagingChatMessage.MessageState.VIEWED;
				break;
			case SUBMITTED:
				messageState = MessagingChatMessage.MessageState.SUBMITTED;
				break;
			case ERROR:
			case ABORTED:
				messageState = MessagingChatMessage.MessageState.ERROR;
				break;
		}
		//LPMobileLog.d(TAG, "pci from DeliveryStatus state: "+status +" to MessageState status: "+messageState);
		return messageState;
	}

	/**
	 * Update the given message state to all messages with the sequence from the given sequence list
	 *
	 * @param targetId
	 * @param dialogId
	 * @param sequenceList
	 * @param messageState
	 */
	private void updateMessagesState(final String targetId, final String dialogId, final int[] sequenceList, final MessagingChatMessage.MessageState messageState) {

		// SQL query: update messages set status = <messageState> where convID = <conversationId> AND serverSequence in (<1>, <2>...)
		if ((sequenceList != null) && sequenceList.length > 0) {
			ContentValues contentValues = new ContentValues();
			StringBuilder whereBuilder = new StringBuilder();

			// Create where args array for the conversation ID and all the sequences from the list


			// The value to be changed
			contentValues.put(MessagesTable.KEY_STATUS, messageState.ordinal());
			buildQueryForUpdateStatus(whereBuilder);
			String whereClause = whereBuilder.toString();

			List<SQLiteCommand> commands = new ArrayList<>(sequenceList.length);

			String[] whereArgs;
			for (int sequence : sequenceList) {
				// Add the conversation ID as the first argument of the where
				whereArgs = buildParamsForUpdateStatus(dialogId,messageState, sequence);
				commands.add(new UpdateSQLCommand(contentValues, whereClause, whereArgs));
			}

			getDB().runTransaction(commands);

			LPMobileLog.d(TAG, String.format("Updated " + commands.size() + " messages on DB with state %s, sequences: %s", messageState, Arrays.toString(sequenceList)));

			//updating list
			updateMessages(false, dialogId, sequenceList[0], sequenceList[sequenceList.length - 1]);
		}

	}

	private void buildQueryForUpdateStatus(StringBuilder whereBuilder) {
		whereBuilder.append(MessagesTable.KEY_DIALOG_ID).append(" =? AND ").
                append(MessagesTable.KEY_STATUS).append(" <? AND ").
                append(MessagesTable.KEY_SERVER_SEQUENCE).append(" =?");
	}

	/**
	 * Create where statement to update list of sequences (messages) state
	 *
	 * @param dialogId
	 * @param sequenceList
	 * @param messageState
	 * @param length
	 * @param contentValues
	 * @param whereBuilder
	 * @param whereArgs
	 */
	private void createStatementForUpdateMessagesState(String dialogId, int[] sequenceList, MessagingChatMessage.MessageState messageState, int length, ContentValues contentValues, StringBuilder whereBuilder, String[] whereArgs) {
		// The value to be changed
		contentValues.put(MessagesTable.KEY_STATUS, messageState.ordinal());

		// Add the conversation ID as the first argument of the where
		whereArgs[0] = String.valueOf(dialogId);
		whereArgs[1] = String.valueOf(messageState.ordinal());
		whereBuilder.append(MessagesTable.KEY_DIALOG_ID).append(" =? AND ").
				append(MessagesTable.KEY_STATUS).append(" <?  AND ").
				append(MessagesTable.KEY_SERVER_SEQUENCE).append(" in (");

		// Add all sequences to the where args and to the where clause builder
		for (int i = 0; i < length; i++) {
			whereArgs[i + 2] = String.valueOf(sequenceList[i]);
			whereBuilder.append("?");

			// If last item don't add comma
			if (i != length - 1) {
				whereBuilder.append(",");
			}
		}

		whereBuilder.append(")");
	}

	private UpdateSQLCommand createStatementForUpdateMaxMessagesState(String dialogId,MessagingChatMessage.MessageState messageState, int maxSequence) {
		ContentValues contentValues = new ContentValues();
		StringBuilder whereBuilder = new StringBuilder();

		// The value to be changed
		contentValues.put(MessagesTable.KEY_STATUS, messageState.ordinal());

		// Add the conversation ID as the first argument of the where
		String[] whereArgs = new String[]{
				String.valueOf(dialogId),
				String.valueOf(messageState.ordinal()),
				String.valueOf(maxSequence),
				String.valueOf(PENDING_MSG_SEQUENCE_NUMBER)};

		whereBuilder.append(MessagesTable.KEY_DIALOG_ID).append(" =? AND ").
				append(MessagesTable.KEY_STATUS).append(" <? AND ").
				append(MessagesTable.KEY_SERVER_SEQUENCE).append(" <=? AND " ).
				append(MessagesTable.KEY_SERVER_SEQUENCE).append(" >? ");

		return new UpdateSQLCommand(contentValues, whereBuilder.toString(), whereArgs);


	}

	@NonNull
	private String[] buildParamsForUpdateStatus(String dialogId, MessagingChatMessage.MessageState messageState, int maxSequence) {
		return new String[]{String.valueOf(dialogId), String.valueOf(messageState.ordinal()), String.valueOf(maxSequence)};
	}

	public void updateAllMessagesStateByDialogId(final String targetId, final String dialogId, final MessagingChatMessage.MessageState messageState) {
		DataBaseExecutor.execute(new Runnable() {
			@Override
			public void run() {
				ContentValues contentValues = new ContentValues();
				// The value to be changed
				contentValues.put(MessagesTable.KEY_STATUS, messageState.ordinal());

				int result = getDB().update(contentValues, MessagesTable.KEY_DIALOG_ID + " = ? ", new String[]{dialogId});
				LPMobileLog.d(TAG, String.format("Updated %d messages on DB with state %s", result, messageState));

				//update list listener
				updateAllMessagesForDialog(dialogId);
			}
		});
	}


	// Change the messages state of all the closed conversation from pending to close
	private void markPendingMessagesAsFailedOnCloseConv(final String brandId) {

		DataBaseExecutor.execute(new Runnable() {
			@Override
			public void run() {

				// Get images' rowId currently in upload progress.
				// This is required since we don't want to mark these messages as failed
				String rowIds = MessagingFactory.getInstance().getController().getInProgressUploadMessageRowIdsString();

				String[] whereArgs = getPendingMessagesQueryParams(brandId, rowIds, String.valueOf(ConversationState.CLOSE.ordinal()));
				String where = getPendingMessagesQuery(rowIds);

				//update list listener - need to query all messages that are going to be effected before setting state - than update UI on each message that something changed.
				Cursor cursor = getDB().query(null, where, whereArgs, null, null, null);
				if (cursor != null) {
					try {
						if (cursor.getCount() == 0) {
							//No pending failed messages to update
							return;
						}
						if (cursor.moveToFirst()) {
							do {
								//there are failed messages that need update.
								MessagingChatMessage message = getSingleMessageFromCursor(cursor);
								message.setMessageState(MessagingChatMessage.MessageState.ERROR);
								long rowId = cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_ID));
								updateMessageByRowId(rowId, message);
							} while (cursor.moveToNext());
						}
					} finally {
						cursor.close();
					}

					// Update the DB only if we have result for the query
					ContentValues contentValues = new ContentValues();
					// The value to be changed
					contentValues.put(MessagesTable.KEY_STATUS, MessagingChatMessage.MessageState.ERROR.ordinal());

					//m.status = 3 and c.brandId = "qa81253845" and m.convID = c.conversationId
					int result = getDB().update(contentValues, where, whereArgs);
					LPMobileLog.d(TAG, String.format("Updated %d messages on DB with state %s", result, MessagingChatMessage.MessageState.ERROR));
				}
			}
		});
	}

	public void resendAllPendingMessages(final String brandId) {

		final long resendMessageTimeout = Configuration.getInteger(R.integer.sendingMessageTimeoutInMinutes);

		// Change the messages state of all the closed conversation from pending to close
		markPendingMessagesAsFailedOnCloseConv(brandId);

		DataBaseExecutor.execute(new Runnable() {
			@Override
			public void run() {

				// Get images' rowId currently in upload progress.
				// This is required since we don't want to mark these messages as failed
				String rowIds = MessagingFactory.getInstance().getController().getInProgressUploadMessageRowIdsString();

				String[] whereArgs = getPendingMessagesQueryParams(brandId, rowIds, String.valueOf(ConversationState.OPEN.ordinal()));
				String where = getPendingMessagesQuery(rowIds);

				//update list listener - need to query all messages that are going to be effected before setting state - than update UI on each message that something changed.
				Cursor cursor = getDB().query(null, where, whereArgs, null, null, null);
				if (cursor != null) {
					try {
						if (cursor.getCount() == 0) {
							//No pending messages
							return;
						}
						if (cursor.moveToFirst()) {
							ArrayList<String> eventIdsToErrorUpdate = new ArrayList<>();
							do {
								//Resend all pending messages that didn't pass the timeout
								MessagingChatMessage message = getSingleMessageFromCursor(cursor);
								String dialogId = message.getDialogId();
								if (resendMessageTimeout > 0 && System.currentTimeMillis() < message.getTimeStamp() + TimeUnit.MINUTES.toMillis(resendMessageTimeout)) {
									LPMobileLog.d(TAG, "Resend message: " + message);
									long rowId = Messaging.NO_FILE_ROW_ID;

									// Get rowId in order to pull the image fileMessage from the DB
									if (MessagingChatMessage.MessageType.isImage(message.getMessageType())) {
										FileMessage fileMessage = mController.amsFiles.getFileByMessageRowId(message.getLocalId());
										if (fileMessage != null) {
											rowId = fileMessage.getFileRowId();
										}
									}
									mController.resendMessage(message.getEventId(), dialogId, rowId, message.getMessageType());
								} else {
									LPMobileLog.d(TAG, "Resend timeout - Set message to FAILED state,  resendMessageTimeout:" + resendMessageTimeout + ", message: " + message);
									eventIdsToErrorUpdate.add(message.getEventId());
								}
							} while (cursor.moveToNext());

							if (!eventIdsToErrorUpdate.isEmpty()) {
								// Change message state to failed - resend timeout
								updateMessagesState(eventIdsToErrorUpdate, brandId, null, MessagingChatMessage.MessageState.ERROR);
							}
						}
					} finally {
						cursor.close();
					}
				}
			}
		});
	}

	private String getPendingMessagesQuery(String rowIds) {
		/*
		update Messages set status = error where exists for all the rows in this query:
        (select row_id from Messages m, Conversations c where c.state = (m.status = pending or m.status = queue) and c.brandId = "qa81253845" and m.convID = c.conversationId)
         */
		StringBuilder whereBuilder = new StringBuilder().append(MessagesTable.KEY_ID).append(" in (select m.").append(MessagesTable.KEY_ID).append(" from ")
				.append(MessagesTable.MESSAGES_TABLE).append(" m , ")
				.append(DialogsTable.TABLE_NAME).append(" c ")
				.append("where (m.").append(MessagesTable.KEY_STATUS).append("=").append(MessagingChatMessage.MessageState.PENDING.ordinal())
				.append(" or m.").append(MessagesTable.KEY_STATUS).append("=").append(MessagingChatMessage.MessageState.QUEUED.ordinal())
				.append(") and c.").append(DialogsTable.Key.BRAND_ID).append("=?")
				.append(" and c.").append(DialogsTable.Key.STATE).append("=?")
				.append(" and m.").append(MessagesTable.KEY_DIALOG_ID).append("= c.").append(DialogsTable.Key.DIALOG_ID);

		// If there are images in upload progress add them to the where clause
		if (!TextUtils.isEmpty(rowIds)) {
			LPMobileLog.d(TAG, "resendAllPendingMessages: There is upload images in progress, ignore these messages rowId: " + rowIds);
			whereBuilder.append(" and m.").append(MessagesTable.KEY_ID).append(" not in (?)");
		}

		whereBuilder.append(")");// Close the where clause
		String where = whereBuilder.toString();
		LPMobileLog.d(TAG, "getPendingMessagesQuery: where clause: " + where);
		return where;
	}

	private String[] getPendingMessagesQueryParams(String brandId, String rowIds, String state) {
		// If there are images in upload progress add them to the where clause
		if (!TextUtils.isEmpty(rowIds)) {
			return new String[]{brandId, state, rowIds};
		} else { // where without image rowId
			return new String[]{brandId, state};
		}
	}

	/**
	 * Update the messageState to a message with the given eventId. The targetId and dialogId params
	 * are required here in order to update the model after the update
	 *
	 * @param eventId
	 * @param targetId
	 * @param dialogId
	 * @param messageState
	 */
	public void updateMessageState(final String eventId, final String targetId, final String dialogId, final MessagingChatMessage.MessageState messageState) {

		// SQL query: update messages set status = <messageState> where eventId = <eventId>
		DataBaseExecutor.execute(new Runnable() {
			@Override
			public void run() {

				// Update the status of the message
				ContentValues contentValues = new ContentValues();

				// The value to be changed
				contentValues.put(MessagesTable.KEY_STATUS, messageState.ordinal());

				int result = getDB().update(contentValues, MessagesTable.KEY_EVENT_ID + "=?", new String[]{eventId});

				LPMobileLog.d(TAG, String.format("Updated %d messages on DB with state %s, eventId: %s", result, messageState, eventId));

				//update list listener
				updateMessageByEventId(eventId);
			}
		});
	}

	/**
	 * Update the messageState to array of messages based on given eventId. The targetId and conversationId params
	 * are required here in order to update the model after the update
	 *
	 * @param eventIds
	 * @param targetId
	 * @param dialogId
	 * @param messageState
	 */
	public void updateMessagesState(final ArrayList<String> eventIds, final String targetId, final String dialogId, final MessagingChatMessage.MessageState messageState) {

		// SQL query: update messages set status = <messageState> where eventId = <eventId>
		DataBaseExecutor.execute(new Runnable() {
			@Override
			public void run() {

				if (eventIds != null && !eventIds.isEmpty()) {
					// Update the status of the message
					ContentValues contentValues = new ContentValues();

					// The value to be changed
					contentValues.put(MessagesTable.KEY_STATUS, messageState.ordinal());

					String inOperator = " IN (?";
					for (int i = 1; i < eventIds.size(); i++) {
						inOperator += ",?";
					}
					inOperator += ")";

					String[] eventIdsArray = new String[eventIds.size()];
					eventIdsArray = eventIds.toArray(eventIdsArray);

					int result = getDB().update(contentValues, MessagesTable.KEY_EVENT_ID + inOperator, eventIdsArray);

					LPMobileLog.d(TAG, String.format("Updated %d messages on DB with state %s, eventId: %s", result, messageState, eventIds));

					for (String eventId : eventIdsArray) {
						//update list listener
						updateMessageByEventId(eventId);
					}

				} else {
					LPMobileLog.d(TAG, "updateMessagesState - Skip updated messages , eventIdis is empty. messageState = " + messageState);
				}
			}

		});
	}

	public void sendReadAckOnMessages(final String brandId, final String targetId, final String originatorId) {

		DataBaseExecutor.execute(new Runnable() {
			@Override
			public void run() {

				// select serverSequence from messages where conversation state != CLOSED || LOCKED and targetId=<targetId> and messageState=ACCEPT
				Cursor cursor = null;
				if (!TextUtils.isEmpty(targetId)) {

					LPMobileLog.d(TAG, "Get all unread messages for target " + targetId);

					String queryString = String.format("select m.%s, c.%s, c.%s from %s m, %s c where c.%s>? and c.%s=? and m.%s =? and m.%s=c.%s and m.%s >= '0' and m.%s != ?",
							MessagesTable.KEY_SERVER_SEQUENCE, DialogsTable.Key.DIALOG_ID, DialogsTable.Key.CONVERSATION_ID, MessagesTable.MESSAGES_TABLE, DialogsTable.TABLE_NAME,
							DialogsTable.Key.STATE,DialogsTable.Key.TARGET_ID, MessagesTable.KEY_STATUS, MessagesTable.KEY_DIALOG_ID,
							DialogsTable.Key.DIALOG_ID, MessagesTable.KEY_SERVER_SEQUENCE, MessagesTable.KEY_ORIGINATOR_ID);

					//select m.serverSequence, c.conversationId from messages m, conversations c where c.messageState>? and c.targetId=? and m.status =? and m.convID=c.conversationId and m.serverSequence != '-1' and m.originatorId != ?
					cursor = getDB().rawQuery(queryString, ConversationState.LOCKED.ordinal() , targetId, MessagingChatMessage.MessageState.RECEIVED.ordinal(), originatorId);
				} else if (!TextUtils.isEmpty(brandId)) {
					LPMobileLog.d(TAG, "Get all unread messages for brand " + brandId );

					String queryString = String.format("select m.%s, c.%s, c.%s  from %s m, %s c where c.%s=? and c.%s>? and m.%s =? and m.%s=c.%s and m.%s >= '0' and m.%s != ?",
							MessagesTable.KEY_SERVER_SEQUENCE, DialogsTable.Key.DIALOG_ID, DialogsTable.Key.CONVERSATION_ID, MessagesTable.MESSAGES_TABLE, DialogsTable.TABLE_NAME,
							DialogsTable.Key.BRAND_ID, DialogsTable.Key.STATE, MessagesTable.KEY_STATUS,
							MessagesTable.KEY_DIALOG_ID, DialogsTable.Key.DIALOG_ID, MessagesTable.KEY_SERVER_SEQUENCE, MessagesTable.KEY_ORIGINATOR_ID);

					//select m.serverSequence, c.conversationId from messages m, conversations c where c.brandId=? and m.status =? and m.convID=c.conversationId and m.serverSequence != '-1' and m.originatorId != ?
					cursor = getDB().rawQuery(queryString, brandId, ConversationState.LOCKED.ordinal() , MessagingChatMessage.MessageState.RECEIVED.ordinal(), originatorId);
				}

				HashMap<String, List<Integer>> sequenceMap = new HashMap<>();
				HashMap<String, String> dialogIdsAndConversationIds = new HashMap<>();

				if (cursor != null) {
					try {
						List<Integer> list;

						if (cursor.moveToFirst()) {
							do {
								String dialogId = cursor.getString(cursor.getColumnIndex(DialogsTable.Key.DIALOG_ID));
								String conversationId = cursor.getString(cursor.getColumnIndex(DialogsTable.Key.CONVERSATION_ID));
								dialogIdsAndConversationIds.put(dialogId, conversationId);

								list = sequenceMap.get(dialogId);
								if (list == null) {
									list = new ArrayList<>();
									sequenceMap.put(dialogId, list);
								}

								int sequence = cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_SERVER_SEQUENCE));

								list.add(sequence);

							} while (cursor.moveToNext());
						}


					} finally {
						cursor.close();
					}
				} else {
					// Nothing to do
					return;
				}

				for (String dialogId : sequenceMap.keySet()) {
					LPMobileLog.d(TAG, "Send a read ack to the server for dialog id " + dialogId + " on the following sequences: " + sequenceMap.get(dialogId));

					// We have the sequence list of the messages we need to ack, run the command for updating the server
					String conversationId = dialogIdsAndConversationIds.get(dialogId);
					new DeliveryStatusUpdateCommand(mController.mAccountsController.getConnectionUrl(brandId), (!TextUtils.isEmpty(targetId) ? targetId : brandId), dialogId, conversationId, sequenceMap.get(dialogId)).execute();
				}
			}
		});
	}

	public void setDeliveryStatusUpdateCommand(Form form, DeliveryStatus deliveryStatus) {
		if (form == null) {
			LPMobileLog.w(TAG, "form not found!");
			return;
		}
		new DeliveryStatusUpdateCommand(mController.mAccountsController.getConnectionUrl(form.getSiteId()), form.getSiteId(), form.getDialogId(), form.getConversationId(), form.getSeqId(), deliveryStatus);
	}

	private ContentValues getContentValuesForMessage(final MessagingChatMessage message) {
		ContentValues messageValues = getContentValuesForMessageUpdate(message);
		messageValues.put(MessagesTable.KEY_EVENT_ID, message.getEventId());
		return messageValues;
	}

	private ContentValues getContentValuesForMessageUpdate(final MessagingChatMessage message) {
		ContentValues messageValues = new ContentValues();
		messageValues.put(MessagesTable.KEY_SERVER_SEQUENCE, message.getServerSequence());
		messageValues.put(MessagesTable.KEY_DIALOG_ID, message.getDialogId());

		// Text is encrypted before saving to DB
		final EncryptionVersion messageEncryptionVersion = DBEncryptionKeyHelper.getAppEncryptionVersion(Infra.instance.getApplicationContext());
		messageValues.put(MessagesTable.KEY_ENCRYPTION_VERSION, messageEncryptionVersion.ordinal());

		String encryptedMessage = DBEncryptionHelper.encrypt(messageEncryptionVersion, message.getMessage());

		messageValues.put(MessagesTable.KEY_TEXT, encryptedMessage);

		messageValues.put(MessagesTable.KEY_CONTENT_TYPE, message.getContentType());
		messageValues.put(MessagesTable.KEY_MESSAGE_TYPE, message.getMessageType().ordinal());
		messageValues.put(MessagesTable.KEY_STATUS, message.getMessageState().ordinal());
		messageValues.put(MessagesTable.KEY_TIMESTAMP, message.getTimeStamp());
		messageValues.put(MessagesTable.KEY_ORIGINATOR_ID, message.getOriginatorId());
		return messageValues;
	}


	static private MessagingChatMessage getSingleMessageFromCursor(Cursor cursor) {
		long msgId = cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_ID));

		MessagingChatMessage message = new MessagingChatMessage(
				cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_ORIGINATOR_ID)),
				cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_TEXT)),
				cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_TIMESTAMP)),
				cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_DIALOG_ID)),
				cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_EVENT_ID)),
				MessagingChatMessage.MessageType.values()[cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_MESSAGE_TYPE))],
				MessagingChatMessage.MessageState.values()[cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_STATUS))],
				cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_SERVER_SEQUENCE)),
				cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_CONTENT_TYPE)),
				EncryptionVersion.fromInt(cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_ENCRYPTION_VERSION))));

		message.setMessageId(msgId);
		return message;
	}

	/**
	 * Update the Message Server Sequence & Message Status by event id
	 *
	 * @param targetId
	 * @param dialogId
	 * @param eventId
	 * @param serverSequence
	 */
	public void updateOnMessageAck(final String targetId, final String dialogId, final String eventId, final long serverSequence) {
		DataBaseExecutor.execute(new Runnable() {
			@Override
			public void run() {
				{
					// Update the message server sequence
					final ContentValues contentValuesSequence = new ContentValues();
					contentValuesSequence.put(MessagesTable.KEY_SERVER_SEQUENCE, serverSequence);
					final String whereClauseSequence = MessagesTable.KEY_EVENT_ID + "=?";
					final String[] whereArgsSequence = {String.valueOf(eventId)};
					final int rowsAffected = getDB().update(contentValuesSequence, whereClauseSequence, whereArgsSequence);
					LPMobileLog.d(TAG, "Update msg server seq query. Rows affected=" + rowsAffected + " Seq=" + serverSequence);
				}

				{
					// Update the message status if relevant
					final ContentValues contentValuesStatus = new ContentValues();
					contentValuesStatus.put(MessagesTable.KEY_STATUS, MessagingChatMessage.MessageState.SENT.ordinal());
					final String whereClauseStatus = MessagesTable.KEY_EVENT_ID + "=? AND (" + MessagesTable.KEY_STATUS + "=? OR " + MessagesTable.KEY_STATUS + "=?)";
					final String[] whereArgsStatus = {String.valueOf(eventId), String.valueOf(MessagingChatMessage.MessageState.PENDING.ordinal()), String.valueOf(MessagingChatMessage.MessageState.ERROR.ordinal())}; // Update to SENT but only if current status is PENDING or ERROR (avoid lowering the status if messages are not arriving from the server in order)
					final int rowsAffected = getDB().update(contentValuesStatus, whereClauseStatus, whereArgsStatus);
					LPMobileLog.d(TAG, "Update msg status to SENT. Rows affected=" + rowsAffected);
				}

				//update list listener
				updateMessageByEventId(eventId);
			}
		});
	}


	/**
	 * Get the time of the first message in the db.
	 * This is for adding message before the first message.
	 *
	 * @return
	 */
	public DataBaseCommand<Long> getTimeOfFirstMessage() {
		return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<Long>() {
			@Override
			public Long query() {
				String min_time = "MIN_TIME";
				Cursor cursor = getDB().rawQuery("SELECT MIN( " + MessagesTable.KEY_TIMESTAMP + ") AS " + min_time + " FROM " + MessagesTable.MESSAGES_TABLE);
				Long oldestTimestamp = System.currentTimeMillis();
				if (cursor != null) {
					try {
						if (cursor.moveToFirst()) {
							long minTime = cursor.getLong(cursor.getColumnIndex(min_time));
							if (minTime > 0) {
								oldestTimestamp = minTime;
							}
						}
					} finally {
						cursor.close();
					}
				}
				LPMobileLog.d(TAG, "getTimeOfFirstMessage , oldestTimestamp = " + oldestTimestamp);
				return oldestTimestamp;
			}
		});
	}

	public DataBaseCommand<Boolean> isFirstMessageExists() {
		return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<Boolean>() {
			@Override
			public Boolean query() {
				Cursor cursor = getDB().query(new String[]{MessagesTable.KEY_ID}, MessagesTable.KEY_SERVER_SEQUENCE + " = ?", new String[]{String.valueOf(WELCOME_MSG_SEQUENCE_NUMBER)}, null, null, null);

				boolean firstMessageExists = false;
				if (cursor != null) {
					try {
						if (cursor.moveToFirst()) {
							long firstMessageId = cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_ID));
							if (firstMessageId != -1) {
								firstMessageExists = true;
							}
						}
					} finally {
						cursor.close();
					}
				}
				LPMobileLog.d(TAG, "isFirstMessageExists = " + firstMessageExists);
				return firstMessageExists;
			}
		});
	}

	public DataBaseCommand<Boolean> isLastMessageWelcomeMessage() {
		return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<Boolean>() {
			@Override
			public Boolean query() {
				Cursor cursor = getDB().query(null, null, null, null, null, null);
				cursor.moveToLast();
				if (cursor.getCount() == 0) {
					return false;
				}
				return cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_SERVER_SEQUENCE)) == WELCOME_MSG_SEQUENCE_NUMBER;
			}
		});
	}

	public DataBaseCommand updateLastWelcomeMessage(MessagingChatMessage message) {
		return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<Void>() {
			@Override
			public Void query() {
				Cursor cursor = getDB().query(null, null, null, null, null, null);
				cursor.moveToLast();
				int id = cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_ID));
				String whereString = MessagesTable.KEY_ID + "=?";
				String[] whereArgs = {String.valueOf(id)};
				getDB().update(getContentValuesForMessageUpdate(message), whereString, whereArgs);
				return null;
			}
		});
	}

	/**
	 * Remove welcome message if it's last message and not first message in database.
	 */
	public DataBaseCommand removeLastWelcomeMessage() {
		return new DataBaseCommand(new DataBaseCommand.QueryCommand<Void>() {
			@Override
			public Void query() {
				Cursor cursor = getDB().query(null, null, null, null, null, null);
				cursor.moveToFirst();
				int firstItemId = cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_ID));
				cursor.moveToLast();
				int lastItemId = cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_ID));
				if (firstItemId != lastItemId) {
					String whereString = MessagesTable.KEY_ID + "=?";
					String[] whereArgs = {String.valueOf(lastItemId)};
					getDB().removeAll(whereString, whereArgs);
				}
				return null;
			}
		});
	}

	/**
	 * Update messages serverId for messages that are waiting for create dialog response
	 *
	 * @param tempDialogId - old dialog temporary ID
	 * @param serverDialogId      - new dialog ID
	 * @return list of the updated messages
	 */
	public DataBaseCommand<Void> updateMessagesDialogServerID(final String tempDialogId, final String serverDialogId) {
		return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<Void>() {
			@Override
			public Void query() {
				ContentValues contentValues = new ContentValues();
				contentValues.put(MessagesTable.KEY_DIALOG_ID, serverDialogId);
				int updatedRows = getDB().update(contentValues, MessagesTable.KEY_DIALOG_ID + "=? ", new String[]{tempDialogId});

				LPMobileLog.d(TAG, "updateMessagesConversationServerID , updatedRows = " + updatedRows);
				//update list listeners
				updateAllMessagesForDialog(serverDialogId);
				return null;
			}
		});

	}

	/**
	 * Update messages serverId  and timestamp for message by row id
	 *
	 * @param messageRowId   - specific row to update
	 * @param dialogId - new dialog ID
	 * @return list of the updated messages
	 */
	public DataBaseCommand<Void> updateMessageDialogServerIdAndTime(final long messageRowId, final String dialogId, final long time) {

		return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<Void>() {
			@Override
			public Void query() {
				ContentValues contentValues = new ContentValues();
				contentValues.put(MessagesTable.KEY_DIALOG_ID, dialogId);
				//contentValues.put(MessagesTable.KEY_TIMESTAMP, time);
				int updatedRows = getDB().update(contentValues, MessagesTable.KEY_ID + "=? ", new String[]{String.valueOf(messageRowId)});

				LPMobileLog.d(TAG, "updateMessageDialogServerIdAndTime , rowId to update = " + messageRowId + ", updated = " + updatedRows);
				//update list listeners
				updateMessageByRowIdOnDbThread(messageRowId);
				return null;
			}
		});

	}

	/**
	 * Update messages state for message by row id
	 *
	 * @param messageRowId - specific row to update
	 * @param state        - new state to set
	 * @return list of the updated messages
	 */
	public void updateMessageState(final long messageRowId, final MessagingChatMessage.MessageState state) {

		DataBaseExecutor.execute(new Runnable() {
			@Override
			public void run() {

				{
					ContentValues contentValues = new ContentValues();
					contentValues.put(MessagesTable.KEY_STATUS, state.ordinal());
					//contentValues.put(MessagesTable.KEY_TIMESTAMP, time);
					int updatedRows = getDB().update(contentValues, MessagesTable.KEY_ID + "=? ", new String[]{String.valueOf(messageRowId)});

					LPMobileLog.d(TAG, "updateMessageState , rowId to update = " + messageRowId + ", updated = " + updatedRows);
					//update list listeners
					updateMessageByRowIdOnDbThread(messageRowId);
				}
			}
		});

	}

	/**
	 * Update message related to file when file updated
	 *
	 * @param targetId
	 * @param messageRowId - specific row to update
	 */
	public void updateMessageFileChanged(final String targetId, final long messageRowId) {
		if (messageRowId < 0) {
			LPMobileLog.w(TAG, "updateMessageFileChanged cannot be lower than zero! " + messageRowId);
			return;
		}
		DataBaseExecutor.execute(new Runnable() {
			@Override
			public void run() {
				updateMessageByRowIdOnDbThread(messageRowId);
			}
		});
	}

    private String[] getProjection() {
        return MessagesTable.getProjection();
    }

	/**
	 * Search a specific string whithin all messages.
	 * It finds the latest message per brand that contains the searched string
	 *
	 * @param searchString
	 * @return
	 */
	public DataBaseCommand<List<MessagingSearchedMessage>> searchMessages(final String searchString) {
		return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<List<MessagingSearchedMessage>>() {
			@Override
			public List<MessagingSearchedMessage> query() {
				// select c.brandId, m.text, m.type, u.firstName, max(m.timeStamp)  from messages m, conversations c, users u where (m.text like '%1234%’ or (u.firstName like '%1234%’ and m.type=1)) and m.type <> 0 and m.convID=c.conversationId and m.originatorId=u.originatorId group by c.brandId;

				String queryString = String.format("select c.%1$s, m.%2$s, m.%3$s, u.%4$s, max(m.%5$s) from %6$s m, %7$s c, %8$s u where (m.%2$s like '%%%9$s%%' or (u.%4$s like '%%%9$s%%' and m.%3$s=1)) and m.%3$s<> 0 and m.%10$s=c.%11$s and m.%12$s=u.%13$s group by c.%1$s;",
						DialogsTable.Key.BRAND_ID, // 1$
						MessagesTable.KEY_TEXT, // 2$ // TODO: 2/8/16 fix (text might be encrypted)
						MessagesTable.KEY_MESSAGE_TYPE, // 3$
						UsersTable.KEY_FIRST_NAME, // 4$
						MessagesTable.KEY_TIMESTAMP, // 5$
						MessagesTable.MESSAGES_TABLE, // 6$
						DialogsTable.TABLE_NAME, // 7$
						UsersTable.USERS_TABLE, // 8$
						searchString, // 9$
						MessagesTable.KEY_DIALOG_ID, // 10$
						DialogsTable.Key.DIALOG_ID, // 11$
						MessagesTable.KEY_ORIGINATOR_ID, // 12$
						UsersTable.KEY_ORIGINATOR_ID // 13$
				);

				LPMobileLog.d(TAG, "query: " + queryString);

				// Query the DB
				Cursor cursor = getDB().rawQuery(queryString);

				// Go over the cursor and fills the list with MessagingSearchedMessage objects
				List<MessagingSearchedMessage> searchedMessageList = new ArrayList<>();
				if (cursor != null) {
					MessagingSearchedMessage searchedMessage;
					try {
						if (cursor.moveToFirst()) {
							do {
								searchedMessage = MessagingSearchedMessage.fromCursor(cursor);

								if (searchedMessage != null) {
									searchedMessageList.add(searchedMessage);
								}
							} while (cursor.moveToNext());

						}
					} finally {
						cursor.close();
					}
				}
				return searchedMessageList;
			}
		});
	}

	public DataBaseCommand<int[]> getSequencesForDialog(final String dialogId) {
		return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<int[]>() {
			@Override
			public int[] query() {

				// Query the DB
				String queryString = new StringBuilder()
						.append("SELECT ").append(MessagesTable.KEY_SERVER_SEQUENCE)
						.append(" FROM ").append(MessagesTable.MESSAGES_TABLE)
						.append(" WHERE ").append(MessagesTable.KEY_DIALOG_ID).append(" = ? ")
						.append(" ORDER BY ").append(MessagesTable.KEY_SERVER_SEQUENCE).append(" DESC")
						.toString();
				Cursor cursor = getDB().rawQuery(queryString, new String[]{dialogId});

//				String queryString = new StringBuilder()
//						.append("SELECT ").append(MessagesTable.KEY_SERVER_SEQUENCE)
//						.append(" FROM ").append(MessagesTable.MESSAGES_TABLE)
//						.append(" WHERE ").append(MessagesTable.KEY_CONVERSATION_ID).append(" = ? ")
//						.append(" ORDER BY ").append(MessagesTable.KEY_SERVER_SEQUENCE).append(" DESC").append(conversationId)
//						.toString();
//				Cursor cursor = getDB().rawQuery(queryString);

				int[] searchedMessageList = null;
				if (cursor != null) {
					try {
						searchedMessageList = new int[(cursor.getCount())];
						if (cursor.moveToFirst()) {
							do {
								searchedMessageList[cursor.getPosition()] = cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_SERVER_SEQUENCE));
							} while (cursor.moveToNext());
						}

					} finally {
						cursor.close();
					}
				}

				return searchedMessageList;
			}
		});
	}

	/***************************
	 * UPDATE UI LISTENER
	 ******************************/

	private MessagesListener getMessagesListener() {
		if (mMessagesListener != null){
			return mMessagesListener;
		}
		return mNullMessagesListener;
	}

	//region AmsMessagesLoaderProvider
    /**
     * Adding MessagesListener to get updates for changes in messages
     *
     * @param messagesListener     listener to notify on updates
     * @param messagesSortedByType way to sory messages by brand or target id
     * @param typeValue            value for the selected messagesSortedByType - the target id or brand id
     */
    @Override
    public void addOnUpdateListener(MessagesListener messagesListener, final MessagesSortedBy messagesSortedByType, final String typeValue) {
        mMessagesListener = messagesListener;
        DataBaseExecutor.execute(new Runnable() {
            @Override
            public void run() {
                // Query the DB
                ArrayList<FullMessageRow> searchedMessageList = loadMessagesOnDbThread(messagesSortedByType, typeValue, 100, -1, -1);
                getMessagesListener().initMessages(searchedMessageList);
            }
        });

    }

    @Override
    public void removeOnUpdateListener() {
        mMessagesListener = null;
    }

    @Override
    public boolean hasListener() {
        return mMessagesListener != null;
    }

	/**
	 * Loading messages by specific rules:
	 *
	 * @param messagesSortedByType way to sory messages by brand or target id
	 * @param typeValue            value for the selected messagesSortedByType - the target id or brand id
	 * @param limit                max num of messages required.  0 or negative value for no limit
	 * @param olderThanTimestamp   negative value if there is no need to use this value
	 * @param newerThanTimestamp   negative value if there is no need to use this value
	 * @return
	 */
	@Override
	public DataBaseCommand<ArrayList<FullMessageRow>> loadMessages(final MessagesSortedBy messagesSortedByType, final String typeValue, final int limit, final long olderThanTimestamp, final long newerThanTimestamp) {
		return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<ArrayList<FullMessageRow>>() {
			@Override
			public ArrayList<FullMessageRow> query() {
				// Query the DB
				return loadMessagesOnDbThread(messagesSortedByType, typeValue, limit, olderThanTimestamp, newerThanTimestamp);
			}
		});

	}

	@Nullable
	private ArrayList<FullMessageRow> loadMessagesOnDbThread(MessagesSortedBy messagesSortedByType, String typeValue, int limit, long olderThanTimestamp, long newerThanTimestamp) {
		Cursor cursor = null;
		switch (messagesSortedByType) {
			case TargetId:
				cursor = messagesByTarget(typeValue, limit, olderThanTimestamp, newerThanTimestamp);
				break;
			case ConversationId:
				cursor = messagesByConversationID(typeValue, limit);
				break;
		}

		ArrayList<FullMessageRow> searchedMessageList = null;
		if (cursor != null) {
			try {
				searchedMessageList = new ArrayList<>(cursor.getCount());
				if (cursor.moveToFirst()) {
					do {
						searchedMessageList.add(new FullMessageRow(cursor));
					} while (cursor.moveToNext());
				}
			} finally {
				cursor.close();
			}
		}
		return searchedMessageList;
	}

	@Override
	public String getMyUserId(String targetId) {
		return mController.getOriginatorId(targetId);
	}

	@Override
	public MessagingUserProfile loadMessagingUserProfile(String originatorId) {
		return mController.amsUsers.getUserById(originatorId).executeSynchronously();
	}
	//endregion

	@NonNull
	private FullMessageRow createFullMessageRow(long messageRowId, @NonNull MessagingChatMessage message, long fileRowId) {
		MessagingUserProfile messagingUserProfile = mController.amsUsers.getUserById(message.getOriginatorId()).executeSynchronously();
		String avatarUrl = messagingUserProfile == null ? "" : messagingUserProfile.getAvatarUrl();
		FileMessage fileMessage;
		if (fileRowId != -1) {
			fileMessage = mController.amsFiles.getFileByFileRowIdOnDbThread(fileRowId);
		} else {
			fileMessage = mController.amsFiles.getFileByMessageRowId(messageRowId);
		}
		return new FullMessageRow(message, avatarUrl, fileMessage);
	}

	/**
	 * Clear all closed conversation messages of the given targetId
	 *
	 * @param targetId
	 * @return the number of messages removed
	 */
	public DataBaseCommand<Integer> clearMessagesOfClosedConversations(final String targetId) {

		return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<Integer>() {

			@Override
			public Integer query() {
				//"where messages._id in (select m._id from messages m, dialogs d where d.targetId='21337028' and d.state=0 and d.dialog_id=m.dialogId or m.dialogId=KEY_WELCOME_DIALOG_ID)"
				String whereString = MessagesTable.KEY_ID + " in (select m." + MessagesTable.KEY_ID + " from " + MessagesTable.MESSAGES_TABLE + " m, "
						+ DialogsTable.TABLE_NAME + " d where d." + DialogsTable.Key.TARGET_ID + "=? and d." + DialogsTable.Key.STATE
						+ "=? and d." + DialogsTable.Key.DIALOG_ID+ "=m." + MessagesTable.KEY_DIALOG_ID + " or m." + MessagesTable.KEY_DIALOG_ID + "=?)";
				String[] whereArgs = {targetId, String.valueOf(DialogState.CLOSE.ordinal()), AmsDialogs.KEY_WELCOME_DIALOG_ID}; // Closed conversations for targetId


				int removed = getDB().removeAll(whereString, whereArgs);
				LPMobileLog.d(TAG, "clearMessagesOfClosedConversations: removed: " + removed + " where: " + whereString + ", whereArgs: " + Arrays.toString(whereArgs));

				//update list listeners
				updateMessagesRemoved(targetId);
				return removed;
			}
		});
	}

	/**
	 * Clear all closed conversation messages of the given targetId
	 *
	 * @param targetId
	 * @return the number of messages removed
	 */
	public DataBaseCommand<Integer> clearAllMessages(final String targetId) {

		return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<Integer>() {

			@Override
			public Integer query() {
				//"where messages._id in (select m._id from messages m, conversations c where c.targetId='21337028' and c.state=0 and c.conversationId=m.convID)"
				String whereString = MessagesTable.KEY_ID + " in (select m." + MessagesTable.KEY_ID + " from " + MessagesTable.MESSAGES_TABLE + " m, "
						+ DialogsTable.TABLE_NAME + " d where d." + DialogsTable.Key.TARGET_ID + "=? and d." + DialogsTable.Key.DIALOG_ID+ "=m." + MessagesTable.KEY_DIALOG_ID + ")";
				String[] whereArgs = {targetId}; // All messages for targetId


				int removed = getDB().removeAll(whereString, whereArgs);
				LPMobileLog.d(TAG, "clearAllMessages: removed: " + removed + " where: " + whereString + ", whereArgs: " + Arrays.toString(whereArgs));

				//update list listeners
				updateMessagesRemoved(targetId);
				return removed;
			}
		});
	}

    /**
     * RUN ONLY ON DB THREAD!!
     *
	 * @param firstNotification
	 * @param dialogId
	 * @param firstSequence
	 * @param lastSequence
	 */
    private void updateMessages(boolean firstNotification, String dialogId, int firstSequence, int lastSequence) {
        long firstTimestamp = getTimestampMessage(dialogId, firstSequence);
        long lastTimestamp = getTimestampMessage(dialogId, lastSequence);
        if (firstNotification){
			LPMobileLog.d(TAG, "updateMessages first notification event. onQueryMessagesResult ");
	        getMessagesListener().onQueryMessagesResult(firstTimestamp, lastTimestamp);
		}else{
			LPMobileLog.d(TAG, "updateMessages NOT first notification event. onUpdateMessages ");
			getMessagesListener().onUpdateMessages(firstTimestamp, lastTimestamp);
		}
    }

    /**
     * RUN ONLY ON DB THREAD!!
     *
     * @param targetId
     */
    private void updateMessagesRemoved(String targetId) {
        getMessagesListener().removeAll(targetId);
    }

    /**
     * RUN ONLY ON DB THREAD!!
     *
     * @param eventId
     */
    private void updateMessageByEventId(String eventId) {
        Cursor cursor = getDB().query(null, MessagesTable.KEY_EVENT_ID + " = ?", new String[]{eventId}, null, null, null);
        if (cursor != null) {
            try {
                if (cursor.moveToFirst()) {
                    MessagingChatMessage message = getSingleMessageFromCursor(cursor);
                    int rowId = cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_ID));
                    getMessagesListener().onUpdateMessage(createFullMessageRow(rowId, message, -1));
                }
            } finally {
                cursor.close();
            }
        }

	}

	/**
	 * RUN ONLY ON DB THREAD!!
	 *
	 * @param dialogId
	 * @param sequence
	 */
	private String getEventIdForMessage(String dialogId, int sequence) {
		String eventId = null;
		Cursor cursor = getDB().rawQuery("SELECT " + MessagesTable.KEY_EVENT_ID + " FROM " + MessagesTable.MESSAGES_TABLE + " WHERE " + MessagesTable.KEY_DIALOG_ID
				+ " =? AND " + MessagesTable.KEY_SERVER_SEQUENCE + " =? ", dialogId, sequence);

		if (cursor != null) {
			try {
				if (cursor.moveToFirst()) {
					eventId = cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_EVENT_ID));
				}
			} finally {
				cursor.close();
			}
		}
		return eventId;
	}


	/**
	 * Get all messages of a dialog by the dialogId excluding the messages with the type:
	 * SYSTEM_RESOLVED and SYSTEM_DIALOG_RESOLVED
	 * */
	public DataBaseCommand<ArrayList<MessagingChatMessage>> getMessagesByDialogId(@Nullable final String dialogId) {
		return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<ArrayList<MessagingChatMessage>>() {
			@Override
			public ArrayList<MessagingChatMessage> query() {
				if (TextUtils.isEmpty(dialogId)) {
					LPMobileLog.w(TAG, "getMessagesByDialogId - dialogId is empty");
					return new ArrayList<>();
				}

				Cursor cursor = null;
				try {
					String selection = MessagesTable.KEY_DIALOG_ID + " = ? AND " + MessagesTable.KEY_MESSAGE_TYPE + " != ? AND " + MessagesTable.KEY_MESSAGE_TYPE + " != ?";
					String[] args = new String[]{
							dialogId,
							String.valueOf(MessagingChatMessage.MessageType.SYSTEM_RESOLVED.ordinal()),
							String.valueOf(MessagingChatMessage.MessageType.SYSTEM_DIALOG_RESOLVED.ordinal())
					};

					cursor = getDB().query(getProjection(), selection, args, null, null, null);
					if (cursor != null) {
						return getMessagesFromCursor(cursor);
					}
				} finally {
					if (cursor != null) {
						cursor.close();
					}
				}
				return null;
			}
		});
	}

	public DataBaseCommand<MessagingChatMessage> getMessageByEventId(final String eventId) {
		if (TextUtils.isEmpty(eventId)) {
			LPMobileLog.w(TAG, "getMessageByEventId - eventId is empty");
			return null;
		}
		return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<MessagingChatMessage>() {
			@Override
			public MessagingChatMessage query() {
				Cursor cursor = null;
				try {
					cursor = getDB().query(getProjection(), MessagesTable.KEY_EVENT_ID + " = ?", new String[]{eventId}, null, null, null);
					if (cursor != null) {
						return getSingleMessageFromCursor(cursor);
					}
				} finally {
					if (cursor != null) {
						cursor.close();
					}
				}
				return null;
			}
		});
	}

	@NonNull
 	private static ArrayList<MessagingChatMessage> getMessagesFromCursor(Cursor cursor) {
		ArrayList<MessagingChatMessage> messages = new ArrayList<>();
		if (cursor != null) {
			try {
				if (cursor.moveToFirst()) {
					do {
						messages.add(getSingleMessageFromCursor(cursor));
					} while (cursor.moveToNext());
				}
			} catch (Exception e) {
				LPMobileLog.e(TAG, e);
			}
		}

		return messages;
	}

	public DataBaseCommand<Long> getRowIdByEventId(final String eventId) {
		if (TextUtils.isEmpty(eventId)) {
			LPMobileLog.w(TAG, "getRowIdByEventId - eventId is empty");
			return null;
		}
		return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<Long>() {
			@Override
			public Long query() {
				Cursor cursor = null;
				try {
					cursor = getDB().query(new String[]{MessagesTable.KEY_ID}, MessagesTable.KEY_EVENT_ID + " = ?", new String[]{eventId}, null, null, null);
					if (cursor != null) {
						return cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_ID));
					}
				} catch (Exception e) {

				} finally {
					if (cursor != null) {
						cursor.close();
					}
				}
				return -1L;
			}
		});
	}

    /**
     * RUN ONLY ON DB THREAD!!
     *
     * @param messageRowId
     */
    private void updateMessageByRowIdOnDbThread(long messageRowId) {
        MessagingChatMessage message = getMessageByRowIdOnDbThread(messageRowId);
        if (message != null){
			getMessagesListener().onUpdateMessage(createFullMessageRow(messageRowId, message, -1));
		} else {
			LPMobileLog.e(TAG, "updateMessageByRowIdOnDbThread - message does not exist");
		}
    }

	public DataBaseCommand<Void> updateFileMessageByRowId(final long messageRowId, final long fileRowId) {

		return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<Void>() {

            @Override
            public Void query() {
                MessagingChatMessage message = getMessageByRowIdOnDbThread(messageRowId);
                getMessagesListener().onUpdateMessage(createFullMessageRow(messageRowId, message, fileRowId));
                return null;
            }
        });
    }

    @Nullable
	private MessagingChatMessage getMessageByRowIdOnDbThread(long messageRowId) {
		Cursor cursor = getDB().query(null, MessagesTable.KEY_ID + " = ?", new String[]{String.valueOf(messageRowId)}, null, null, null);
		if (cursor != null) {
			try {
				if (cursor.moveToFirst()) {
					return getSingleMessageFromCursor(cursor);
				}
			} finally {
				cursor.close();
			}
		}
		return null;
	}

    /**
     * RUN ONLY ON DB THREAD!!
     *
     * @param rowId
     * @param message
     */
    private void updateMessageByRowId(long rowId, MessagingChatMessage message) {
        getMessagesListener().onUpdateMessage(createFullMessageRow(rowId, message, -1));
    }

    /**
     * RUN ONLY ON DB THREAD!!
     *
     * @param dialogId
     */
    private void updateAllMessagesForDialog(String dialogId) {
        Cursor cursor = getDB().query(new String[]{"MIN(" + MessagesTable.KEY_TIMESTAMP + ")", "MAX(" + MessagesTable.KEY_TIMESTAMP + ")"},
                MessagesTable.KEY_DIALOG_ID + " = ?", new String[]{dialogId}, null, null, null);
        if (cursor != null) {
            try {
                if (cursor.moveToFirst()) {
                    long firstMessageTimestampForConversation = cursor.getLong(0);
                    long lastMessageTimestampForConversation = cursor.getLong(1);
                    getMessagesListener().onUpdateMessages(firstMessageTimestampForConversation, lastMessageTimestampForConversation);

				}
			} finally {
				cursor.close();
			}
		}
	}


	/**
	 * updating messages after fetch history.
	 *
	 * @param brandId
	 */
	public void updateFetchHistoryEnded(String brandId) {
		updateFetchHistoryEnded(brandId, true);
	}

	public void updateFetchHistoryEnded(final String brandId, final boolean updated) {

		DataBaseExecutor.execute(new Runnable() {
			@Override
			public void run() {

                if (updated) {
                    getMessagesListener().onHistoryFetched();
                } else {
                    getMessagesListener().onHistoryFetchedFailed();
                }
            }
        });

	}

	/**
	 * Updating messages when agent details relevant for conversation was updated.
	 *
	 * @param brandId
	 * @param dialogId
	 */
	public void updateAgentDetailsUpdated(String brandId, String dialogId) {
		updateAllMessagesForDialog(dialogId);
	}

    public void updateHandledExConversation(String brandId, boolean emptyNotification) {
        getMessagesListener().onExConversationHandled(emptyNotification);
    }

	/**
	 * RUN ONLY ON DB THREAD!!
	 *
	 * @param dialogId
	 * @param sequence
	 */
	private long getTimestampMessage(String dialogId, int sequence) {
		long timestamp = 0;
		Cursor cursor = getDB().rawQuery("SELECT " + MessagesTable.KEY_TIMESTAMP + " FROM " + MessagesTable.MESSAGES_TABLE + " WHERE " + MessagesTable.KEY_DIALOG_ID
				+ " =? AND " + MessagesTable.KEY_SERVER_SEQUENCE + " =? ", dialogId, sequence);

		if (cursor != null) {
			try {
				if (cursor.moveToFirst()) {
					timestamp = cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_TIMESTAMP));
				}
			} finally {
				cursor.close();
			}
		}
		return timestamp;
	}

	@Override
	public void shutDown() {
		mMessageTimeoutQueue.removeAll();
	}

	@Override
	public void clear() {

	}
}