package com.liveperson.messaging.model;

import android.text.TextUtils;

import com.liveperson.api.response.model.UserProfile;
import com.liveperson.api.response.types.TTRType;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.database.DataBaseCommand;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.model.LPWelcomeMessage;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.network.http.AgentProfileRequest;

/**
 * Created by shiranr on 28/01/2016.
 */
public class ConversationUtils {

    private static final String TAG = ConversationUtils.class.getSimpleName();
    protected final Messaging mController;

    public ConversationUtils(Messaging controller) {
        mController = controller;
    }

    public void addFirstMessage(final String targetId, final String message) {

        mController.amsMessages.isFirstMessageExists().setPostQueryOnBackground(new DataBaseCommand.QueryCallback<Boolean>() {
            @Override
            public void onResult(Boolean isFirstMessageExists) {

                if (isFirstMessageExists){
                    return;
                }
                mController.amsMessages.getTimeOfFirstMessage().setPostQueryOnBackground(new DataBaseCommand.QueryCallback<Long>() {
                    @Override
                    public void onResult(Long messageTimestamp) {

                        //Adding first message before our oldest message.
                        messageTimestamp -= 1 ;
                        String conversationId = AmsConversations.KEY_WELCOME_CONVERSATION_ID;
                        String dialogId = AmsDialogs.KEY_WELCOME_DIALOG_ID;
                        String originatorId = "";
                        String eventId = messageTimestamp + "_" + AmsMessages.WELCOME_MSG_SEQUENCE_NUMBER;

                        mController.amsConversations.createDummyConversationForFirstMessage(targetId, targetId, conversationId, AmsMessages.WELCOME_MSG_SEQUENCE_NUMBER, messageTimestamp);
                        mController.amsDialogs.createDummyDialogForFirstMessage(targetId, targetId, conversationId, dialogId, AmsMessages.WELCOME_MSG_SEQUENCE_NUMBER, messageTimestamp);

                        MessagingChatMessage systemResolved = new MessagingChatMessage(originatorId, message, messageTimestamp, dialogId, eventId,
                                MessagingChatMessage.MessageType.BRAND, MessagingChatMessage.MessageState.RECEIVED, EncryptionVersion.NONE);
                        systemResolved.setServerSequence(AmsMessages.WELCOME_MSG_SEQUENCE_NUMBER);
                        mController.amsMessages.addMessage(targetId, systemResolved, true).execute();
                    }
                }).execute();
            }
        }).execute();
    }

    public void addWelcomeMessage(final String targetId, final String message) {
        long messageTimestamp = System.currentTimeMillis() ;
        String conversationId = AmsConversations.KEY_WELCOME_CONVERSATION_ID;
        String dialogId = AmsDialogs.KEY_WELCOME_DIALOG_ID;
        String originatorId = null;
        String eventId = messageTimestamp + "_" + AmsMessages.WELCOME_MSG_SEQUENCE_NUMBER;

        mController.amsConversations.createDummyConversationForFirstMessage(targetId, targetId, conversationId, AmsMessages.WELCOME_MSG_SEQUENCE_NUMBER, messageTimestamp);
        mController.amsDialogs.createDummyDialogForFirstMessage(targetId, targetId, conversationId, dialogId, AmsMessages.WELCOME_MSG_SEQUENCE_NUMBER, messageTimestamp);

        MessagingChatMessage welcomeMessage = new MessagingChatMessage(originatorId, message, messageTimestamp, dialogId, eventId,
                MessagingChatMessage.MessageType.BRAND, MessagingChatMessage.MessageState.RECEIVED, EncryptionVersion.NONE);
        welcomeMessage.setServerSequence(AmsMessages.WELCOME_MSG_SEQUENCE_NUMBER);
        mController.amsMessages.addMessage(targetId, welcomeMessage, true).execute();
    }

    /**
     * If show welcome message for every new conversation, we need to update welcome everytime enter conversation screen when the conversation is not started.
     * If show welcome message for only first conversation, we remove welcome message if there is already an welcome message and it's not first message or last message.
     */
    public void updateWelcomeMessage(String brandId, LPWelcomeMessage message) {
        mController.amsMessages.isLastMessageWelcomeMessage().setPostQueryOnBackground(
                new DataBaseCommand.QueryCallback<Boolean>() {
                    @Override
                    public void onResult(Boolean isLastMessageWelcomeMessage) {
                        if (isLastMessageWelcomeMessage) {
                            if (message.getMessageFrequency() == LPWelcomeMessage.MessageFrequency.FIRST_TIME_CONVERSATION) {
                                mController.amsMessages.removeLastWelcomeMessage().execute();
                            } else {
                                long messageTimestamp = System.currentTimeMillis();
                                String conversationId = AmsConversations.KEY_WELCOME_CONVERSATION_ID;
                                String dialogId = AmsDialogs.KEY_WELCOME_DIALOG_ID;
                                String originatorId = null;
                                String eventId = messageTimestamp + "_" + AmsMessages.WELCOME_MSG_SEQUENCE_NUMBER;

                                mController.amsConversations.createDummyConversationForFirstMessage(brandId, brandId, conversationId, AmsMessages.WELCOME_MSG_SEQUENCE_NUMBER, messageTimestamp);
                                mController.amsDialogs.createDummyDialogForFirstMessage(brandId, brandId, conversationId, dialogId, AmsMessages.WELCOME_MSG_SEQUENCE_NUMBER, messageTimestamp);

                                MessagingChatMessage welcomeMessage = new MessagingChatMessage(originatorId, message.getWelcomeMessage(), messageTimestamp, dialogId, eventId,
                                        MessagingChatMessage.MessageType.BRAND, MessagingChatMessage.MessageState.RECEIVED, EncryptionVersion.NONE);
                                welcomeMessage.setServerSequence(AmsMessages.WELCOME_MSG_SEQUENCE_NUMBER);
                                mController.amsMessages.updateLastWelcomeMessage(welcomeMessage).execute();
                            }
                        }
                    }
                }
        ).execute();
    }

    /**
     * @param participants
     * @param brandId
     * @param conversationId
     */
    public void updateParticipants(final String targetId, final String brandId, String[] participants, final UserProfile.UserType userType,
                                   final String conversationId, final boolean updateUI, final boolean forceUpdate) {
        updateParticipants(targetId, brandId, participants, userType, conversationId, updateUI, forceUpdate, null);
    }

    /**
     * @param conversationId
     * @param brandId
     * @param participants
     */
    public void updateParticipants(final String targetId, final String brandId, String[] participants, final UserProfile.UserType userType,
                                   final String conversationId, final boolean updateUI, final boolean forceUpdate, final ICallback<MessagingUserProfile, Exception> callback) {
        for (final String userId : participants) {
            if (!TextUtils.isEmpty(userId)) {
                //foreach participant - if does not exist in db - send getUserProfile request
                mController.amsUsers.getUserById(userId).setPostQueryOnBackground(new DataBaseCommand.QueryCallback<MessagingUserProfile>() {
                    @Override
                    public void onResult(MessagingUserProfile userProfile) {
                        if (userProfile == null || userProfile.isEmptyUser() || forceUpdate) {
                            userProfile = new MessagingUserProfile("", "", userType);
                            userProfile.setOriginatorID(userId);
                            mController.amsUsers.updateUserProfile(userProfile);

                            LPMobileLog.i(TAG, "First time bringing information for another participant that joined conversation " + conversationId);
                            sendUpdateUserRequest(targetId, brandId, userId, conversationId, updateUI, callback);
                        }else{
                            if(callback != null){
                                callback.onSuccess(userProfile);
                            }
                        }

                    }
                }).execute();

            }
        }
    }

    /**
     * @param targetId
     * @param brandId
     * @param userId
     * @param conversationId
     * @param shouldUpdateUi
     * @param callback
     */
    private void sendUpdateUserRequest(String targetId, String brandId, String userId, String conversationId, boolean shouldUpdateUi, ICallback<MessagingUserProfile, Exception> callback) {
        if (!TextUtils.isEmpty(userId)) {
            // There is an assigned agent, get his details and update
            new AgentProfileRequest(mController, targetId, userId, conversationId, shouldUpdateUi).setCallback(callback).execute();
        } else {
            if (TextUtils.isEmpty(conversationId)){
                LPMobileLog.d(TAG, "sendUpdateUserRequest: no dialog id");
                return;
            }

            // There is no assigned agent, get the conversation from the DB and update the callback
            mController.amsConversations.queryConversationById(conversationId)
                    .setPostQueryOnBackground(new DataBaseCommand.QueryCallback<Conversation>() {
                        @Override
                        public void onResult(Conversation data) {
                            LPMobileLog.d(TAG, "onResult: Calling agent details callback with null agent");
                            mController.onAgentDetailsChanged(null, data.isConversationOpen());
                        }
                    }).execute();
        }

    }

    public void updateTTR(TTRType type, long effectiveTTR, long delayTTR, String targetId) {
        LPMobileLog.d(TAG, "update TTR type to - " + type + ". EffectiveTTR = " + effectiveTTR);
        long clockDiff = mController.mConnectionController.getClockDiff(targetId);
        mController.amsConversations.updateTTRType(targetId, type, effectiveTTR, delayTTR, clockDiff);

    }

	public long calculateEffectiveTTR(final String targetId, final TTRType type, final long ttrValue, final long manualTTR, final long delayTTR) {
		long clockDiff = mController.mConnectionController.getClockDiff(targetId);

		return mController.amsConversations.calculateEffectiveTTR(targetId, type, ttrValue, manualTTR, delayTTR, clockDiff);
	}

	public static void showTTR(final Messaging controller, String targetId) {
    	controller.amsConversations.showTTR(targetId);
	}
}
