package com.liveperson.messaging.model;

import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.liveperson.api.response.model.UserProfile;
import com.liveperson.api.response.types.CloseReason;
import com.liveperson.api.response.types.DialogType;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.Infra;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.database.DataBaseCommand;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.messaging.R;
import com.liveperson.infra.utils.DateUtils;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.Utils.ClockUtils;
import com.liveperson.messaging.network.http.AgentProfileRequest;

import java.text.DateFormat;
import java.util.Date;

public class DialogUtils {

    private static final String TAG = DialogUtils.class.getSimpleName();
    protected final Messaging mController;
    private final DateFormat mFormatter;

    public DialogUtils(Messaging controller) {
        mController = controller;
        String customFormat = Configuration.getString(R.string.lp_date_time_format);
        mFormatter = DateUtils.getDateFormat(customFormat, DateFormat.SHORT, DateFormat.SHORT);
    }

    /**
     * Add resolve message and notify to the UI that the conversation has resolved
     *
     * @param resolvedBy
     */
    public void addClosedDialogDivider(final String brandId, final Dialog dialog, String assignedAgentServerId, CloseReason resolvedBy, final boolean updateUI, @Nullable final ICallback<Void, Exception> callback) {
        final String originatorId;

        if (resolvedBy == CloseReason.CONSUMER) {
            originatorId = mController.getOriginatorId(brandId);
        } else if (resolvedBy == CloseReason.TIMEOUT || resolvedBy == CloseReason.SYSTEM) {
            originatorId = null;
        } else {
            originatorId = assignedAgentServerId;
        }

        addResolvedDivider(brandId, dialog, originatorId, resolvedBy, updateUI, callback);
    }

    protected String getResolveTextMessage(String message, long timestamp, String agentName) {
        String time = mFormatter.format(new Date(timestamp));

        if (!TextUtils.isEmpty(agentName)) {
            return String.format(message, agentName, time);
        } else {
            return String.format(message, time);
        }
    }

    /**
     * Adds a divider between the messages, this methods usually helps to add a simple divider without a timestamp between dialogs.
     *
     * @param targetId
     * @param dialog
     * @param originatorId
     * @param resolvedBy
     * @param updateUI
     * @param callback
     */
    private void addResolvedDivider(final String targetId, final Dialog dialog, final String originatorId, CloseReason resolvedBy, final boolean updateUI, @Nullable final ICallback<Void, Exception> callback) {
        final long timestamp = ClockUtils.addDiffToTimestamp(dialog.getEndTimestamp());
        if (dialog.getDialogType() == DialogType.MAIN) {
            if (resolvedBy == CloseReason.CONSUMER) {
                String resolvedByName = Configuration.getString(R.string.lp_conversation_ended_by_you);
                String resolvedMsg = getResolveTextMessage(resolvedByName, timestamp, null);
                addResolveMessage(targetId, dialog.getDialogId(), timestamp, resolvedMsg, originatorId, updateUI, MessagingChatMessage.MessageType.SYSTEM_RESOLVED, callback);
            } else if (resolvedBy == CloseReason.TIMEOUT || resolvedBy == CloseReason.SYSTEM) {
                LPMobileLog.d(TAG, "This conversation was Auto closed - Don't add Resolved message");
                if (callback != null){
                    callback.onError(new Exception("This conversation timed out or Auto closed"));
                }
            } else {
                //Need agent name from DB to show fragment
                mController.amsUsers.getUserById(originatorId)
                        .setPostQueryOnUI(new DataBaseCommand.QueryCallback<MessagingUserProfile>() {
                            @Override
                            public void onResult(MessagingUserProfile data) {
                                String resolvedMsg;
                                String agentNickName = null;
                                if (data != null) {
                                    agentNickName = data.getNickname();
                                }
                                if (!TextUtils.isEmpty(agentNickName)) {
                                    resolvedMsg = Configuration.getString(R.string.lp_conversation_ended_by_agent_with_name);
                                } else {
                                    resolvedMsg = Configuration.getString(R.string.lp_conversation_ended_by_agent_no_name);
                                }
                                resolvedMsg = getResolveTextMessage(resolvedMsg, timestamp, agentNickName);

                                addResolveMessage(targetId, dialog.getDialogId(), timestamp, resolvedMsg, originatorId, updateUI, MessagingChatMessage.MessageType.SYSTEM_RESOLVED, callback);
                            }
                        }).execute();
            }
        } else {
            addResolveMessage(targetId, dialog.getDialogId(), timestamp, "dialog closed", originatorId, updateUI, MessagingChatMessage.MessageType.SYSTEM_DIALOG_RESOLVED, callback);
        }
    }

    private void addResolveMessage(String targetId, String dialogId, long timestamp, String message, String originatorId, boolean updateUI, MessagingChatMessage.MessageType messageType, final ICallback callback) {
        LPMobileLog.d(TAG, "creating message resolve at time: " + timestamp
                + " timestamp: " + timestamp);
        String eventId = timestamp + "_" + AmsMessages.RESOLVE_MSG_SEQUENCE_NUMBER;

        MessagingChatMessage systemResolved = new MessagingChatMessage(originatorId, message, timestamp, dialogId, eventId,
                messageType, MessagingChatMessage.MessageState.RECEIVED, EncryptionVersion.NONE);
        systemResolved.setServerSequence(AmsMessages.RESOLVE_MSG_SEQUENCE_NUMBER);
        mController.amsMessages.addMessage(targetId, systemResolved, updateUI).setPostQueryOnBackground(new DataBaseCommand.QueryCallback<Long>() {
            @Override
            public void onResult(Long data) {
                if (callback != null) {
                    callback.onSuccess(null);
                }
            }
        }).execute();
    }

    /**
     * @param participants
     * @param brandId
     * @param dialogId
     */
    public void updateParticipants(final String targetId, final String brandId, String[] participants, final UserProfile.UserType userType,
                                   final String dialogId, final boolean updateUI, final boolean forceUpdate) {
        updateParticipants(targetId, brandId, participants, userType, dialogId, updateUI, forceUpdate, null);
    }

    /**
     * @param dialogId
     * @param brandId
     * @param participants
     */
    public void updateParticipants(final String targetId, final String brandId, String[] participants, final UserProfile.UserType userType,
                                   final String dialogId, final boolean updateUI, final boolean forceUpdate, final ICallback<MessagingUserProfile, Exception> callback) {
        for (final String userId : participants) {
            if (!TextUtils.isEmpty(userId)) {
                //foreach participant - if does not exist in db - send getUserProfile request
                mController.amsUsers.getUserById(userId).setPostQueryOnBackground(new DataBaseCommand.QueryCallback<MessagingUserProfile>() {
                    @Override
                    public void onResult(MessagingUserProfile userProfile) {
                        if (userProfile == null || userProfile.isEmptyUser() || forceUpdate) {
                            userProfile = new MessagingUserProfile("", "", userType);
                            userProfile.setOriginatorID(userId);
                            mController.amsUsers.updateUserProfile(userProfile);

                            LPMobileLog.i(TAG, "First time bringing information for another participant that joined dialog " + dialogId);
                            sendUpdateUserRequest(targetId, brandId, userId, dialogId, updateUI, callback);
                        }else{
                            if(callback != null){
                                callback.onSuccess(userProfile);
                            }
                        }
                    }
                }).execute();
            } else {
                LPMobileLog.e(TAG, "Missing agent ID!");
            }
        }
    }

    /**
     * @param targetId
     * @param brandId
     * @param userId
     * @param dialogId
     * @param shouldUpdateUi
     * @param callback
     */
    private void sendUpdateUserRequest(String targetId, String brandId, String userId, String dialogId, boolean shouldUpdateUi, ICallback<MessagingUserProfile, Exception> callback) {
        if (!TextUtils.isEmpty(userId)) {
            // There is an assigned agent, get his details and update
            new AgentProfileRequest(mController, targetId, userId, dialogId, shouldUpdateUi).setCallback(callback).execute();
        } else {
            if (TextUtils.isEmpty(dialogId)){
                LPMobileLog.d(TAG, "sendUpdateUserRequest: no dialog id");
                return;
            }

            // There is no assigned agent, get the conversation from the DB and update the callback
            mController.amsDialogs.queryDialogById(dialogId)
                    .setPostQueryOnBackground(new DataBaseCommand.QueryCallback<Dialog>() {
                        @Override
                        public void onResult(Dialog dialog) {
                            LPMobileLog.d(TAG, "onResult: Calling agent details callback with null agent");
                            mController.onAgentDetailsChanged(null, dialog.isOpen());
                        }
                    }).execute();
        }
    }

}
