package com.liveperson.messaging.network.socket

import android.os.Bundle
import android.util.Log
import android.webkit.URLUtil
import com.liveperson.api.response.events.MessagingEventNotification
import com.liveperson.api.response.model.Event
import com.liveperson.infra.ICallback
import com.liveperson.infra.log.LPMobileLog
import com.liveperson.infra.model.types.ChatState
import com.liveperson.infra.network.socket.BaseResponseHandler
import com.liveperson.infra.network.socket.BaseSocketRequest
import com.liveperson.infra.utils.LocalBroadcast
import com.liveperson.messaging.Messaging
import com.liveperson.messaging.model.Dialog
import org.json.JSONException
import org.json.JSONObject
import java.util.*

/**
 * Created by shiranr on 2/1/18.
 */

class MessagingEventNotificationHandler(internal var mController: Messaging) : BaseResponseHandler<MessagingEventNotification, BaseSocketRequest<*, *>>() {

    val TAG = "MessagingEvent"

    override fun getAPIResponseType(): String = "ms.MessagingEventNotification"

    /**
     * Parse the incoming event into the OnlineEventDistribution data object
     *
     * @param jsonObject
     * @return
     * @throws JSONException
     */
    @Throws(JSONException::class)
    override fun parse(jsonObject: JSONObject): MessagingEventNotification? {
        var responseBody: MessagingEventNotification? = null

        try {
            responseBody = MessagingEventNotification(jsonObject)
        } catch (e: JSONException) {
            LPMobileLog.e(TAG, e.message)
        }

        return responseBody
    }

    /**
     * Handle the incoming event. The body of the incoming event is ContentEventNotification on which we act upon.
     * This handles 3 event types:
     * AcceptStatusEvent - indicates whether the agent received or read the message the consumer sent it
     * ChatStateEvent - indicates whether the agent is typing or not
     * ContentEvent - an incoming message from the agent
     *
     *
     * Note that on AcceptStatusEvent and on ContentEvent, the server sends an echo message, so we need to ignore
     * an event that is sent with the same originator ID as ours
     *
     * @param event
     */
    override fun handle(event: MessagingEventNotification): Boolean {

        if (event.body.isEmpty()){
            return true
        }
        val dialogId = event.body[0].dialogId

        var firstNotification = mController.messagingEventSubscriptionManager.onReceivedEvent(dialogId)

        val shouldUpdateUi = mController.messagingEventSubscriptionManager.shouldUpdateUI(dialogId)

        mController.amsDialogs.saveMessagesResult(dialogId, event.body, firstNotification, shouldUpdateUi, object : ICallback<Dialog, Exception>{
            override fun onSuccess(dialog: Dialog?) {

                // Get the consumerID from the model
                val consumerId = mController.getOriginatorId(dialog!!.targetId)
                event.body.forEach {
                    LPMobileLog.d(TAG, "saving message in dialog " + dialogId)
                    val eventType = it.event.type
                    val originatorId = it.originatorId

                    when (eventType) {
                    // Event whether the agent is typing of not
                        Event.Types.ChatStateEvent -> {

                            // If this message is not from myself
                            if (originatorId != consumerId) {
                                var isTyping = false

                                when (it.event.chatState) {
                                    ChatState.COMPOSING -> isTyping = true
                                    ChatState.ACTIVE -> isTyping = false
                                }

                                // Send a broadcast on the typing status
                                val agentBundle = Bundle()
                                agentBundle.putBoolean(AGENT_TYPING_ACTION_IS_TYPING_EXTRA, isTyping)
                                agentBundle.putString(ORIGINATOR_ID_EXTRA, originatorId);
                                LocalBroadcast.sendBroadcast(AGENT_TYPING_ACTION, agentBundle)

                                mController.mEventsProxy.onAgentTyping(isTyping)

                            }
                        }
                    }
                }
            }

            override fun onError(exception: Exception?) {

            }
        })

        return true
    }


    /**
     * extractLinks - a method to extract only the url from the whole message
     * for later parse
     *
     * @param text - the original msg
     * @return String[] of all the links
     */
    fun extractLinks(text: String): Array<String> {
        val links = ArrayList<String>()
        val l = text.split("\\s+".toRegex()).dropLastWhile { it.isEmpty() }.toTypedArray()
        for (i in l.indices) {
            if (URLUtil.isValidUrl(l[i])) {
                links.add(l[i])
            }
        }
        return links.toTypedArray()
    }

   /* private fun handleResponse(mController: Messaging, notification: ContentEventNotification, targetId: String, brandId: String, conversationServerId: String) {

        val messageServerSequence = notification.sequence
        val originatorId = notification.originatorId
        val eventType = notification.event.type

        // Set the received last sequence to the current conversation
        if (messageServerSequence != -1 && eventType != null && eventType != Event.Types.ChatStateEvent) {
            //not is typing message
            mController.amsConversations.updateLastServerSequenceByConvId(conversationServerId, messageServerSequence)
        }


        // Get the consumerID from the model
        val consumerId = mController.getOriginatorId(targetId)

        when (eventType) {

        // Event on the status of the message sent by the consumer (received (ACCEPT) or read
        // This is an indication that can be received on several messages, so we aggregate the
        // sequence in a list
            Event.Types.AcceptStatusEvent -> {
                LPMobileLog.i(TAG, String.format("Got status from agent (on messages sent from the consumer)." + " State %s sequenceList length: %s", notification.event.status, Arrays.toString(notification.event.sequenceList)))

                mController.amsMessages.updateMessageReceivedState(targetId, conversationServerId, notification.event.sequenceList, notification.event.status)
            }

        // Event whether the agent is typing of not
            Event.Types.ChatStateEvent -> {

                // If this message is not from myself
                if (originatorId != consumerId) {
                    var isTyping = false

                    when (notification.event.chatState) {
                        ChatState.COMPOSING -> isTyping = true
                        ChatState.ACTIVE -> isTyping = false
                    }

                    // Send a broadcast on the typing status
                    val agentBundle = Bundle()
                    agentBundle.putBoolean(AGENT_TYPING_ACTION_IS_TYPING_EXTRA, isTyping)
                    LocalBroadcast.sendBroadcast(AGENT_TYPING_ACTION, agentBundle)

                    mController.mEventsProxy.onAgentTyping(isTyping)

                }
            }

        // Content event received from the agent
            Event.Types.ContentEvent -> {
                LPMobileLog.i(TAG, "New ContentEvent. messageServerSequence = " + messageServerSequence)

                // Get the actual message
                var publishMessage: BasePublishMessage? = null
                if (notification.event.message != null) {
                    publishMessage = notification.event.message
                }

                // If the publish message is a text and the message itself is empty we ignore this
                if (publishMessage == null || publishMessage.type == BasePublishMessage.PublishMessageType.TEXT && TextUtils.isEmpty(publishMessage.messageText)) {
                    LPMobileLog.w(TAG, "handleResponse: received a text message with no text")
                    //no text, ignore message
                    return
                }
                val clockDiff = mController.mConnectionController.getClockDiff(brandId)

                var messageType: MessagingChatMessage.MessageType
                val messageState: MessagingChatMessage.MessageState
                //this is an echo message from other connected device.

                val contentType = ContentType.fromString(notification.event.contentType)
                if (originatorId == consumerId) {

                    messageType = MessagingChatMessage.MessageType.getMessageTypeForConsumer(contentType)
                    messageState = MessagingChatMessage.MessageState.SENT
                } else {

                    // If this message is not from myself

                    // If message from Controller
                    if (notification.originatorMetadata != null && notification.originatorMetadata.mRole == Participants.ParticipantRole.CONTROLLER) {
                        messageType = MessagingChatMessage.MessageType.CONTROLLER_SYSTEM
                    } else { // Message from Agent
                        messageType = MessagingChatMessage.MessageType.getMessageTypeForAgent(contentType)
                    }
                    messageState = MessagingChatMessage.MessageState.RECEIVED
                }

                // check if we have a link than we need to set the message type to be with URL
                if (messageType == MessagingChatMessage.MessageType.CONSUMER ||
                        messageType == MessagingChatMessage.MessageType.CONSUMER_MASKED ||
                        messageType == MessagingChatMessage.MessageType.AGENT) {

                    messageType = checkIfMessageContainsURLandChangeType(messageType, publishMessage.messageText)
                }

                val messageText = publishMessage.messageText

                // If the message text starts with STRUCTURED_CONTENT_PREFIX than this is a structured content message.
                // We set the type of the message accordingly and remove the prefix from the message text
                // Note: this is a temporary implementation until the server side implements the new event for structured content
                // TODO: This is left here just for debug purposes until UMS is fully ready with structured content
                *//*
				if (publishMessage.getMessageText().startsWith(AmsMessages.STRUCTURED_CONTENT_PREFIX)) {
					messageType = MessagingChatMessage.MessageType.AGENT_STRUCTURED_CONTENT;
					messageText = messageText.substring(AmsMessages.STRUCTURED_CONTENT_PREFIX.length(), messageText.length());
				}
*//*

                val message = createMessage(notification,
                        conversationServerId,
                        messageServerSequence,
                        originatorId,
                        messageText,
                        clockDiff,
                        messageType,
                        contentType,
                        messageState,
                        notification.eventId)
                LPMobileLog.i(TAG, "Received new message. message = " + message)


                // Add the message to the current conversation
                addMessageToDB(message, targetId, originatorId, conversationServerId, notification, clockDiff, publishMessage)
               *//* if (originatorId != consumerId) {
                    // Send an ack to the server that we received the agent message
                    sendStatusUpdate(targetId, brandId, conversationServerId, messageServerSequence)
                }*//*
            }

            Event.Types.RichContentEvent -> {

                // Get the actual message
                var publishMessage: BasePublishMessage? = null
                if (notification.event.message != null) {
                    publishMessage = notification.event.message
                }

                // If the publish message is a text and the message itself is empty we ignore this
                if (publishMessage == null || publishMessage.type == BasePublishMessage.PublishMessageType.STRUCTURED_CONTENT && TextUtils.isEmpty(publishMessage.messageText)) {
                    LPMobileLog.w(TAG, "handleResponse: received a text message with no text")
                    //no text, ignore message
                    return
                }
                val clockDiff = mController.mConnectionController.getClockDiff(brandId)

                val messageType: MessagingChatMessage.MessageType
                val messageState: MessagingChatMessage.MessageState
                val contentType = ContentType.text_structured_content

                messageType = MessagingChatMessage.MessageType.AGENT_STRUCTURED_CONTENT
                if (originatorId == consumerId) {

                    messageState = MessagingChatMessage.MessageState.SENT
                } else {

                    // If this message is not from myself
                    messageState = MessagingChatMessage.MessageState.RECEIVED
                }


                val messageText = publishMessage.messageText

                val message = createMessage(notification,
                        conversationServerId,
                        messageServerSequence,
                        originatorId,
                        messageText,
                        clockDiff,
                        messageType,
                        contentType,
                        messageState,
                        notification.eventId)
                LPMobileLog.i(TAG, "Received new message. message = " + message)


                // Add the message to the current conversation
                addMessageToDB(message, targetId, originatorId, conversationServerId, notification, clockDiff, publishMessage)
                if (originatorId != consumerId) {
                    // Send an ack to the server that we received the agent message
                    sendStatusUpdate(targetId, brandId, conversationServerId, messageServerSequence)
                }
            }
        }

    }
*/

   /* private fun checkIfMessageContainsURLandChangeType(type: MessagingChatMessage.MessageType, msg: String): MessagingChatMessage.MessageType {
        val urls = extractLinks(msg)
        if (urls.size > 0) {
            if (type == MessagingChatMessage.MessageType.CONSUMER) {
                return MessagingChatMessage.MessageType.CONSUMER_URL
            }
            if (type == MessagingChatMessage.MessageType.CONSUMER_MASKED) {
                return MessagingChatMessage.MessageType.CONSUMER_URL_MASKED
            }
            if (type == MessagingChatMessage.MessageType.AGENT) {
                return MessagingChatMessage.MessageType.AGENT_URL
            }
        }
        return type
    }

    private fun createMessage(notification: ContentEventNotification, conversationServerId: String,
                              messageServerSequence: Int, originatorId: String, messageText: String,
                              clockDiff: Long, messageType: MessagingChatMessage.MessageType,
                              contentType: ContentType, messageState: MessagingChatMessage.MessageState, eventId: String?): MessagingChatMessage {
        return MessagingChatMessage(originatorId, messageText,
                notification.serverTimestamp + clockDiff,
                conversationServerId,
                eventId,
                messageType,
                messageState,
                messageServerSequence,
                contentType.text,
                EncryptionVersion.NONE)
    }

    private fun sendStatusUpdate(targetId: String, brandId: String, conversationServerId: String, messageServerSequence: Int) {
        DeliveryStatusUpdateCommand(mController.mAccountsController.getConnectionUrl(brandId), targetId, conversationServerId, messageServerSequence).execute()
    }

    private fun addMessageToDB(message: MessagingChatMessage, targetId: String, originatorId: String,
                               conversationServerId: String, notification: ContentEventNotification, clockDiff: Long, basePublishMessage: BasePublishMessage) {
        // Add the message to the current conversation
        val conversation = mController.amsConversations.getConversation(targetId)
        val isMsgFromAssignedAgent = TextUtils.equals(originatorId, conversation.assignedAgentId)
        LPMobileLog.i(TAG, "Received new message from " + (if (isMsgFromAssignedAgent) "assigned agent" else " someone else / myself in a different device")
                + ", conversation: " + conversationServerId + ", event id : " + message.eventId + ", sequence: " + message.serverSequence +
                " time: " + (notification.serverTimestamp + clockDiff))

        mController.amsMessages.addMessage(targetId, message, true)
                .setPostQueryOnBackground { messageRowId ->
                    if (messageRowId == DBUtilities.ROW_UPDATED) {
                        LPMobileLog.d(TAG, "onResult: message was updated on DB (and not inserted). No need to add the file to DB")
                    } else if (basePublishMessage.type == BasePublishMessage.PublishMessageType.FILE) {
                        val tag = "OnlineEvent"
                        mController.amsMessages.addFileFromPublishMessageToDB(messageRowId!!, tag, basePublishMessage as FilePublishMessage, targetId, true)

                    } else if (basePublishMessage.type == BasePublishMessage.PublishMessageType.FORM_INVITATION) {
                        val formPublishMessage = basePublishMessage as FormPublishMessage
                        mController.amsMessages.mFormsManager.addForm(formPublishMessage.invitationId,
                                Form(
                                        conversationServerId,
                                        formPublishMessage.invitationId,
                                        formPublishMessage.formId,
                                        formPublishMessage.formTitle,
                                        mController.mAccountsController.getTokenizerUrl(mController.amsConversations.getConversationFromServerIdMap(conversationServerId)!!.brandId),
                                        mController.amsConversations.getConversationFromServerIdMap(conversationServerId)!!.brandId,
                                        message.serverSequence,
                                        message.eventId
                                ))
                    } else if (basePublishMessage.type == BasePublishMessage.PublishMessageType.FORM_SUBMISSION) {
                        val formSubmissionPublishMessage = basePublishMessage as FormSubmissionPublishMessage
                        val form = mController.amsMessages.mFormsManager.getForm(formSubmissionPublishMessage.invitationId)
                        if (form != null) {
                            mController.amsMessages.mFormsManager.updateForm(formSubmissionPublishMessage.invitationId, formSubmissionPublishMessage.getmSubmissionId())
                            mController.updateMessage(form.invitationId, form.conversationId, MessagingChatMessage.MessageType.AGENT_FORM, MessagingChatMessage.MessageState.SUBMITTED)
                        }
                    } else {
                        LPMobileLog.d(TAG, "**************** addMessageToDB ************")

                        // TODO: 9/7/16 what should we do here?
                    }// in case this is a message with form submission - we need to update the original form and add consumer message
                    // in case this is a message with form invitation we need to create new form and add to the manager
                }.execute()

    }
*/
    companion object {

        private val TAG = MessagingEventNotificationHandler::class.java.simpleName

        val AGENT_TYPING_ACTION = "agent_typing"
        val AGENT_TYPING_ACTION_IS_TYPING_EXTRA = "is_typing"
        val ORIGINATOR_ID_EXTRA = "originator_id"
    }
}
