package com.liveperson.messaging.controller;

import android.net.Uri;
import android.os.Build;
import android.text.TextUtils;

import com.liveperson.infra.Clearable;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.messaging.R;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.TimeZone;

/**
 * Created by shiranr on 30/06/2016.
 */
public class ClientProperties implements Clearable{
    private static final String TAG = ClientProperties.class.getSimpleName();

    private static final String APP_ID_PREFERENCE_KEY = "APP_ID_PREFERENCE_KEY";
    private static final String SDK_VERSION_PREFERENCE_KEY = "SDK_VERSION_PREFERENCE_KEY";

    private static final String APP_ID = "appId";
    private static final String DEVICE_FAMILY = "deviceFamily";
    private static final String OS_NAME = "os";
    private static final String IP_ADDRESS = "ipAddress";
    private static final String OS_VERSION = "osVersion";
    private static final String INTEGRATION = "integration";
    private static final String INTEGRATION_VERSION = "integrationVersion";
    private static final String BROWSER = "browser";
    private static final String BROWSER_VERSION = "browserVersion";
    private static final String TIME_ZONE = "timeZone";
    private static final String DEVICE_MANUFACTOR = "deviceManufactor"; // TODO the correct spelling here is Manufacturer - can we change this safely?
    private static final String DEVICE_MODEL = "deviceModel";
    private static final String APP_VERSION = "appVersion";
    private static final String FEATURES = "features";
    private static final String TYPE = "type";
    private static final String TYPE_VALUE = ".ClientProperties";


    private String mAppId;
    private String mSdkVersion;
    private final String ipAddress;

	private JSONObject mJson;

    public ClientProperties(String appId, String appVersion) {
            String _ipAddress;
            try {
                _ipAddress = InetAddress.getLocalHost().getHostAddress();
            } catch (UnknownHostException e) {
                _ipAddress = "unknown";
                LPMobileLog.e(TAG, "Failed to get ip address, unknown host exception: " + e);
            }

            ipAddress = _ipAddress;
            saveAppId(appId);
            saveSDKVersion(appVersion);
            createJson();

        LPMobileLog.w(TAG, "## DEVICE_FAMILY: " + DeviceFamily.MOBILE.name());
        LPMobileLog.w(TAG, "## OS_NAME: " + OS.ANDROID.name());
        LPMobileLog.w(TAG, "## OS_VERSION: " + String.valueOf(Build.VERSION.SDK_INT));
        LPMobileLog.w(TAG, "## INTEGRATION: " + Integration.MOBILE_SDK.name());
        LPMobileLog.w(TAG, "## OS_VERSION: " + String.valueOf(Build.VERSION.CODENAME));
    }
    public ClientProperties() {
            this(null, null);
        }
    private void createJson() {
        mJson = new JSONObject();
        try {
            mJson.put(APP_ID, mAppId);
            mJson.put(DEVICE_FAMILY,  DeviceFamily.MOBILE.name());
            mJson.put(OS_NAME, OS.ANDROID.name());
            mJson.put(OS_VERSION, String.valueOf(Build.VERSION.SDK_INT));
            mJson.put(INTEGRATION, Integration.MOBILE_SDK.name());
            mJson.put(INTEGRATION_VERSION, mSdkVersion);
            mJson.put(TYPE, TYPE_VALUE);

            // Build the features array
			JSONArray featuresArray = new JSONArray();
            featuresArray.put(Features.PHOTO_SHARING.name())
                    .put(Features.CO_APP.name())
                    .put(Features.RICH_CONTENT.name())
                    .put(Features.SECURE_FORMS.name())
                    .put(Features.AUTO_MESSAGES.name())
                    .put(Features.QUICK_REPLIES.name())
                    .put(Features.MULTI_DIALOG.name());

            if (Configuration.getBoolean(R.bool.enable_file_Sharing)) {
                featuresArray.put(Features.FILE_SHARING.name());
            }

			mJson.put(FEATURES, featuresArray);

            mJson.put(TIME_ZONE, TimeZone.getDefault().getID());

        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    public String getJsonString() {
        return mJson.toString();
    }

	public void addToUriBuilder(Uri.Builder builder) {
        builder.appendQueryParameter(APP_ID, mAppId)
                .appendQueryParameter(DEVICE_FAMILY,  DeviceFamily.MOBILE.name())
                .appendQueryParameter(OS_NAME, OS.ANDROID.name())
                .appendQueryParameter(IP_ADDRESS, ipAddress)
                .appendQueryParameter(OS_VERSION, String.valueOf(Build.VERSION.SDK_INT))
                .appendQueryParameter(INTEGRATION, Integration.MOBILE_SDK.name())
                .appendQueryParameter(INTEGRATION_VERSION, mSdkVersion)
                .appendQueryParameter(FEATURES, Features.PHOTO_SHARING.name() + "," +
                        Features.CO_APP.name() + "," +
                        Features.RICH_CONTENT.name() + "," +
                        Features.SECURE_FORMS.name() + "," +
                        Features.AUTO_MESSAGES.name() + "," +
                        Features.QUICK_REPLIES.name() + "," +
						Features.MULTI_DIALOG.name() )


                //.appendQueryParameter(APP_VERSION, BuildConfig.VERSION_NAME)
//                .appendQueryParameter(BROWSER, Browser.OTHER.name()) // No need to send from mobile
//                .appendQueryParameter(BROWSER_VERSION, "") // No need to send from mobile
                //.appendQueryParameter(DEVICE_MANUFACTOR, Build.MANUFACTURER)
                //.appendQueryParameter(DEVICE_MODEL, Build.MODEL)
                .appendQueryParameter(TIME_ZONE, TimeZone.getDefault().getID());
    }

    public void saveAppId(String newAppId) {
        if (TextUtils.isEmpty(newAppId)){
			mAppId = PreferenceManager.getInstance().getStringValue(APP_ID_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, "");
        }else{
			mAppId = newAppId;
            PreferenceManager.getInstance().setStringValue(APP_ID_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, mAppId);
        }
    }

    public void saveSDKVersion(String newSdkVersion) {
        if (TextUtils.isEmpty(newSdkVersion)){
			mSdkVersion = PreferenceManager.getInstance().getStringValue(SDK_VERSION_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, "");
        }else{
			mSdkVersion = newSdkVersion;
            PreferenceManager.getInstance().setStringValue(SDK_VERSION_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, mSdkVersion);
        }
    }

	@Override
	public void clear() {
		PreferenceManager.getInstance().remove(APP_ID_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES);
		PreferenceManager.getInstance().remove(SDK_VERSION_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES);
	}

	/* https://lpgithub.dev.lprnd.net/Architecture/api-ums/blob/314b3b72eaf0c276a35118414c79ad16b4ad7c6a/3.0.0.9/scheme.json
	  "appId" : "appIdExample",
	  "ipAddress" : "127.321.22.1",
	  "deviceFamily" : "MOBILE_PHONE",
	  "os" : "ANDROID",
	  "osVersion" : "18",
	  "integration" : "MOBILE_SDK",
	  "integrationVersion" : "1.2",
	  "browser" : "OTHER",
	  "browserVersion" : "",
	  "timeZone" : "Europe/Athens"
	  */
    private enum DeviceFamily {
        MOBILE,
        TABLET,
        PERSONAL_COMPUTER,
        PDA,
        OTHER,
        UNKNOWN
    }

    private enum OS {
        WINDOWS,
        LINUX,
        OSX,
        ANDROID,
        IOS,
        OTHER
    }

    private enum Integration {
        /** In case of connecting via regular browser */
        WEB_SDK,

        /** In case of connecting via native application */
        MOBILE_SDK,

        /** In case of connecting via singe brand connection (GW) */
        BRAND_SDK
    }

    private enum Browser {
        CHROME,
        FIREFOX,
        IE,
        OPERA,
        SAFARI,
        OTHER
    }

	public enum Features {
		CO_BROWSE,
		CO_APP,
		PHOTO_SHARING,
		SECURE_FORMS,
		RICH_CONTENT,
        AUTO_MESSAGES,
		QUICK_REPLIES,
		MULTI_DIALOG,
        FILE_SHARING
	}
}
