package com.liveperson.messaging.model;

import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.support.annotation.NonNull;

import com.liveperson.infra.ForegroundService;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.Infra;
import com.liveperson.infra.InternetConnectionService;
import com.liveperson.infra.LocalBroadcastReceiver;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.log.logreporter.loggos.Loggos;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.network.socket.SocketHandler;
import com.liveperson.infra.network.socket.SocketManager;
import com.liveperson.infra.network.socket.SocketState;
import com.liveperson.infra.network.socket.state.SocketStateListener;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDownAsync;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDownCompletionListener;
import com.liveperson.infra.statemachine.StateMachineExecutor;
import com.liveperson.infra.utils.LocalBroadcast;
import com.liveperson.messaging.IMessaging;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.TaskType;
import com.liveperson.messaging.commands.InitAmsSessionCommand;
import com.liveperson.messaging.commands.tasks.BaseAmsAccountConnectionCallback;
import com.liveperson.messaging.controller.AccountsController;
import com.liveperson.messaging.controller.ConnectionsController;
import com.liveperson.messaging.controller.connection.ConnectionStateMachine;
import com.liveperson.messaging.controller.connection.IConnectionListener;
import com.liveperson.messaging.controller.connection.InternetInformationProvider;
import com.liveperson.messaging.controller.connection.MessagingStateMachineInterface;

/**
 * Connection model
 */
public class AmsConnection implements ShutDownAsync {

    protected static final String TAG = AmsConnection.class.getSimpleName();

    private static final int IDP_TOKEN_EXPIRED_UNAUTHENTICATED_FLOW = 4407; //Unauthenticated
    private static final int IDP_TOKEN_EXPIRED_AUTHENTICATED_FLOW = 4401; //Authenticated
    public static final String BROADCAST_CONNECTING_TO_SERVER_ERROR = "BROADCAST_CONNECTING_TO_SERVER_ERROR";
    public static final String BROADCAST_START_CONNECTING = "BROADCAST_START_CONNECTING";
    public static final String BROADCAST_SOCKET_OPEN_ACTION = "BROADCAST_SOCKET_OPEN_ACTION";
    public static final String BROADCAST_KEY_SOCKET_READY_ACTION = "BROADCAST_KEY_SOCKET_READY_ACTION";
    public static final String BROADCAST_KEY_SOCKET_READY_EXTRA = "BROADCAST_KEY_SOCKET_READY_EXTRA";
    public static final String BROADCAST_KEY_BRAND_ID = "BROADCAST_KEY_BRAND_ID";
    public static final String BROADCAST_AMS_CONNECTION_UPDATE_ACTION = "BROADCAST_AMS_CONNECTION_UPDATE_ACTION";
    public static final String BROADCAST_AMS_CONNECTION_UPDATE_EXTRA = "BROADCAST_AMS_CONNECTION_UPDATE_EXTRA";

    protected String mBrandId;

    protected long mClockDiff;
    protected long mLastUpdateTime;
    private boolean firstNotificationAfterSubscribe = true;
    protected PreferenceManager mPreferenceManager;
    protected AmsSocketState mSocketState;
    protected final Messaging mController;

    protected ConnectionStateMachine mStateMachine;
    private LocalBroadcastReceiver mCertificateReceiver = null;


    /**
     * Keep alive callback - responsible to keep the socket alive by sending GetClock requests.
     */
    private boolean mIsUpdated = false;
    private boolean mIsAgentDetailsUpdated = false;
    private boolean socketReady = false;

    public AmsConnection(Messaging messagingController, String brandId) {
        mController = messagingController;
        mPreferenceManager = PreferenceManager.getInstance();
        mLastUpdateTime = mPreferenceManager.getLongValue(ConnectionsController.KEY_PREF_LAST_UPDATE_TIME, brandId, 0);
        mBrandId = brandId;
        mSocketState = new AmsSocketState();

        mCertificateReceiver = new LocalBroadcastReceiver.Builder()
                .addAction(Loggos.CERTIFICATE_ERROR_ACTION)
                .build(new LocalBroadcastReceiver.IOnReceive() {
                    @Override
                    public void onBroadcastReceived(Context context, Intent intent) {
                        mController.mEventsProxy.onError(TaskType.INVALID_CERTIFICATE, "Certificate Error");
                    }
                });

        initConnectionStateMachine();
    }

    private void initConnectionStateMachine() {
        IConnectionListener connectionListener = createConnectionListener();
        InternetInformationProvider internetInformationProvider = createInternetInformationProvider();
        MessagingStateMachineInterface controller = createMessagingStateMachineInterface();

        mStateMachine = new ConnectionStateMachine(controller, internetInformationProvider, ForegroundService.getInstance(), mBrandId, connectionListener);
        mStateMachine.setStateMachineExecutor(new StateMachineExecutor(mStateMachine.getTag(), mStateMachine));
    }

    @NonNull
    private MessagingStateMachineInterface createMessagingStateMachineInterface() {
        return new MessagingStateMachineInterface() {

            @Override
            public AccountsController getAccountsController() {
                return mController.mAccountsController;
            }

            @Override
            public ConnectionsController getConnectionController() {
                return mController.mConnectionController;
            }

            @Override
            public AmsMessages getAmsMessages() {
                return mController.amsMessages;
            }

            @Override
            public AmsConversations getAmsConversations() {
                return mController.amsConversations;
            }

            @Override
            public AmsDialogs getAmsDialogs() {
                return mController.amsDialogs;
            }

            @Override
            public AmsUsers getAmsUsers() {
                return mController.amsUsers;
            }

            @Override
            public IMessaging getMessagingController() {
                return mController;
            }

        };
    }

    @NonNull
    private InternetInformationProvider createInternetInformationProvider() {
        return new InternetInformationProvider() {
            @Override
            public boolean isNetworkAvailable() {
                return InternetConnectionService.isNetworkAvailable();
            }

            @Override
            public void unregisterToNetworkChanges() {
                Infra.instance.unregisterToNetworkChanges();

            }

            @Override
            public void registerToNetworkChanges() {
                Infra.instance.registerToNetworkChanges();
            }
        };
    }

    @NonNull
    private IConnectionListener createConnectionListener() {
        return new IConnectionListener() {
            @Override
            public void notifyError(final TaskType type, final String message) {
                mController.mEventsProxy.onError(type, message);
                LocalBroadcast.sendBroadcast(BROADCAST_CONNECTING_TO_SERVER_ERROR);

            }

            @Override
            public void notifyStartConnecting() {
                LocalBroadcast.sendBroadcast(BROADCAST_START_CONNECTING);
                AmsConnectionAnalytics.startConnecting();
                LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, "Start connecting for brand: " + mBrandId);
            }

            @Override
            public void notifyStartDisconnecting() {
                onDisconnected();
                AmsConnectionAnalytics.startDisconnecting();
            }

            @Override
            public void notifyDisconnected() {
                onDisconnected();
                //only when completely disconnected we can clear all the waiting messages
                mController.amsMessages.mMessageTimeoutQueue.removeAll();
            }

            @Override
            public void notifyConnected() {
                //state machine connected - means socket open.
                LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, "On connected for brand: " + mBrandId);
                AmsConnectionAnalytics.connected();
            }

            @Override
            public void notifyTokenExpired() {
                notifyHostAppTokenExpired();
            }

            @Override
            public void notifyUserExpired() {
                notifyHostAppUserExpired();
            }
        };
    }

    public AmsSocketState registerSocket() {
        String connectionUrl = mController.mAccountsController.getConnectionUrl(mBrandId);
        String connectionUrlForLogs = mController.mAccountsController.getConnectionUrlForLogs(mBrandId);
        LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, "Register socket for brand " + mBrandId + ", connectionUrl = " + connectionUrlForLogs);
        SocketManager.getInstance().registerToSocketState(connectionUrl, mSocketState);
        return mSocketState;
    }


    public void unregisterSocketListener() {
        String connectionUrl = mController.mAccountsController.getConnectionUrl(mBrandId);
        String connectionUrlForLogs = mController.mAccountsController.getConnectionUrlForLogs(mBrandId);
        LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, "Unregister socket for brand " + mBrandId + ", connectionUrl = " + connectionUrlForLogs);
        SocketManager.getInstance().unregisterFromSocketState(connectionUrl, mSocketState);
    }


    public void networkLost() {
        LPMobileLog.d(TAG, LPMobileLog.FlowTags.LOGIN, "networkLost: brand " + mBrandId);
        mStateMachine.networkLost();
    }

    public void networkAvailable() {
        LPMobileLog.d(TAG, LPMobileLog.FlowTags.LOGIN, "networkAvailable: brand " + mBrandId);
        mStateMachine.networkAvailable();
    }

    public void moveToForeground() {
        LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, "moveToForeground for brand " + mBrandId);
        mStateMachine.moveToForeground();
    }

    public void moveToBackground(long timeoutTimerMs) {
        LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, "moveToBackground for brand " + mBrandId);
        mStateMachine.moveToBackground(timeoutTimerMs);
    }

    public void serviceStarted() {
        LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, "serviceStarted for brand " + mBrandId);
        mStateMachine.serviceStarted();
    }

    public void serviceStopped() {
        LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, "serviceStopped for brand " + mBrandId);
        mStateMachine.serviceStopped();
    }

    public void startConnecting(boolean connectInBackground,boolean clearToken) {
        LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, "startConnecting for brand, connectInBackground = " + connectInBackground);
        if (!mStateMachine.isConnecting()) {
            LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, "startConnecting for brand clearToken = " + clearToken);
            if (clearToken) {
                mController.mAccountsController.setToken(mBrandId, null);
            }
            mStateMachine.startConnecting(connectInBackground);
        }
    }

    public void startDisconnecting() {
        LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, "startDisconnecting for brand " + mBrandId);
        mStateMachine.startDisconnecting();
    }

    protected void onSocketProblem() {
        LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, "onSocketProblem for brand " + mBrandId);
        mStateMachine.onSocketProblem();
    }

    /**
     * Indicates if the connection process been finished
     *
     * @return TRUE in case there is a connection, return FALSE otherwise
     */
    public boolean isSocketOpen() {
        synchronized (this) {
            return mStateMachine.isConnected();
        }
    }

    public void setSocketReady(boolean connectedAndUpdating) {
        this.socketReady = connectedAndUpdating;
        sendSocketReadyStatus();
    }

    public boolean isSocketReady() {
        return socketReady;
    }

    private void sendSocketReadyStatus() {
        Bundle connectionBundle = new Bundle();
        connectionBundle.putBoolean(BROADCAST_KEY_SOCKET_READY_EXTRA, isSocketReady());
        connectionBundle.putString(BROADCAST_KEY_BRAND_ID, mBrandId);
        LocalBroadcast.sendBroadcast(BROADCAST_KEY_SOCKET_READY_ACTION, connectionBundle);
    }

    public boolean isConnecting() {
        return mStateMachine.isConnecting();
    }

    public long getClockDiff() {
        return mClockDiff;
    }

    public void setClock(long clock) {
        mClockDiff = clock;
    }

    public long getLastUpdateTime() {
        return mLastUpdateTime;
    }

    protected void getUpdates() {
        LPMobileLog.i(TAG, "Socket open - starting updating data...");
        initAmsSession();
    }

    private void initAmsSession(){
        new InitAmsSessionCommand(mController, mBrandId, new ICallback() {
            @Override
            public void onSuccess(Object value) {
                LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, "getUpdates - Socket connection updates Success");
                AmsConnectionAnalytics.socketReady();
                setSocketReady(true);
            }

            @Override
            public void onError(Throwable exception) {
                LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, "getUpdates - Error. " + exception.getMessage());
                onSocketProblem();
            }
        }).execute();
    }

    public boolean setLastUpdateTime(long lastUpdateTime) {
        boolean firstTime = false;
        if (mLastUpdateTime == 0) {
            firstTime = true;
        }
        mPreferenceManager.setLongValue(ConnectionsController.KEY_PREF_LAST_UPDATE_TIME, mBrandId, lastUpdateTime);
        mLastUpdateTime = lastUpdateTime;
        return firstTime;
    }

    public boolean isLastUpdateTimeExists() {
        return (mLastUpdateTime == 0);
    }

    @Override
    public void shutDown(ShutDownCompletionListener listener) {
        //shutting down the connection state machine.
        mStateMachine.shutDown(listener);
    }

    public void init() {
        if (!mStateMachine.isInitialized()) {
            mStateMachine.setStateMachineExecutor(new StateMachineExecutor(mStateMachine.getTag(), mStateMachine));
        }
    }

    public class AmsSocketState implements SocketStateListener {

        private BaseAmsAccountConnectionCallback callback;

        @Override
        public void onStateChanged(SocketState state) {
            LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, "onStateChanged with state " + state.name());
            switch (state) {
                case ERROR:
                case CLOSED:
                    if (callback != null) {
                        LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, "Notify error to task callback");
                        callback.onTaskError(TaskType.OPEN_SOCKET, new Exception("Open Socket - " + state.name()));
                        callback = null;
                    } else {
                        LPMobileLog.i(TAG, "Notify socket closed to state machine");
                        onSocketProblem();
                    }
                    break;
                case OPEN:
                    if (callback != null) {
                        LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, "Notify socket open successfully to task callback");
                        AmsConnectionAnalytics.openSocketTaskEnd();
                        callback.onTaskSuccess();
                        callback = null;
                    } else {
                        LPMobileLog.i(TAG, "Notify socket open successfully but callback is null - CHECK THIS OUT!");
                    }

                    //notify socket open
                    LocalBroadcast.sendBroadcast(BROADCAST_SOCKET_OPEN_ACTION);
                    //update data from socket.
                    getUpdates();
                    break;
            }
        }

        @Override
        public void onDisconnected(String reason, int code) {
            LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, "onDisconnected, reason " + reason + " code " + code);

            if (code == 0) {
                //that's a normal closing. will be handled by onStateChanged
                return;
            }

            //Notify the host app only when the token is expired
            if (code == IDP_TOKEN_EXPIRED_UNAUTHENTICATED_FLOW || code == IDP_TOKEN_EXPIRED_AUTHENTICATED_FLOW) {
                startDisconnecting();
                notifyHostAppTokenExpired();
            } else if(code == SocketHandler.CERTIFICATE_ERROR){
                startDisconnecting();
                notifyHostAppCertificateError(reason);
            } else {
                LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, "on disconnect: " + " code " + code + ", Notify socket closed to state machine");
                onSocketProblem();
            }


        }

        public void setCallback(BaseAmsAccountConnectionCallback callback) {
            this.callback = callback;
        }
    }

    private void notifyHostAppTokenExpired() {
            LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, mBrandId + ": notifying host app token expired!");
            mController.mEventsProxy.onTokenExpired();
    }

    private void notifyHostAppUserExpired() {
            LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, mBrandId + ": notifying host app user expired!");
            mController.mEventsProxy.onUnauthenticatedUserExpired();
    }

    private void notifyHostAppCertificateError(String errorMessage){
        LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, mBrandId + ": notifying host app invalid certificate");
        mController.mEventsProxy.onError(TaskType.INVALID_CERTIFICATE, errorMessage);
        LocalBroadcast.sendBroadcast(BROADCAST_CONNECTING_TO_SERVER_ERROR);
    }

    private void onDisconnected() {
        LPMobileLog.i(TAG, LPMobileLog.FlowTags.LOGIN, mBrandId + ": disconnected!");
        setIsUpdated(false);
        setSocketReady(false);
        unregisterSocketListener();
        mController.getMessagingEventSubscriptionManager().clearAllSubscriptions();
    }

       public boolean isUpdated() {
        return mIsUpdated;
    }

    public void setIsUpdated(boolean isUpdated) {
        if (isUpdated != mIsUpdated) {
            LPMobileLog.d(TAG, LPMobileLog.FlowTags.LOGIN, mBrandId + ": setIsUpdated = " + isUpdated);
            mIsUpdated = isUpdated;
            notifyOnConnectionCompleted(mIsUpdated);
            if (mIsUpdated) {
                mController.amsReadController.registerForegroundConnectionReceiver(mBrandId);
            }
        }
    }

    private void notifyOnConnectionCompleted(final boolean isUpdated) {
		Bundle connectionBundle = new Bundle();
		connectionBundle.putString(BROADCAST_KEY_BRAND_ID, mBrandId);
		connectionBundle.putBoolean(BROADCAST_AMS_CONNECTION_UPDATE_EXTRA, isUpdated);
		LocalBroadcast.sendBroadcast(BROADCAST_AMS_CONNECTION_UPDATE_ACTION, connectionBundle);

        mController.mEventsProxy.onConnectionChanged(isUpdated);
    }

    public boolean isAgentDetailsUpdated() {
        return mIsAgentDetailsUpdated;
    }

    public void setAgentDetailsUpdated(boolean updated) {
        this.mIsAgentDetailsUpdated = updated;
    }

    public boolean isFirstNotificationAfterSubscribe() {
        return firstNotificationAfterSubscribe;
    }

    public void setFirstNotificationAfterSubscribe(boolean firstNotificationAfterSubscribe) {
        this.firstNotificationAfterSubscribe = firstNotificationAfterSubscribe;
    }

}