package com.liveperson.messaging.model;

import android.content.Context;
import android.content.Intent;

import com.liveperson.infra.Command;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.LocalBroadcastReceiver;
import com.liveperson.messaging.controller.AccountsController;

/**
 * Created by Ilya Gazman on 11/26/2015.
 * <p/>
 * A command to perform a synchronized connection and provide a callback
 */
public class SynchronizedAuthenticationCompletedCallback implements Command {
    private static final String TAG = SynchronizedAuthenticationCompletedCallback.class.getSimpleName();
    private final AccountsController mAccountController;

    private String mBrandId;
    private boolean handled = false;
    private LocalBroadcastReceiver mLocalBroadcastReceiver;
    private ICallback<Void, Exception> mCallback;

    /**
     * Synchronously listening to when connection is accomplished
     *
     * @param accountsController
     * @param callback
     */
    public SynchronizedAuthenticationCompletedCallback(AccountsController accountsController, String brandId, ICallback<Void, Exception> callback) {
        mAccountController = accountsController;
        mBrandId = brandId;
        mCallback = callback;
    }

    @Override
    public void execute() {
        executeWithReturnValue();
    }

    public boolean executeWithReturnValue() {
        if (isValidAuthentication()) {
            handleConnection();
            return true;
        } else {
            registerToConnectionStateChanges();
            validateStatusDidNotChangedDuringRegistration();
            return false;
        }
    }

    private boolean isValidAuthentication() {
        return mAccountController.getAccount(mBrandId) != null && mAccountController.getAccount(mBrandId).isAuthenticatedCompleted() && !mAccountController.getAccount(mBrandId).isTokenExpired();
    }

    private void registerToConnectionStateChanges() {
        mLocalBroadcastReceiver = new LocalBroadcastReceiver.Builder()
                .addAction(AmsAccount.BROADCAST_KEY_AUTH_COMPLETED_ACTION)
                .addAction(AmsConnection.BROADCAST_CONNECTING_TO_SERVER_ERROR)
                .build(new LocalBroadcastReceiver.IOnReceive() {
                    @Override
                    public void onBroadcastReceived(Context context, Intent intent) {
                        if (AmsAccount.BROADCAST_KEY_AUTH_COMPLETED_ACTION.equals(intent.getAction())){
                            handleConnection();
                        }else if (AmsConnection.BROADCAST_CONNECTING_TO_SERVER_ERROR.equals(intent.getAction())) {
                            handleError();
                        }
                    }
                });
    }

    private synchronized void validateStatusDidNotChangedDuringRegistration() {
        if (!handled) {
            if (isValidAuthentication()) {
                handleConnection();
            }
        }
    }

    private synchronized void handleConnection() {
        if (handled) {
            return;
        }
        if (mLocalBroadcastReceiver != null) {
            mLocalBroadcastReceiver.unregister();
        }
        handled = true;
        mCallback.onSuccess(null);
    }
    private synchronized void handleError() {
        if (handled) {
            return;
        }
        if (mLocalBroadcastReceiver != null) {
            mLocalBroadcastReceiver.unregister();
        }
        handled = true;
        mCallback.onError(new Exception("Failed to connect"));
    }
}