package com.liveperson.messaging.controller.connection;

import com.liveperson.infra.ForegroundServiceInterface;
import com.liveperson.infra.log.FlowTags;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.model.types.AuthFailureReason;
import com.liveperson.infra.model.types.ChatState;
import com.liveperson.infra.network.BackOff;
import com.liveperson.infra.network.ExponentialBackOff;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDownAsync;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDownCompletionListener;
import com.liveperson.infra.statemachine.BaseStateMachine;
import com.liveperson.messaging.LpError;
import com.liveperson.messaging.TaskType;
import com.liveperson.messaging.commands.ResolveConversationCommand;
import com.liveperson.messaging.commands.tasks.BaseAmsAccountConnectionTask;
import com.liveperson.messaging.controller.connection.connectionevents.BackgroundEvent;
import com.liveperson.messaging.controller.connection.connectionevents.BackgroundTimeOutEvent;
import com.liveperson.messaging.controller.connection.connectionevents.ConnectEvent;
import com.liveperson.messaging.controller.connection.connectionevents.DisconnectEvent;
import com.liveperson.messaging.controller.connection.connectionevents.ForegroundEvent;
import com.liveperson.messaging.controller.connection.connectionevents.NetworkAvailableEvent;
import com.liveperson.messaging.controller.connection.connectionevents.NetworkLostEvent;
import com.liveperson.messaging.controller.connection.connectionevents.RunTaskEvent;
import com.liveperson.messaging.controller.connection.connectionevents.SecondaryConnectEvent;
import com.liveperson.messaging.controller.connection.connectionevents.SecondaryTaskSuccessEvent;
import com.liveperson.messaging.controller.connection.connectionevents.SendStateEvent;
import com.liveperson.messaging.controller.connection.connectionevents.ServiceOffEvent;
import com.liveperson.messaging.controller.connection.connectionevents.ServiceOnEvent;
import com.liveperson.messaging.controller.connection.connectionevents.ShutDownEvent;
import com.liveperson.messaging.controller.connection.connectionevents.SocketProblemEvent;
import com.liveperson.messaging.controller.connection.connectionevents.TaskFailedEvent;
import com.liveperson.messaging.controller.connection.connectionevents.TaskSuccessEvent;
import com.liveperson.messaging.controller.connection.connectionevents.TaskTimeOutEvent;

import java.util.ArrayList;
import java.util.List;

import static com.liveperson.infra.errors.ErrorCode.ERR_000000A4;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000A5;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000A6;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000146;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000147;

/**
 * Created by shiranr on 13/03/2016.
 */
public class ConnectionStateMachine extends BaseStateMachine implements ShutDownAsync {

    private static final String TAG = "ConnectionStateMachine";
    private final static long TIMEOUT_INTERVAL = 2 * 1000;

    private int mCurrentTaskIndex;
    private String mBrandId;
    private List<BaseAmsAccountConnectionTask> mConnectingTasks;

    private List<BaseAmsAccountConnectionTask> mDisconnectingTasks;

    private ConnectingState mConnectingState;
    private DisconnectingState mDisconnectingState;
	private ConnectedForegroundState mConnectedForegroundState;
	private ConnectedForegroundServiceState mConnectedForegroundServiceState;
	private ConnectedBackgroundState mConnectedBackgroundState;
	private ConnectedBackgroundServiceState mConnectedBackgroundServiceState;
	private DisconnectedState mDisconnectedState;

	private final MessagingStateMachineInterface mController;
    private final InternetInformationProvider mInternetInformationProvider;
    private final IConnectionListener mStateListener;
    private final ForegroundServiceInterface mForegroundServiceInterface;

    private final ExponentialBackOff mExponentialBackOff;
	private long mBackgroundConnectionTimerMs;

	// This stores the next timeout time (absolute) so when we get a new background time
	// don't set it if the current timeout is longer than the new
	private long mAbsoluteTimeForTimeout;

	// Holds all the connection tasks for all types of connection (primary/secondary)
	private ConnectionTasksHolder mConnectionTasksHolder;

	public ConnectionStateMachine(MessagingStateMachineInterface controller, InternetInformationProvider internetInformationProvider,
                                  ForegroundServiceInterface foregroundServiceInterface,
                                  String brandId, IConnectionListener connectionListener) {
        super(TAG);
        mController = controller;
		mBrandId = brandId;
		mConnectionTasksHolder = new ConnectionTasksHolder(this, mController, mBrandId);
		mDisconnectingTasks = mConnectionTasksHolder.getDisconnectionTasks();
		initStates(getTag());
        initActiveState(mDisconnectedState);

        mInternetInformationProvider = internetInformationProvider;
        mForegroundServiceInterface = foregroundServiceInterface;
        mStateListener = connectionListener;

        mExponentialBackOff = new ExponentialBackOff.Builder()
                .setInitialIntervalMillis(1)
                .setMaxElapsedTimeMillis(300000) //max of 5 minutes
                .setMaxIntervalMillis(40000)
                .setMultiplier(2)
                .build();
    }

	/**
	 * Initialize all states of this state machine:
	 *  ConnectingState
	 *  DisconnectingState
	 *  ConnectedForegroundState
	 *  ConnectedForegroundServiceState
	 *  ConnectedBackgroundState
	 *  ConnectedBackgroundServiceState
	 *  DisconnectedState
	 *
	 * @param tag TAG for logs
	 */
	private void initStates(String tag) {
		ForegroundParentState mForegroundParentState = new ForegroundParentState("ForegroundParentState", tag + "_ForegroundParentState");
		BackgroundParentState mBackgroundParentState = new BackgroundParentState("BackgroundParentState", tag + "_BackgroundParentState");

		mConnectingState = new ConnectingState("ConnectingState", tag + "_ConnectingState");
		mDisconnectingState = new DisconnectingState("DisconnectingState", tag + "_DisconnectingState");
		mConnectedForegroundState = new ConnectedForegroundState(mForegroundParentState, "ConnectedForegroundState", tag + "_ConnectedForegroundState");
		mConnectedForegroundServiceState = new ConnectedForegroundServiceState(mForegroundParentState, "ConnectedForegroundServiceState", tag + "_ConnectedForegroundServiceState");
		mConnectedBackgroundState = new ConnectedBackgroundState(mBackgroundParentState, "ConnectedBackgroundState", tag + "_ConnectedBackgroundState");
		mConnectedBackgroundServiceState = new ConnectedBackgroundServiceState(mBackgroundParentState, "ConnectedBackgroundServiceState", tag + "_ConnectedBackgroundServiceState");
		mDisconnectedState = new DisconnectedState("DisconnectedState", tag + "_DisconnectedState");
	}

    public boolean isConnected(){
        return ((BaseConnectionState)activeState()).isConnected();
    }

    public boolean isConnecting(){
        return ((BaseConnectionState)activeState()).isConnecting();
    }

	/*---------------------- Operations ---------------------*/

	/**
	 * Start the connection process
	 */
    public void startConnecting(boolean connectInBackground) {
        postEvent(new ConnectEvent(connectInBackground));
    }

	/**
	 * Start the disconnection process
	 */
    public void startDisconnecting() {
        postEvent(new DisconnectEvent());
    }

	/**
	 * Inform network was lost
	 */
	public void networkLost(){
		postEvent(new NetworkLostEvent());
	}

	/**
	 * Inform network is available
	 */
	public void networkAvailable(){
		postEvent(new NetworkAvailableEvent());
	}

	/**
	 * Inform that the conversation screen was moved to foreground
	 */
	public void moveToForeground() {
		postEvent(new ForegroundEvent());
	}

	/**
	 * Inform that the conversation screen was moved to background
	 */
	public void moveToBackground(long timeoutTimerMs) {
		mBackgroundConnectionTimerMs = timeoutTimerMs;
		postEvent(new BackgroundEvent());
	}

	/**
	 * Inform that the upload image service was started
	 */
	public void serviceStarted() {
		postEvent(new ServiceOnEvent());
	}

	/**
	 * Inform that the upload image service was stopped
	 */
	public void serviceStopped() {
		postEvent(new ServiceOffEvent());
	}

	/**
	 * Inform that the socket closed
	 */
	public void onSocketProblem() {
		postEvent(new SocketProblemEvent());
	}

	/**
	 * Start the shutdown process on this state machine
	 * @param listener
	 */
    @Override
    public void shutDown(ShutDownCompletionListener listener) {
        // running disconnect flow and shutting down when finished.
        postEvent(new ShutDownEvent(listener));
    }

    /**
     * Update value of fullConnectionFlowRequired indicating whether to run fullConnectingTasks or shortConnectingTasks
     * @param brandId
     * @param isFullConnectionFlowRequired
     */
    private void setFullConnectionRequired(String brandId, boolean isFullConnectionFlowRequired) {
        IConnectionParamsCache cache = mController.getAccountsController().getConnectionParamsCache(brandId);
        if (cache != null) {
            cache.setFullConnectionFlowRequired(isFullConnectionFlowRequired);
        } else {
            LPLog.INSTANCE.d(TAG, "Failed to get connection params cache - brandId = " + brandId
                    + " - isFullConnectionFlowRequired = " + isFullConnectionFlowRequired);
        }
    }

    /*--------------------------- STATES ------------------------------------*/

	//------------------------- ConnectingState -----------------------------

	/**
     * ConnectingState is active during connecting process. after finishing it moves back to ConnectedForeground state.
     * When getting @link DisconnectEvent we wait until the current task is success or failed, than start disconnect process.
     * When getting @link ConnectEvent we cancel waiting disconnect event if exists.
     */
    class ConnectingState extends BaseConnectionState{

        boolean isDisconnectWaiting = false;
        boolean isBackgroundWaiting = false;
        private Runnable mDelayedTask = null;
        private ShutDownEvent mShutDownEvent = null;
		private boolean mSecondaryConnectionNeeded = false;

        public ConnectingState(String name, String logTag) {
            super(name, logTag);
        }

        @Override
        public boolean isConnecting() {
            return true;
        }

        @Override
        public void actionOnEntry() {
            super.actionOnEntry();
            mStateListener.notifyStartConnecting();


            // If CSDS information does not exist OR one of the versions (LE or AC) incompatible (kill-switch activated) we initiate the full connection flow
            if (mController.getAccountsController().isCsdsDataMissing(mBrandId) ||
                    !mController.getAccountsController().getConnectionParamsCache(mBrandId).isVersionsCompatible() ||
                    mController.getAccountsController().getConnectionParamsCache(mBrandId).isFullConnectionFlowRequired()) {

	            LPLog.INSTANCE.d(TAG, FlowTags.LOGIN, "ConnectingState actionOnEntry: CSDS data is missing from persistence OR version wasn't compatible on last connection or full flow required.");
                mConnectingTasks = mConnectionTasksHolder.getPrimaryFullConnectionTasks();
                // Since we're in the full connection flow we don't need the secondary connection
                mSecondaryConnectionNeeded = false;
            } else {
	            LPLog.INSTANCE.d(TAG, FlowTags.LOGIN, "ConnectingState actionOnEntry: Persistence has CSDS data. Start short connection process");
                mConnectingTasks = mConnectionTasksHolder.getPrimaryShortConnectionTasks();
                // Since we're in the short connection flow we need the secondary connection
                mSecondaryConnectionNeeded = true;
            }
            mCurrentTaskIndex = 0;
            mExponentialBackOff.reset();
            isDisconnectWaiting = false;
            isBackgroundWaiting = false;
            mShutDownEvent = null;
            apply(new RunTaskEvent());
        }

        /**
         * A connection task finished successfully
         * @param ev
         */
        @Override
        public void visit(SecondaryTaskSuccessEvent ev) {
            LPLog.INSTANCE.d(TAG, "Ignoring SecondaryTaskSuccessEvent: " + ev);
        }

        @Override
        public void visit(ConnectEvent ev) {
            super.visit(ev);
            LPLog.INSTANCE.d(TAG, "Already Connecting");
            isDisconnectWaiting = false;
        }

        @Override
        public void visit(DisconnectEvent ev) {
            super.visit(ev);
            LPLog.INSTANCE.d(TAG, "Got DisconnectEvent...");
            isDisconnectWaiting = true;
            if (mDelayedTask != null){
                changeState(mDisconnectingState);
            }
        }

		@Override
		public void visit(ForegroundEvent ev) {
			super.visit(ev);
			LPLog.INSTANCE.d(TAG, "Cancel any background waiting event");
			isBackgroundWaiting = false;
		}

		@Override
        public void visit(BackgroundEvent ev) {
            super.visit(ev);
            LPLog.INSTANCE.d(TAG, "Got BackgroundEvent. Disconnecting...");
            isBackgroundWaiting = true;
            if (mDelayedTask != null){
                changeState(mDisconnectingState);
            }
        }

		/**
		 * A connection task finished successfully
		 * @param ev
		 */
        @Override
        public void visit(TaskSuccessEvent ev) {
            super.visit(ev);

	        LPLog.INSTANCE.d(TAG, "Task " + ev.getTaskName() + " finished successfully");
            if(!handleWaitingEvents()){
                mCurrentTaskIndex++;
                mExponentialBackOff.reset();
                LPLog.INSTANCE.d(TAG, "Running next task...");

                postDelayEvent(new RunTaskEvent(mCurrentTaskIndex, mExponentialBackOff.getRetryNumber()), mExponentialBackOff.getNextBackOffMillis());
            }
        }

		/**
		 * A connection task failed
		 * @param ev
		 */
        @Override
        public void visit(TaskFailedEvent ev) {
            super.visit(ev);

            String errorMessage;
            if (ev.getException() == null) {
                errorMessage = "--no exception--";
            } else {
                errorMessage = ev.getException().getMessage();
            }

            LPLog.INSTANCE.e(TAG, ERR_000000A4, "Connection task " + ev.getType() + " failed.", ev.getException());

            if (ev.getType() == TaskType.IDP && ev.getFailureReason() == AuthFailureReason.USER_EXPIRED) {
                LPLog.INSTANCE.e(TAG, ERR_000000A5, "User expired! reconnecting with new User");

                onUserExpired();
                //after notifying user is expired, we create new user, in order to do so we need to clean the
                //non auth code, so next retry of idp request will not have it and we'll generate new non-auth code for new user.

                //resetting the retry counter
                mExponentialBackOff.reset();
            } else if (ev.getType() == TaskType.IDP && ev.getFailureReason() == AuthFailureReason.TOKEN_EXPIRED) {
                mController.getAmsUsers().clearConsumerFromDB(mBrandId);
                mStateListener.notifyTokenExpired();
                changeState(mDisconnectedState);
                mExponentialBackOff.reset();
            } else if (ev.getType() == TaskType.IDP && ev.getFailureReason() == AuthFailureReason.INVALID_CERTIFICATE) {
                // We should see If we can change task type to IDP
                notifyError(TaskType.INVALID_CERTIFICATE, LpError.INVALID_CERTIFICATE, errorMessage);
            } else {
                //continue as usual.. calculate next retry..
                mExponentialBackOff.calculateNextBackOffMillis();
            }

            if (ev.getType() == TaskType.IDP || mExponentialBackOff.getNextBackOffMillis() == BackOff.STOP) {
                LPLog.INSTANCE.w(TAG, "Connection task " + ev.getType() + " failed. Finishing connecting flow.");
                notifyError(ev.getType(), ev.getLpError(), errorMessage);
                changeState(mDisconnectedState);
                setFullConnectionRequired(mBrandId, true);
                return;
            }

            if (!handleWaitingEvents()) {
                LPLog.INSTANCE.d(TAG, "Scheduling Task " + ev.getTaskName() + " in " + mExponentialBackOff.getNextBackOffMillis() + " millis. retry number = " + mExponentialBackOff.getRetryNumber());
                mDelayedTask = postDelayEvent(new RunTaskEvent(mCurrentTaskIndex, mExponentialBackOff.getRetryNumber()), mExponentialBackOff.getNextBackOffMillis());
            }
        }

		/**
		 * Run the next connection tasks
		 * @param ev
		 */
        @Override
        public void visit(RunTaskEvent ev) {
            mDelayedTask = null;
            if(!handleWaitingEvents()){
                boolean isBrandForeground = mForegroundServiceInterface.isBrandForeground(mBrandId);
                if (mCurrentTaskIndex < mConnectingTasks.size()) {
                    BaseAmsAccountConnectionTask currentTask = mConnectingTasks.get(mCurrentTaskIndex);
                    //some tasks require being in FG- for example: OPEN SOCKET - if we are in BG- we'll skip those tasks
                    if (currentTask.requireSDKinForeground() && !isBrandForeground) {
                        //If we are not in FG - we move to the next task
                        mCurrentTaskIndex++;
                        apply(ev);

                    } else {
                        currentTask.setBrandId(mBrandId);
                        currentTask.setIsSecondaryTask(false);

                        LPLog.INSTANCE.d(TAG, "Running task: " + currentTask.getClass().getSimpleName() + " Retry #" + mExponentialBackOff.getRetryNumber() + ", After delay: " + mExponentialBackOff.getNextBackOffMillis());
                        currentTask.execute();
                    }

                } else {
                    LPLog.INSTANCE.d(TAG, "Connected flow finished successfully! :)");
					// If we connected using the short flow, we need to initiate the secondary connection flow
					if(mSecondaryConnectionNeeded) {
						LPLog.INSTANCE.d(TAG, "RunTaskEvent: initiate the secondary connection flow");
						if (isBrandForeground){
						    changeStateAndPassEvent(mConnectedForegroundState, new SecondaryConnectEvent());
                        }else{
    						//in case we completed connecting on BG - we move back to disconnected state
                            changeState(mDisconnectingState);
						}
					}
					else{
						// We finished successfully a full connection flow
                        setFullConnectionRequired(mBrandId, false);

                        if (isBrandForeground){
                            changeState(mConnectedForegroundState);
                        }else{
                            //in case we completed connecting on BG - we move back to disconnected state
                            changeState(mDisconnectingState);
                        }
					}
                }
            }
        }

		/**
		 * Handle any pending connecting/disconnecting events
		 * @return
		 */
        @SuppressWarnings("BooleanMethodIsAlwaysInverted") //Currently method is better for understanding.
		private boolean handleWaitingEvents() {
            if (mShutDownEvent != null){
                LPLog.INSTANCE.d(TAG, "Disconnected event is waiting. running disconnect flow");
                changeStateAndPassEvent(mDisconnectingState, mShutDownEvent);
                return true;
            }if (isDisconnectWaiting || isBackgroundWaiting){
                LPLog.INSTANCE.d(TAG, "ShutDown-Disconnect event is waiting. running disconnect flow with ShutDown flow");
                changeState(mDisconnectingState);
                return true;
            }
            return false;
        }

        @Override
        public void visit(ShutDownEvent ev) {
            LPLog.INSTANCE.d(TAG, "Got ShutDownEvent");
            mShutDownEvent = ev;
        }

        @Override
        public void actionOnExit() {
            super.actionOnExit();
            if (mDelayedTask != null){
                LPLog.INSTANCE.d(TAG, "Clearing waiting delayed tasks.");
                cancelDelayedEvent(mDelayedTask);
                mDelayedTask = null;
            }
        }
    }

    /**
     * Clean all data related to this user. also, close locally the current conversation
     */
    private void onUserExpired() {
        //because IDP task failed, we will try again to connect to IDP. this time with new user.

        // close current open conversations if exists
        ResolveConversationCommand resolveConversationCommand = new ResolveConversationCommand(mController.getAmsConversations(), mBrandId, null);
        resolveConversationCommand.setOfflineMode(true);
        resolveConversationCommand.execute();

        //remove user id from DB -
        mController.getAmsUsers().clearConsumerFromDB(mBrandId);

        //Notify the host app the user expired
        mStateListener.notifyUserExpired();
    }

    //------------------------- DisconnectingState -----------------------------

     /**
     * DisconnectingState is active during disconnecting process. after finishing it moves back to Disconnected state.
     * When getting @link ConnectEvent we wait until the current task is success or failed, than start connecting process.
     * When getting @link DisconnectEvent we cancel waiting connected event if exists.
     */
    class DisconnectingState extends BaseConnectionState{

        private boolean isConnectingWaiting = false;
        private ShutDownEvent mShutDownFlowEvent = null;
        private Runnable mTimeoutEvent = null;

        public DisconnectingState(String name, String logTag) {
            super(name, logTag);
        }

        @Override
        public void actionOnEntry() {
            super.actionOnEntry();
            mStateListener.notifyStartDisconnecting();
            mCurrentTaskIndex = 0;
            mExponentialBackOff.reset();
            mShutDownFlowEvent = null;
            mTimeoutEvent = null;
            apply(new RunTaskEvent());
        }

        @Override
        public void visit(ShutDownEvent ev) {
            LPLog.INSTANCE.d(TAG, "Got ShutDownEvent, Disconnecting...");
            mShutDownFlowEvent = ev;
        }

        @Override
        public void visit(ConnectEvent ev) {
            LPLog.INSTANCE.d(TAG, "Got ConnectEvent...");
            isConnectingWaiting = true;
        }

        @Override
        public void visit(ForegroundEvent ev) {
            LPLog.INSTANCE.d(TAG, "Got ForegroundEvent...");
            isConnectingWaiting = true;
        }

        @Override
        public void visit(DisconnectEvent ev) {
            LPLog.INSTANCE.d(TAG, "Already Disconnecting");
            isConnectingWaiting = false;
        }

		 @Override
		 public void visit(NetworkAvailableEvent ev) {
			 LPLog.INSTANCE.d(TAG, "NetworkAvailableEvent. Currently disconnecting, set pending connection");
			 isConnectingWaiting = true;
		 }

		 @Override
		 public void visit(NetworkLostEvent ev) {
			 LPLog.INSTANCE.d(TAG, "Network lost. Remove the pending connection");
			 isConnectingWaiting = false;
		 }

	     /**
	      * A disconnection task finished successfully
	      */
	     @Override
	     public void visit(TaskSuccessEvent ev) {
		     LPLog.INSTANCE.d(TAG, "Task " + ev.getTaskName() + " finished successfully");
		     cancelDelayedEvent(mTimeoutEvent);
		     mExponentialBackOff.reset();
		     runNextTask();
	     }

		 /**
		  * A disconnection task failed
		  * @param ev
		  */
		 @Override
        public void visit(TaskFailedEvent ev) {
            LPLog.INSTANCE.e(TAG, ERR_00000146, "Task " + ev.getTaskName() + " failed");
            //no retry in disconnect flow.
            runNextTask();
        }

        @Override
        public void visit(TaskTimeOutEvent ev) {
            LPLog.INSTANCE.e(TAG, ERR_00000147, "Timeout expired for task " + ev.getTaskName() + ". failing this task.");
            ev.failTask();
        }

		 /**
		  * Run the next disconnection task
		  */
		 private void runNextTask() {
            if (isConnectingWaiting && mForegroundServiceInterface.isBrandForeground(mBrandId)){
				LPLog.INSTANCE.d(TAG, "runNextTask: Connected event is waiting and we're in foreground. Running connect flow...");
				changeState(mConnectingState);
            }else{
                mCurrentTaskIndex++;
                LPLog.INSTANCE.d(TAG,"Running Next Task......");
                postDelayEvent(new RunTaskEvent(mCurrentTaskIndex, mExponentialBackOff.getRetryNumber()), mExponentialBackOff.getNextBackOffMillis());
            }
        }

        @Override
        public void visit(RunTaskEvent ev) {
            if (isConnectingWaiting && mForegroundServiceInterface.isBrandForeground(mBrandId)){
				LPLog.INSTANCE.d(TAG, "RunTaskEvent: Connected event is waiting and we're in foreground. Running connect flow...");
				changeState(mConnectingState);
            }else{

                if (mCurrentTaskIndex < mDisconnectingTasks.size()) {
                    BaseAmsAccountConnectionTask currentTask = mDisconnectingTasks.get(mCurrentTaskIndex);
                    currentTask.setBrandId(mBrandId);

                    LPLog.INSTANCE.d(TAG, "Running task: " + currentTask.getClass().getSimpleName() + " Retry #" + mExponentialBackOff.getRetryNumber() + ", After delay: " + mExponentialBackOff.getNextBackOffMillis());
                    currentTask.execute();
                    //schedule timeout event in case we don't get response in a reasonable time.
                    scheduleTimeoutTask(currentTask);

                } else {
                    LPLog.INSTANCE.d(TAG, "Disconnected flow finished successfully! :)");
                    changeState(mDisconnectedState);
                    if (mShutDownFlowEvent != null) {
                        notifyShutDownCompleted(mShutDownFlowEvent);
                    }
                }
            }
        }
        /**
         * schedule timeout event in case we don't get response in a reasonable time.
         */
        private void scheduleTimeoutTask(BaseAmsAccountConnectionTask currentTask) {
            TaskTimeOutEvent timeoutEvent = new TaskTimeOutEvent(currentTask, TaskType.CLOSING_SOCKET);
            mTimeoutEvent = postDelayEvent(timeoutEvent, TIMEOUT_INTERVAL);
            LPLog.INSTANCE.d(TAG, "scheduling TimeOut for currentTask = [" + currentTask.getName() + "]");
        }

        @Override
        public void actionOnExit() {
            cancelDelayedEvent(mTimeoutEvent);
            mTimeoutEvent = null;
            isConnectingWaiting = false;
            mExponentialBackOff.reset();
        }

    }

	//------------------------- ConnectedForegroundState -----------------------------

	/**
	 * ConnectedForegroundState is active when we are connected and the the conversation screen is in the foreground
	 */
	class ConnectedForegroundState extends ConnectedBaseState {

        @Override
        public void actionOnEntry() {
            super.actionOnEntry();
            mStateListener.notifyConnected();
        }

        public ConnectedForegroundState(BaseConnectionState parent, String name, String logTag) {
			super(parent, name, logTag);
		}

		/**
		 * A service was started and we're still on the foreground. Change to {@link ConnectedForegroundServiceState}
		 * @param ev
		 */
		@Override
		public void visit(ServiceOnEvent ev) {
			super.visit(ev);
			changeState(mConnectedForegroundServiceState);
		}

		/**
		 * Conversation screen was moved to background. Change state to {@link ConnectedBackgroundState} and send the {@link SendStateEvent}
		 * so the background state know that it needs to send the change state event
		 * @param ev
		 */
		@Override
		public void visit(BackgroundEvent ev) {
			super.visit(ev);
			changeStateAndPassEvent(mConnectedBackgroundState, new SendStateEvent());
		}

		/**
		 * Indicate that this state should send a change state publish event
		 * @param ev
		 */
		@Override
		public void visit(SendStateEvent ev) {
			super.visit(ev);
		}

	}

    private void reconnect() {
        //if we got here, could be we lost network, and the socket got it before network lost event.
        //no need to try reconnect if we don't have network connection.
        //from disconnect state we'll reconnect when we'll get network available event.
        if (mInternetInformationProvider != null &&
                mInternetInformationProvider.isNetworkAvailable()){
            changeStateAndPassEvent(mDisconnectingState, new ConnectEvent());
        }else{
            changeState(mDisconnectedState);
        }
    }

    //------------------------- ConnectedBackgroundState -----------------------------

	/**
	 * ConnectedBackgroundState is active when we are connected but on the background
	 */
	class ConnectedBackgroundState extends ConnectedBaseState {
		public ConnectedBackgroundState(BaseConnectionState parent, String name, String logTag) {
			super(parent, name, logTag);
		}

		/**
		 * A service was started. Change to {@link ConnectedBackgroundServiceState}
		 * @param ev
		 */
		@Override
		public void visit(ServiceOnEvent ev) {
			super.visit(ev);
			changeState(mConnectedBackgroundServiceState);
		}

		/**
		 * Move the foreground. Change state to {@link ConnectedForegroundState} and send the {@link SendStateEvent}
		 * so the receiving state will know it should send and change state event
		 * @param ev
		 */
		@Override
		public void visit(ForegroundEvent ev) {
			super.visit(ev);
			changeStateAndPassEvent(mConnectedForegroundState, new SendStateEvent());
		}

		/**
		 * A new background event while in the background. We let the parent update the background timeout time if needed
		 * @param ev
		 */
		public void visit(BackgroundEvent ev) {
			super.visit(ev);
		}

		/**
		 * A background timeout expired. It's time to disconnect.
		 * @param ev
		 */
		@Override
		public void visit(BackgroundTimeOutEvent ev) {
			super.visit(ev);
			LPLog.INSTANCE.d(TAG, "BackgroundTimeoutEvent. We're in background without service. Disconnecting...");
			changeState(mDisconnectingState);
		}

		/**
		 * Send a stateChange publish message
		 * @param ev
		 */
		@Override
		public void visit(SendStateEvent ev) {
			super.visit(ev);
		}

	}

	//------------------------- ConnectedForegroundServiceState -----------------------------

	/**
	 * ConnectedForegroundServiceState is active when we are connected, in the foreground and an upload
	 * image service is active
	 */
	class ConnectedForegroundServiceState extends ConnectedBaseState {
		public ConnectedForegroundServiceState(BaseConnectionState parent, String name, String logTag) {
			super(parent, name, logTag);
		}

		/**
		 * The upload image service has turned off. Change to {@link ConnectedForegroundState}
		 * @param ev
		 */
		@Override
		public void visit(ServiceOffEvent ev) {
			super.visit(ev);
			changeState(mConnectedForegroundState);
		}

		/**
		 * Conversation screen was moved to background. Change to {@link ConnectedBackgroundServiceState}
		 * and send the {@link SendStateEvent} so the receiving state know it needs to send the BACKGROUND state
		 * @param ev
		 */
		@Override
		public void visit(BackgroundEvent ev) {
			super.visit(ev);
			changeStateAndPassEvent(mConnectedBackgroundServiceState, new SendStateEvent());
		}


		/**
		 * Send a stateChange publish message
		 * @param ev
		 */
		@Override
		public void visit(SendStateEvent ev) {
			super.visit(ev);
		}

	}

	//------------------------- ConnectedBackgroundServiceState -----------------------------

	/**
	 * ConnectedBackgroundServiceState is active when we are connected, the conversation screen is in
	 * the background and the upload image service is on
	 */
	class ConnectedBackgroundServiceState extends ConnectedBaseState {

		public ConnectedBackgroundServiceState(AbstractBackgroundParentState parent, String name, String logTag) {
			super(parent, name, logTag);
		}

		/**
		 * The upload image service turned off.
		 * If there is a pending timeout we change to {@link ConnectedBackgroundState}.
		 * If there is no pending timeout, we disconnect.
		 * @param ev
		 */
		@Override
		public void visit(ServiceOffEvent ev) {

			if (((AbstractBackgroundParentState) parent).getBackgroundConnectionTimeoutEvent() != null) {
				changeState(mConnectedBackgroundState);
			}
			else {
				changeState(mDisconnectingState);
			}
		}

		/**
		 * We got a new background event. Update the background timeout if needed
		 * @param ev
		 */
		public void visit(BackgroundEvent ev) {
			super.visit(ev);
		}

		/**
		 * The conversation screen was moved to the foreground. Change to {@link ConnectedForegroundServiceState}
		 * and send the {@link SendStateEvent} so the receiving state know it needs to send the ACTIVE state
		 * @param ev
		 */
		@Override
		public void visit(ForegroundEvent ev) {
			super.visit(ev);
			changeStateAndPassEvent(mConnectedForegroundServiceState, new SendStateEvent());
		}

		/**
		 * Inform that we need to send the StateChange publish message
		 * @param ev
		 */
		@Override
		public void visit(SendStateEvent ev) {
			super.visit(ev);
		}

	}

	//------------------------- DisconnectedState -----------------------------

	/**
	 * DisconnectedState is active when we are disconnected from the server
	 */
	class DisconnectedState extends BaseConnectionState{

		public DisconnectedState(String name, String logTag) {
			super(name, logTag);
		}


        @Override
        public void actionOnEntry() {
            super.actionOnEntry();
            mStateListener.notifyDisconnected();
            handleConnectionRegistration();
        }

        @Override
        public void actionOnExit() {
            super.actionOnExit();
            //where ever we go - we will need network changes
            mInternetInformationProvider.registerToNetworkChanges();
        }

        /**
		 * Connect to server if the is and internet connection and we are still not connected
		 * @param ev
		 */
		@Override
		public void visit(ConnectEvent ev) {
			// Connect if network available and we're not already connected
			if (mInternetInformationProvider != null &&
                    mInternetInformationProvider.isNetworkAvailable() &&
                    !mController.getMessagingController().isSocketOpen(mBrandId) &&
                    //we only connect in FG unless we got special request to connect in BG
                    (mForegroundServiceInterface.isBrandForeground(mBrandId) || ev.connectInBG())
                    ){
				changeState(mConnectingState);
			}else{
				LPLog.INSTANCE.d(TAG, "ignoring ConnectEvent (either no connection or already connected or in bg");
			}
		}

		/**
		 * The conversation screen was moved to foreground. Connect to server if there is and internet connection and we are still not connected
		 * @param ev
		 */
		@Override
		public void visit(ForegroundEvent ev) {

            handleConnectionRegistration();

            boolean socketOpen = mController.getMessagingController().isSocketOpen(mBrandId);
            boolean networkAvailable = mInternetInformationProvider != null && mInternetInformationProvider.isNetworkAvailable();
            if (networkAvailable && !socketOpen){
				changeState(mConnectingState);
			}else{
                LPLog.INSTANCE.d(TAG, "ignoring ForegroundEvent (either no connection or already connected) network: " + networkAvailable
                 + "is socket open = " + socketOpen );
			}
		}

        @Override
        public void visit(BackgroundEvent ev) {
            handleConnectionRegistration();
        }


        /**
         * Network is available. Connect to server if the conversation is on the foreground
         * @param ev
         */
		@Override
		public void visit(NetworkAvailableEvent ev) {
			if (mForegroundServiceInterface.isBrandForeground(mBrandId)){
				changeState(mConnectingState);
			}else{
				LPLog.INSTANCE.d(TAG, "ignoring NetworkAvailableEvent (conversation is not in the foreground)");
			}
		}

		@Override
		public void visit(DisconnectEvent ev) {
			LPLog.INSTANCE.d(TAG, "ignoring DisconnectEvent (already connected");

		}

		@Override
		public void visit(ShutDownEvent ev) {
			//nothing to do.. we are not connected and this is impossible..
			notifyShutDownCompleted(ev);
		}


        private void handleConnectionRegistration() {
            //if we are in FG and there is no connection available - register to network changes
            if (mForegroundServiceInterface.isBrandForeground(mBrandId)){
                if (!mInternetInformationProvider.isNetworkAvailable()){
                    //keep listening to network changes
                    LPLog.INSTANCE.d(TAG, "Register to network changes");
                    mInternetInformationProvider.registerToNetworkChanges();
                }
            }else{
                LPLog.INSTANCE.d(TAG, "Unregister to network changes");
                mInternetInformationProvider.unregisterToNetworkChanges();
            }
        }


    }

	//------------------------- ForegroundParentState -----------------------------

	/**
	 * This is the parent state for foreground states. Currently it has the responsibility to send the changeChatState (ACTIVE) message
	 */
	class ForegroundParentState extends BaseConnectionState {


		private List<BaseAmsAccountConnectionTask> mSecondaryConnectingTasks = new ArrayList<>(4);

		public ForegroundParentState(String name, String logTag) {
			super(name, logTag);
		}


		@Override
		public void visit(SecondaryConnectEvent ev) {
			super.visit(ev);

			mSecondaryConnectingTasks = mConnectionTasksHolder.getSecondaryConnectionTasks();
			mCurrentTaskIndex = 0;
			mExponentialBackOff.reset();
			apply(new RunTaskEvent());

		}

		/**
		 * This indicates that we need to send here the ACTIVE chatState
		 * @param ev
		 */
		@Override
		public void visit(SendStateEvent ev) {
			super.visit(ev);

			LPLog.INSTANCE.d(TAG, "ForegroundParentState: SendStateEvent: Sending ACTIVE state (if conversation active)");
			// Send ACTIVE state
			mController.getMessagingController().changeChatState(mBrandId, mBrandId, ChatState.ACTIVE);
		}

		/**
		 * The network is available. Connect
		 * @param ev
		 */
		@Override
		public void visit(NetworkAvailableEvent ev) {
			super.visit(ev);
			LPLog.INSTANCE.d(TAG, "visit: We're in the foreground and got NetworkAvailable. Connect...");
            reconnect();
		}

        /**
         * Socket is closed for some reason, we are in FG. need to reconnect.
         * @param ev
         */
        @Override
        public void visit(SocketProblemEvent ev) {
            super.visit(ev);
            reconnect();
        }

		/**
		 * We're in the foreground. If we got a connect event we try to reconnect
		 * @param ev
		 */
		@Override
		public void visit(ConnectEvent ev) {
			super.visit(ev);
            reconnect();
		}

		/**
		 * A connection task finished successfully
		 * @param ev
		 */
		@Override
		public void visit(SecondaryTaskSuccessEvent ev) {
			LPLog.INSTANCE.d(TAG, "Task " + ev.getTaskName() + " finished successfully");
			mCurrentTaskIndex++;
			mExponentialBackOff.reset();
			LPLog.INSTANCE.d(TAG, "Running next task...");

			postDelayEvent(new RunTaskEvent(mCurrentTaskIndex, mExponentialBackOff.getRetryNumber()), mExponentialBackOff.getNextBackOffMillis());
		}

		/**
		 * A connection task failed
		 * @param ev
		 */
		@Override
		public void visit(TaskFailedEvent ev) {
			super.visit(ev);

			LPLog.INSTANCE.e(TAG, ERR_000000A6, "Secondary Connection task " + ev.getType() + " failed.", ev.getException());

			// If we got error regarding version it means that the kill switch activated. Disconnect.
			if (ev.getType() == TaskType.VERSION) {
				changeState(mDisconnectingState);
				return;
			}

			// If we got error regarding CSDS it means that a value might have been updated. We disconnect and reconnect
			if (ev.getType() == TaskType.CSDS) {
                setFullConnectionRequired(mBrandId, true);
				reconnect();
				return;
			}

			// Retries
			mExponentialBackOff.calculateNextBackOffMillis();
			if(mExponentialBackOff.getNextBackOffMillis() == BackOff.STOP){
				LPLog.INSTANCE.w(TAG, "Connection task " + ev.getType() + " failed. max retries achieved. Finishing connecting flow.");
				// Since we failed the connection (for some reason) we set the flag to connect with full flow on the next time but won't reconnect
                setFullConnectionRequired(mBrandId, true);
			}
		}

		/**
		 * Run the next connection tasks
		 * @param ev
		 */
		@Override
		public void visit(RunTaskEvent ev) {
			if (mCurrentTaskIndex < mSecondaryConnectingTasks.size()) {
				BaseAmsAccountConnectionTask currentTask = mSecondaryConnectingTasks.get(mCurrentTaskIndex);
				currentTask.setBrandId(mBrandId);
				currentTask.setIsSecondaryTask(true);

				LPLog.INSTANCE.d(TAG, "Running task: " + currentTask.getClass().getSimpleName() + " Retry #" + mExponentialBackOff.getRetryNumber() + ", After delay: " + mExponentialBackOff.getNextBackOffMillis());
				currentTask.execute();

			} else {
				LPLog.INSTANCE.d(TAG, "Secondary connection flow finished successfully! :)");
			}
		}

	}

	//------------------------- BackgroundParentState -----------------------------

	/**
	 * An abstract background parent state that holds a background timeout event
	 */
	abstract class AbstractBackgroundParentState extends BaseConnectionState{

		protected Runnable mBackgroundConnectionTimeoutEvent = null;

		public AbstractBackgroundParentState(String name, String logTag) {
			super(name, logTag);
		}

		public Runnable getBackgroundConnectionTimeoutEvent() {
			return mBackgroundConnectionTimeoutEvent;
		}

	}

	/**
	 * This is the parent state for background states. It has the implementations for setting the timeout event
	 * for the background state.
	 * It also has the implementation of sending the changeChatState (BACKGROUND) to server
	 */
	class BackgroundParentState extends AbstractBackgroundParentState {


		public BackgroundParentState(String name, String logTag) {
			super(name, logTag);
		}

		/**
		 * Upon entering this state, schedule a background timeout
		 */
		@Override
		public void actionOnEntry() {
			super.actionOnEntry();
			// Set timer to disconnection
			scheduleBackgroundConnectionTimeout();

		}

		/**
		 * Upon exiting this state, cancel any pending background timeout event
		 */
		@Override
		public void actionOnExit() {
			super.actionOnExit();

			LPLog.INSTANCE.d(TAG, "actionOnExit: Canceling backgroundTimeoutEvent");

			// Remove the disconnection timer
			cancelDelayedEvent(mBackgroundConnectionTimeoutEvent);
			mBackgroundConnectionTimeoutEvent = null;
		}

		/**
		 * Update the current pending background timeout if the new given timeout is greater than the current one
		 * @param ev
		 */
		@Override
		public void visit(BackgroundEvent ev) {
			super.visit(ev);
			LPLog.INSTANCE.d(TAG, "BackgroundTimeoutEvent: Got a new background event");

			// Check if the new timeout time is later than the one that is already set. If not, we don't set the new timeout
			long newTime = System.currentTimeMillis() + mBackgroundConnectionTimerMs;
			if (newTime > mAbsoluteTimeForTimeout) {
				LPLog.INSTANCE.d(TAG, "BackgroundTimeoutEvent: new time for background is greater than the current (" + newTime + " > " + mAbsoluteTimeForTimeout + "). Cancel old and set new timeout");
				cancelDelayedEvent(mBackgroundConnectionTimeoutEvent);
				scheduleBackgroundConnectionTimeout();
			}
			else {
				LPLog.INSTANCE.d(TAG, "BackgroundTimeoutEvent: new time for background is smaller than the current (" + newTime + " < " + mAbsoluteTimeForTimeout + "). Do nothing");
			}
		}

		/**
		 * A timeout event has expired. Clear the timeout event parameter
		 * @param ev
		 */
		@Override
		public void visit(BackgroundTimeOutEvent ev) {
			super.visit(ev);
			LPLog.INSTANCE.d(TAG, "BackgroundTimeoutEvent: Timeout expired. Setting mBackgroundConnectionTimeoutEvent to null");
			mBackgroundConnectionTimeoutEvent = null;
		}

		/**
		 * This indicates that we need to send here the BACKGROUND chatState
		 * @param ev
		 */
		@Override
		public void visit(SendStateEvent ev) {
			super.visit(ev);

			LPLog.INSTANCE.d(TAG, "BackgroundParentState: actionOnEntry: Sending BACKGROUND state (if conversation active)");
			// Send BACKGROUND state
			mController.getMessagingController().changeChatState(mBrandId, mBrandId, ChatState.BACKGROUND);
		}

		/**
		 * The network was lost. Disconnect
		 * @param ev
		 */
		@Override
		public void visit(NetworkLostEvent ev) {
			super.visit(ev);
			changeState(mDisconnectingState);
		}

		/**
		 * If we got this event when we are in the background state it means that we got into a case
		 * where the network was transitioned wifi-3/4G. In this case, we didn't get the NetworkLost event
		 * but only the NetworkAvailable event. Since we're in the background it means that the socket is
		 * already disconnected and we need to move the disconnecting state
		 * @param ev
		 */
		@Override
		public void visit(NetworkAvailableEvent ev) {
			super.visit(ev);
			changeState(mDisconnectingState);
		}

        /**
         * Since we're in the background it means that the socket is
         * already disconnected and we need to move the disconnecting state
         * @param ev
         */
        @Override
        public void visit(SocketProblemEvent ev) {
            super.visit(ev);
            changeState(mDisconnectingState);
        }
		/**
		 * schedule timeout event in case we don't get response in a reasonable time.
		 */
		private void scheduleBackgroundConnectionTimeout() {

			// If the timeout we get is negative, set it to 0
			if (mBackgroundConnectionTimerMs < 0) {
				mAbsoluteTimeForTimeout = 0;
			}

			mAbsoluteTimeForTimeout = System.currentTimeMillis() + mBackgroundConnectionTimerMs;

			LPLog.INSTANCE.d(TAG, "scheduleBackgroundConnectionTimeout: Setting background connection timeout for: " + mBackgroundConnectionTimerMs);
			BackgroundTimeOutEvent timeoutEvent = new BackgroundTimeOutEvent();
			mBackgroundConnectionTimeoutEvent = postDelayEvent(timeoutEvent, mBackgroundConnectionTimerMs);
		}
	}


	/**
	 * Base state for all connected states
	 */
	abstract class ConnectedBaseState extends BaseConnectionState {

		public ConnectedBaseState(BaseConnectionState parent, String name, String logTag) {
			super(parent, name, logTag);
		}

        @Override
		public void visit(DisconnectEvent ev) {
			super.visit(ev);
			changeState(mDisconnectingState);
		}

        @Override
        public boolean isConnected() {
            return true;
        }

        @Override
		public void visit(NetworkLostEvent ev) {
			super.visit(ev);
			changeState(mDisconnectingState);
		}

		@Override
		public void visit(ShutDownEvent ev) {
			super.visit(ev);
			changeStateAndPassEvent(mDisconnectingState, ev);
		}


	}

	private void notifyShutDownCompleted(ShutDownEvent ev) {
        ev.getListener().shutDownCompleted();
        LPLog.INSTANCE.d(TAG, "ShutDown completed!");
        super.shutDown();
    }

    private void notifyError(final TaskType type, LpError lpError, String message) {
        if (mStateListener != null){
            mStateListener.notifyError(type, lpError, message);
        }
    }
}
