package com.liveperson.messaging.model;

import com.liveperson.api.response.model.ConversationINCADetails;
import com.liveperson.api.response.model.DialogData;
import com.liveperson.api.response.model.Participants;
import com.liveperson.api.response.model.Result;
import com.liveperson.api.response.types.CSAT;
import com.liveperson.api.response.types.CloseReason;
import com.liveperson.api.response.types.ConversationState;
import com.liveperson.api.response.types.DialogState;
import com.liveperson.api.response.types.TTRType;
import com.liveperson.infra.log.LPLog;
import com.liveperson.messaging.commands.tasks.FetchConversationManager;

import java.util.ArrayList;
import java.util.Arrays;

import static com.liveperson.infra.errors.ErrorCode.ERR_00000061;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000062;

/**
 * Created by Ilya Gazman on 11/11/2015.
 *
 * A bunch of properties representing conversation
 */
public class ConversationData implements Comparable<ConversationData> {
    private static final String TAG = "ConversationData";
    public String conversationId;
    public String brandId;
    public String targetId;
    public String consumerId;
    public ConversationState state;
    public long requestId;
    public Participants participants;
    public DialogData[] dialogs;
    public CSAT csat;
    public CloseReason closeReason;
    public long endTs;
    public long startTs;
	public long ttrValue; // Number of milliseconds
	public long manualTTR; // TTR timestamp (absolute time set on the agent side)
    public int unreadMessages;
    public TTRType conversationTTRType;
    public long delayTillWhen;
    public FetchConversationManager.DATA_SOURCE source;

    public ConversationData(ConversationState conversationState, Result result, String mBrandID) {
        conversationId = result.conversationId;
        brandId = mBrandID;
        targetId = mBrandID;
        consumerId = getConsumerId(result.conversationDetails.participants);
        conversationTTRType = TTRType.valueOf(result.conversationDetails.ttr != null ? result.conversationDetails.ttr.ttrType : "");
        state = conversationState;
        participants = result.conversationDetails.participants;
        requestId = -1;
        csat = result.conversationDetails.csat;
        closeReason = result.conversationDetails.closeReason;
        endTs = result.conversationDetails.endTs;
        startTs = result.conversationDetails.startTs;
		ttrValue = result.conversationDetails.ttr.value * 1000;
		manualTTR = result.conversationDetails.manualETTR;
		if (result.conversationDetails.delay != null) {
            delayTillWhen = result.conversationDetails.delay.tillWhen;
        }

        source = FetchConversationManager.DATA_SOURCE.UMS;

        dialogs = result.conversationDetails.dialogs;
        if (dialogs == null) {
            LPLog.INSTANCE.e(TAG, ERR_00000061, "The data from server did not include any dialog, injecting an inducted dialog from the conversation's data");
            dialogs = DialogData.extractDialogsData(this);
        }

        if (dialogs.length == 1 && dialogs[0].state == DialogState.OPEN && state == ConversationState.CLOSE) {
            //TODO Perry: "PATCHING ON A SERVER'S BUG"... Nizry: "Let's talk about it with the UMS again"
            // We got an open dialog while the conversation is closed :O
            // Probably due to a server's hard coded mock without a logic behind.
            dialogs = DialogData.extractDialogsData(this);
        }
    }

	public ConversationData(ConversationINCADetails conv, String mBrandID) {
        conversationId = conv.conversationId;
        brandId = mBrandID;
        targetId = mBrandID;
        consumerId = getConsumerId(conv.participants);
        conversationTTRType = TTRType.NORMAL;
        state = conv.getState();
        participants = conv.participants;
        requestId = -1;
        csat = conv.csat;
        closeReason = conv.closeReason;
        endTs = conv.endTs;
        startTs = conv.startTs;
        delayTillWhen = -1;
        unreadMessages = 0;
        source = FetchConversationManager.DATA_SOURCE.INCA;
        dialogs = conv.dialogs;
    }

    public ConversationData(String brandId, Conversation conversation) {
        this(brandId, conversation, new ArrayList<>());
    }

    public ConversationData(String brandId, Conversation conversation, ArrayList<Dialog> dialogs) {
        this.brandId = brandId;
        conversationId = conversation.getConversationId();
        targetId = conversation.getTargetId();
        conversationTTRType = TTRType.NORMAL;
        state = conversation.getState();
        requestId = -1;
        closeReason = conversation.getCloseReason();
        endTs = conversation.getEndTimestamp();
        setDialogs(dialogs);
    }

    public ConversationData() {
    }

    private String getConsumerId(Participants participants) {
        String[] consumers = participants.CONSUMER;
        if (consumers != null && consumers.length > 0) {
            return consumers[0];
        }else{
            LPLog.INSTANCE.e(TAG, ERR_00000062, "Failed to retrieve the consumer id for conversation: " + LPLog.INSTANCE.mask(participants));
            return "";
        }
    }

    public String getAssignedAgentId() {
        String agentOriginatorId = "";
        // first check if there is an "AGENT" (BOT - in Post_Survey dialog)
        // if there isn't we must still be in MAIN dialog and we return the "ASSIGNED_AGENT"
        if (participants != null &&
                participants.AGENTS.length > 0) {
            agentOriginatorId = participants.AGENTS[0];
        } else {
            if (participants != null &&
                    participants.ASSIGNED_AGENT.length > 0) {
                agentOriginatorId = participants.ASSIGNED_AGENT[0];
            }
        }

        return agentOriginatorId;
    }

    @Override
    public int compareTo(ConversationData another) {
        long timeDiff = startTs - another.startTs;
        if (timeDiff < 0 ) return 1;
        if (timeDiff > 0 ) return -1;
        return 0;
    }

    @Override
    public String toString() {
        return "{ data: {conversationId: " + conversationId + ", state: " + state + ", dialogs: " + Arrays.toString(dialogs) + "} }";
    }

    public void setDialogs(ArrayList<Dialog> dialogs) {
        if (dialogs != null && dialogs.size() > 0) {
            DialogData[] dialogsData = new DialogData[dialogs.size()];
            for (int i = 0; i < dialogsData.length; i++) {
                dialogsData[i] = new DialogData(dialogs.get(i));
            }

            this.dialogs = dialogsData;
        }
    }
}
