package com.liveperson.messaging;

import android.app.Notification;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.widget.Toast;

import com.liveperson.api.LivePersonCallback;
import com.liveperson.api.request.GeneratedUploadTokenField;
import com.liveperson.api.response.model.DeliveryStatusUpdateInfo;
import com.liveperson.api.response.types.ConversationState;
import com.liveperson.api.response.types.DeliveryStatus;
import com.liveperson.api.response.types.DialogState;
import com.liveperson.api.response.types.DialogType;
import com.liveperson.api.response.types.TTRType;
import com.liveperson.infra.Clearable;
import com.liveperson.infra.ConversationViewParams;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.Infra;
import com.liveperson.infra.LPAuthenticationParams;
import com.liveperson.infra.callbacks.InitLivePersonCallBack;
import com.liveperson.infra.callbacks.LogoutLivePersonCallBack;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.database.DataBaseCommand;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.messaging.R;
import com.liveperson.infra.model.LPWelcomeMessage;
import com.liveperson.infra.model.types.ChatState;
import com.liveperson.infra.network.socket.SocketManager;
import com.liveperson.infra.sdkstatemachine.logout.PreLogoutCompletionListener;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDownAsync;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDownCompletionListener;
import com.liveperson.infra.statemachine.InitProcess;
import com.liveperson.infra.statemachine.LogoutProcess;
import com.liveperson.infra.statemachine.ShutDownProcess;
import com.liveperson.infra.utils.FileUtils;
import com.liveperson.infra.utils.LPAudioUtils;
import com.liveperson.infra.utils.LocalBroadcast;
import com.liveperson.infra.utils.MaskedMessage;
import com.liveperson.infra.utils.MessageValidator;
import com.liveperson.messaging.background.BackgroundActionsService;
import com.liveperson.messaging.background.FileSharingManager;
import com.liveperson.messaging.background.filesharing.FileExtensionTypes;
import com.liveperson.messaging.background.filesharing.FileSharingType;
import com.liveperson.messaging.commands.ChangeChatStateCommand;
import com.liveperson.messaging.commands.ChangeConversationTTRCommand;
import com.liveperson.messaging.commands.CloseDialogCommand;
import com.liveperson.messaging.commands.DeliveryStatusUpdateCommand;
import com.liveperson.messaging.commands.GetUnreadMessagesCountCommand;
import com.liveperson.messaging.commands.RegisterPusherCommand;
import com.liveperson.messaging.commands.ResendMessageCommand;
import com.liveperson.messaging.commands.ResendURLMessageCommand;
import com.liveperson.messaging.commands.ResolveConversationCommand;
import com.liveperson.messaging.commands.SendCsatCommand;
import com.liveperson.messaging.commands.SendFormSubmissionMessageCommand;
import com.liveperson.messaging.commands.SendGenerateUploadTokenCommand;
import com.liveperson.messaging.commands.SendMessageCommand;
import com.liveperson.messaging.commands.SendMessageWithURLCommand;
import com.liveperson.messaging.commands.SendSetUserProfileCommand;
import com.liveperson.messaging.commands.UnregisterPusherCommand;
import com.liveperson.messaging.commands.GetCombinedUnreadMessagesCountCommand;
import com.liveperson.messaging.commands.tasks.MessagingEventSubscriptionManager;
import com.liveperson.messaging.controller.AccountsController;
import com.liveperson.messaging.controller.AmsReadController;
import com.liveperson.messaging.controller.ClientProperties;
import com.liveperson.messaging.controller.ConnectionsController;
import com.liveperson.messaging.model.AgentData;
import com.liveperson.messaging.model.AmsConversations;
import com.liveperson.messaging.model.AmsDialogs;
import com.liveperson.messaging.model.AmsFiles;
import com.liveperson.messaging.model.AmsMessages;
import com.liveperson.messaging.model.AmsUsers;
import com.liveperson.messaging.model.Conversation;
import com.liveperson.messaging.model.ConversationUtils;
import com.liveperson.messaging.model.Dialog;
import com.liveperson.messaging.model.Form;
import com.liveperson.messaging.model.MessagingChatMessage;
import com.liveperson.messaging.model.MessagingUserProfile;
import com.liveperson.messaging.model.QuickRepliesMessageHolder;
import com.liveperson.messaging.model.UserProfileBundle;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;

/**
 * Created by Ilya Gazman on 11/10/2015.
 * <p/>
 * A singleton holder, it helps with intelligence
 */
public class Messaging implements IMessaging, ShutDownAsync, Clearable {

	private static final String TAG = Messaging.class.getSimpleName();
	public static final int NO_FILE_ROW_ID = -1;
	public static final String SUBMISSION_ID = "submissionId";
	public static final String INVITATION_ID = "invitationId";
	public static final String FORM_TITLE = "formTitle";

	public ConnectionsController mConnectionController;
	public AccountsController mAccountsController;

	public AmsMessages amsMessages;
	public AmsConversations amsConversations;
	public AmsDialogs amsDialogs;
	public AmsUsers amsUsers;
	public AmsFiles amsFiles;

	// Flag indicating if still busy fetching history.
	private boolean isStillBusyFetching;

	// Indicate if we're using a service to upload/download images
	private boolean mUploadUsingService;

	ClientProperties mClientProperties;

	// This controller does not have access from outside. It is used to listen on connection and foreground
	// changes and act on them
	public AmsReadController amsReadController;

	public LivePersonEventsProxy mEventsProxy;

	private MessageValidator mMessageValidator;

	private FileSharingManager mFileSharingManager;
	/**
	 * The PendingIntent to use on the foreground service notification
	 */
	private PendingIntent mImageServicePendingIntent = null;
	/**
	 * The notification builder for the image upload/download foreground service
	 */
	private Notification.Builder mImageForegroundServiceUploadNotificationBuilder = null;
	private Notification.Builder mImageForegroundServiceDownloadNotificationBuilder = null;

	/**
	 * Indication whether the structured content feature is on in branding
	 */
	private boolean mEnableStructuredContent;

	private MessagingEventSubscriptionManager mMessagingEventSubscriptionManager;

	private ConversationViewParams mConversationViewParams;

	private LPAudioUtils mAudioUtils;

	// Max number of files to keep in storage
	private int mMaxNumberOfStoredImageFiles;
	private int mMaxNumberOfStoredVoiceFiles;
	private int mMaxNumberofStoredDocumentFiles;
	private Context mContext;


	public Messaging() {
		mEventsProxy = new LivePersonEventsProxy();
	}

	private void initMembers(Context context) {
		mConnectionController = new ConnectionsController(this);
		mAccountsController = new AccountsController(mClientProperties);
		amsMessages = new AmsMessages(this);
		amsConversations = new AmsConversations(this);
		amsDialogs = new AmsDialogs(this);
		amsUsers = new AmsUsers();
		amsFiles = new AmsFiles();
		amsReadController = new AmsReadController(this);
		mUploadUsingService = Configuration.getBoolean(R.bool.upload_photo_using_service);
		mFileSharingManager = new FileSharingManager(this, context);
		mEnableStructuredContent = Configuration.getBoolean(R.bool.enable_structured_content);
		mMessagingEventSubscriptionManager = new MessagingEventSubscriptionManager();
		// Get limit from resources
		mMaxNumberOfStoredImageFiles = Configuration.getInteger(R.integer.max_number_stored_images);
		mMaxNumberOfStoredVoiceFiles = Configuration.getInteger(R.integer.max_number_stored_voice_files);
        mMaxNumberofStoredDocumentFiles = Configuration.getInteger(R.integer.max_number_stored_documents);
		mAudioUtils = new LPAudioUtils();
	}

	/**
	 * Init Messaging
	 * @param context  - app context
	 * @param initData - the sdk version.
	 * @param callBack - InitLivePersonCallBack callback that will notify if init succeeded or failed.
	 */
	@Override
	public void init(final Context context, final MessagingInitData initData, final InitLivePersonCallBack callBack) {
		init(context, initData, new InitProcess() {
			@Override
			public void init() {
				initMessaging(context, initData);
			}

			@Override
			public InitLivePersonCallBack getInitCallback() {
				return callBack;
			}
		});
	}

	/**
	 * Init Messaging under upper module (MessagingUI) with it's own InitProcess.
	 * If Messaging is being used without upper module of UI, use {@link #init(Context, MessagingInitData, InitLivePersonCallBack)}
	 * @param context     - app context
	 * @param initData    - the sdk version and app id.
	 * @param initProcess - init process of upper module (MessagingUI)
	 */
	@Override
	public void init(final Context context, final MessagingInitData initData, final InitProcess initProcess) {

		Infra.instance.init(context, initData, new InitProcess() {
			@Override
			public InitLivePersonCallBack getInitCallback() {
				return initProcess.getInitCallback();
			}

			@Override
			public void init() {
				initMessaging(context, initData);
				initProcess.init();
			}

		});
	}


	MessageValidator getMessageValidator(String brandId) {
		if (mMessageValidator == null) {
			//todo maayan remove this
			if (mAccountsController.getAccount(brandId) == null){
				return null;
			}
			boolean isAuthenticated = mAccountsController.getAccount(brandId).isAuthenticated();
			mMessageValidator = new MessageValidator(mContext, isAuthenticated);
		}
		return mMessageValidator;
	}

	private void initMessaging(Context context, MessagingInitData initData) {
		LPMobileLog.d(TAG, "Initializing...");

		updateClientProperties(initData);

		// Get message masking information
		initMembers(context);
		mConnectionController.initConnectionReceiver();
		bootstrapRegistration();
		setContext(Infra.instance.getApplicationContext());
	}

	/**
	 * saving important data of the device, sdk and the host app.
	 * @param initData
	 */
	private void updateClientProperties(MessagingInitData initData) {
		if (initData != null) {
			//any new init data will overwrite the old client properties.
			mClientProperties = new ClientProperties(initData.getAppId(), initData.getSdkVersion());
		} else if (mClientProperties == null) {
			//in case init data is null, we need to restore client properties from shared preferences.
			mClientProperties = new ClientProperties();
		}
	}

//	void restart() {
//		for (String brand : ForegroundService.getInstance().getForegroundBrandId()) {
//			LPMobileLog.i(TAG, "Restarting connecting to brand " + brand);
//			mConnectionController.connect(brand);
//		}
//	}

	public void updateWelcomeMessage(String brandId) {
		if (isConversationEmptyOrClose(brandId)) {
			LPWelcomeMessage welcomeMessage = mConversationViewParams.getLpWelcomeMessage();
			ConversationUtils conversationUtils = new ConversationUtils(MessagingFactory.getInstance().getController());
			conversationUtils.updateWelcomeMessage(brandId, welcomeMessage);
			amsMessages.resetQuickRepliesMessageHolder();
			QuickRepliesMessageHolder.updateQuickReplies(brandId, welcomeMessage.getQuickReplies(false));
		}
	}

	public boolean isConversationEmptyOrClose(String brandId) {
		Conversation conversation = amsConversations.getConversationFromTargetIdMap(brandId);
		return  (conversation == null || conversation.getState() == ConversationState.CLOSE);
	}

	/**
	 * Maps response handlers for server responses
	 */
	protected void bootstrapRegistration() {
		// Register a handler for content event notifications
		SocketManager.getInstance().putGeneralHandlerMap(new GeneralMessagingResponseHandler(this));
	}

	/**
	 * Start connecting to the AMS server
	 */
	@Override
	public void connect(String brandId, LPAuthenticationParams lpAuthenticationParams, @Nullable ConversationViewParams conversationViewParams, boolean connectInBackground, boolean clearToken) {
		initBrand(brandId, lpAuthenticationParams, conversationViewParams);
		LPMobileLog.d(TAG, "Connecting to brand " + brandId);
		mConnectionController.connect(brandId, connectInBackground, clearToken);
	}
	/**
	 * Start connecting to the AMS server
	 */
	@Override
	public void connect(String brandId, LPAuthenticationParams lpAuthenticationParams, @Nullable ConversationViewParams conversationViewParams) {
		connect(brandId, lpAuthenticationParams, conversationViewParams, false, false);
	}

	@Override
	public void reconnect(String brandId, LPAuthenticationParams lpAuthenticationParams) {
		boolean isTokenExpired = mAccountsController.isTokenExpired(brandId);
		// Replace auth key. We first remove the current token from preferences so the IDP will use the new key
		LPMobileLog.d(TAG, "reconnect: set a new authentication key for brand with lpAuthenticationParams of type " + lpAuthenticationParams.getAuthType());
		// Connect without view params and clear the LpToken
		if (isTokenExpired) {
            LPMobileLog.d(TAG,"reconnect called clearing Token as TokenExpired");
                    connect(brandId, lpAuthenticationParams, null, false, true);
		}

	}

	/**
	 * Disconnect the socket for the brand
	 *
	 * @param brandId
	 */
	@Override
	public void disconnect(String brandId) {
		LPMobileLog.d(TAG, "Disconnecting from brand " + brandId);
		mConnectionController.disconnect(brandId);
	}

	/**
	 * Indicate move to background
	 * @param brandId
	 * @param timeout the time to leave the socket open
	 */
	@Override
	public void moveToBackground(String brandId, long timeout) {

		// Stop all playing voice
		MessagingFactory.getInstance().getController().getAudioUtils().getPlayingAudioManager().stopAllCurrentlyPlaying();

		mConnectionController.moveToBackground(brandId, timeout);
	}

	/**
	 * Indicate move to foreground
	 * @param brandId
	 * @param lpAuthenticationParams
	 */
	@Override
	public void moveToForeground(String brandId, LPAuthenticationParams lpAuthenticationParams, @Nullable ConversationViewParams conversationViewParams) {
		LPMobileLog.d(TAG, "moveToForeground: brandId = " + brandId);
		initBrand(brandId, lpAuthenticationParams, conversationViewParams);
		mConnectionController.moveToForeground(brandId);
	}

	/**
	 * Indicate the upload image service has started
	 * @param brandId
	 */
	@Override
	public void serviceStarted(String brandId) {
		LPMobileLog.d(TAG, "serviceStarted: brandId = " + brandId);
		mConnectionController.serviceStarted(brandId);
	}

	/**
	 * Indicate service was stopped
	 * @param brandId
	 */
	@Override
	public void serviceStopped(String brandId) {
		LPMobileLog.d(TAG, "serviceStopped: brandId = " + brandId);
		mConnectionController.serviceStopped(brandId);
	}



	/**
	 * A root method for init the Messaging
	 *
	 * @param brandId
	 */
	protected void initBrand(String brandId, LPAuthenticationParams lpAuthenticationParams, @Nullable ConversationViewParams conversationViewParams) {
		LPMobileLog.d(TAG, "Init brand " + brandId);
		mAccountsController.addNewAccount(brandId);
		mAccountsController.setLPAuthenticationParams(brandId, lpAuthenticationParams);
		if (conversationViewParams != null) {
			setConversationViewParams(conversationViewParams);
		}
		mConnectionController.addNewConnection(brandId);
	}

	/**
	 * Message timer expired, will try reconnect is necessary
	 *
	 * @param brandId
	 */
	public void onMessageTimeout(String brandId) {
		//reconnect automatically if we are in foreground.
		mConnectionController.connect(brandId);
	}

	/**
	 * Send message that contains at least one URL to the AMS server
	 *
	 * @param message
	 */
	@Override
	public void sendMessageWithURL(String targetId, String brandId, String message, String urlToParse, String title, String description, String imageURL, String siteName) {
		// masking message if needed
		MaskedMessage maskedMessage = getMessageValidator(brandId).maskMessage(message, true);
		new SendMessageWithURLCommand(this, targetId, brandId, maskedMessage, urlToParse, title, description, imageURL, siteName).execute();
	}

	/**
	 * Send message to the AMS server
	 *
	 * @param message
	 * @param info a {@link DeliveryStatusUpdateInfo} object with metadata to be sent
	 */
	@Override
	public void sendMessage(String targetId, String brandId, String message, @Nullable DeliveryStatusUpdateInfo info) {
		// masking message if needed
		MaskedMessage maskedMessage = getMessageValidator(brandId).maskMessage(message, true);
		new SendMessageCommand(this, targetId, brandId, maskedMessage, info).execute();
	}

	@Override
	public MaskedMessage getMaskedMessage(String brandId, String message) {
		return getMessageValidator(brandId).maskMessage(message, true);
	}

	/**
	 * Resend a message
	 *
	 * @param eventId
	 * @param dialogId
	 * @param messageType
	 */
	public int resendMessage(final String eventId, String dialogId, final MessagingChatMessage.MessageType messageType) {
		return resendMessage(eventId, dialogId, NO_FILE_ROW_ID, messageType);
	}
	/**
	 * Resend a message
	 *
	 * @param eventId
	 * @param dialogId
	 */
	@Override
	public int resendMessage(final String eventId, final String dialogId, final long fileRowId, final MessagingChatMessage.MessageType messageType) {
		// Get the brandId
		if (isDialogClosed(dialogId)) {
			LPMobileLog.i(TAG, "Resend message- conversation does not exists or closed.");
			return R.string.lp_resend_failed_conversation_closed;
		}

		if (MessagingChatMessage.MessageType.isConsumerMaskedMessage(messageType)) {
			LPMobileLog.i(TAG, "Resend message- message is masked, resend is not available.");
			return R.string.lp_resend_failed_masked_message;
		}

		amsMessages.getMessageByEventId(eventId).setPostQueryOnBackground(new DataBaseCommand.QueryCallback<MessagingChatMessage>() {

			@Override
			public void onResult(MessagingChatMessage data) {
				//if conversation is null, means it was on temporary conversation id - we will resend it..
				Dialog dialog = amsDialogs.queryDialogById(dialogId).executeSynchronously();
				if (data != null) {
					String message = data.getMessage();
					if (messageType == MessagingChatMessage.MessageType.CONSUMER_IMAGE) {
						reSendFileMessage(FileSharingType.JPG, dialog.getBrandId(), dialog.getTargetId(), eventId, message, data.getTimeStamp(), fileRowId);
					} else if (messageType == MessagingChatMessage.MessageType.CONSUMER_VOICE) {
						reSendFileMessage(FileSharingType.VOICE, dialog.getBrandId(), dialog.getTargetId(), eventId, message, data.getTimeStamp(), fileRowId);
					} else if (messageType == MessagingChatMessage.MessageType.CONSUMER_URL) {
						MaskedMessage maskedMessage = getMessageValidator(dialog.getBrandId()).maskMessage(message, true);
						new ResendURLMessageCommand(Messaging.this, eventId, dialog.getTargetId(), dialog.getBrandId(), maskedMessage).execute();
					} else {
						MaskedMessage maskedMessage = getMessageValidator(dialog.getBrandId()).maskMessage(message, true);
						new ResendMessageCommand(Messaging.this, eventId, dialog.getTargetId(), dialog.getBrandId(), maskedMessage).execute();
					}
				}
			}
		}).execute();
		return -1;
	}

	public boolean isDialogClosed(String dialogId) {
		boolean isDialogClosed = false;
		Dialog dialog = amsDialogs.getDialogById(dialogId);

		if (dialog == null || dialog.getState() == DialogState.CLOSE) {
			LPMobileLog.d(TAG, "isDialogClosed - dialog (dialogId = " + dialogId + ") does not exists or closed. (dialog = " + (dialog == null ? "null" : dialog.getState()) + ")");
			isDialogClosed = true;
		}

		return isDialogClosed;
	}

	private void reSendFileMessage(FileSharingType fileSharingType, String brandId, String targetId, String eventId, String message, long originalMessageTime, long fileRowId) {
		Context applicationContext = Infra.instance.getApplicationContext();
		if (!mUploadUsingService) {
			LPMobileLog.d(TAG, "reSendImageMessage: re-uploading photo without a service");

			mFileSharingManager.reUploadFile(fileSharingType, brandId, targetId, message, eventId, originalMessageTime,
					fileRowId, new FileSharingManager.FileUploadProgressListener() {
						@Override
						public void onDoneUpload() {
							LPMobileLog.d(TAG, "onDoneUpload!");
						}
						@Override
						public void onFailedUpload(Throwable exception) {
							LPMobileLog.d(TAG, "onFailedUpload! " + exception.getMessage());
						}
					});

		} else {
			LPMobileLog.d(TAG, "reSendImageMessage: re-uploading photo using a service");

			Intent intent = new Intent(applicationContext, BackgroundActionsService.class);
			intent.putExtra(BackgroundActionsService.EXTRA_ACTION_TYPE, BackgroundActionsService.EXTRA_TYPE_ACTION_REUPLOAD);
			intent.putExtra(BackgroundActionsService.EXTRA_FILE_TYPE, fileSharingType.ordinal());
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_BRAND_ID, brandId);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_TARGET_ID, targetId);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_MESSAGE, message);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_EVENT_ID, eventId);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_ORIGINAL_MESSAGE_TIME, originalMessageTime);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_FILE_ROW_ID, fileRowId);
			applicationContext.startService(intent);
		}

	}


	@Override
	public void sendFileMessage(FileSharingType fileSharingType, String brandId, String targetId, String imageUriString, String message, boolean imageFromCamera) {
		Context applicationContext = Infra.instance.getApplicationContext();
		if (mUploadUsingService) {

			LPMobileLog.d(TAG, "startUploadPhoto: uploading photo using a service");

			Intent intent = new Intent(applicationContext, BackgroundActionsService.class);
			intent.putExtra(BackgroundActionsService.EXTRA_ACTION_TYPE, BackgroundActionsService.EXTRA_ACTION_TYPE_UPLOAD);
			intent.putExtra(BackgroundActionsService.EXTRA_FILE_TYPE, fileSharingType.ordinal());
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_BRAND_ID, brandId);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_TARGET_ID, targetId);
                /*  for(Uri imageUri : mArrayUri){
                intent.putExtra(BackgroundActionsService.SERVICE_EXTRA_FILE_URI, imageUri.toString());
                getActivity().startService(intent);
                }*/
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_FILE_URI, imageUriString);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_FILE_CAPTION, message);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_IMAGE_FROM_CAMERA, imageFromCamera);
			applicationContext.startService(intent);
		} else {
			LPMobileLog.d(TAG, "startUploadPhoto: uploading photo without a service");

			mFileSharingManager.uploadFile(fileSharingType, brandId, targetId, imageUriString, message, imageFromCamera, new FileSharingManager.FileUploadProgressListener() {
				@Override
				public void onDoneUpload() {
					LPMobileLog.d(TAG, "onDoneUpload!");
				}

				@Override
				public void onFailedUpload(Throwable exception) {
					LPMobileLog.d(TAG, "onFailedUpload! " + exception.getMessage());
					Infra.instance.getApplicationHandler().post( () ->
						Toast.makeText(Infra.instance.getApplicationContext(), R.string.lp_failed_upload_toast_message, Toast.LENGTH_LONG).show());
				}
			});
		}

	}

	@Override
	public void downloadFile(FileSharingType fileSharingType, String brandId, String targetId, String imageSwiftPath, long messageRowId, long fileRowId, String conversationId) {
		Context applicationContext = Infra.instance.getApplicationContext();
		if (mUploadUsingService) {

			LPMobileLog.d(TAG, "startUploadPhoto: uploading photo using a service");

			Intent intent = new Intent(applicationContext, BackgroundActionsService.class);
			intent.putExtra(BackgroundActionsService.EXTRA_ACTION_TYPE, BackgroundActionsService.EXTRA_TYPE_ACTION_DOWNLOAD);
			intent.putExtra(BackgroundActionsService.EXTRA_FILE_TYPE, fileSharingType.ordinal());
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_BRAND_ID, brandId);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_TARGET_ID, targetId);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_FILE_URI, imageSwiftPath);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_FILE_ROW_ID, fileRowId);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_MESSAGE_ROW_ID, messageRowId);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_CONVERSATION_ID, conversationId);
			applicationContext.startService(intent);

		} else {

			mFileSharingManager.downloadFile(fileSharingType, brandId, targetId, imageSwiftPath, messageRowId, fileRowId, conversationId, new FileSharingManager.FileDownloadProgressListener() {

				@Override
				public void onDoneDownload() {
					LPMobileLog.d(TAG, "onDoneDownload!");
				}

				@Override
				public void onFailedDownload(Throwable exception) {
					LPMobileLog.d(TAG, "onFailedDownload! " + exception.getMessage());
					Infra.instance.getApplicationHandler().post(new Runnable() {
						@Override
						public void run() {
							Toast.makeText(Infra.instance.getApplicationContext(), R.string.lp_failed_download_toast_message, Toast.LENGTH_LONG).show();
						}
					});
				}
			});
		}
	}


	/**
	 * A wrapper around SendGenerateUploadTokenCommand
	 *
	 */
	@Override
	public void generateUploadToken(String formId, String brandId, final String invitationId) {
		Dialog activeDialog = amsDialogs.getActiveDialog();
		if (activeDialog == null) {
			LPMobileLog.e(TAG, "Failed to generate upload token, there's no active dialog!");
			MessagingFactory.getInstance().getController().amsMessages.mFormsManager.getForm(invitationId).setFormStatus(Form.FormStatus.ERROR);
			MessagingFactory.getInstance().getController().amsMessages
					.setDeliveryStatusUpdateCommand(MessagingFactory.getInstance()
					.getController().amsMessages
					.mFormsManager.getForm(invitationId),
					DeliveryStatus.ERROR);
		} else {
			new SendGenerateUploadTokenCommand(mAccountsController.getConnectionUrl(brandId),
					formId,
					activeDialog.getDialogId(),
					invitationId,
					new ICallback() {
						@Override
						public void onSuccess(Object value) {

							Messaging.this.amsMessages.mFormsManager.updateForm(invitationId,
									((GeneratedUploadTokenField.Response.Body) value).readOtk,
									((GeneratedUploadTokenField.Response.Body) value).writeOtk);
							Form currentForm = Messaging.this.amsMessages.mFormsManager.getForm(invitationId);

							if (currentForm == null) {
								LPMobileLog.d(TAG, "no form was found ");
								return;
							}
							String url = currentForm.getOpenFormURL();
							LPMobileLog.d(TAG, "url = " + url);
							Bundle bundle = new Bundle();
							bundle.putString("url", url);
							bundle.putString("invitation_id", invitationId);
							bundle.putString("form_title", currentForm.getFormTitle());
							LPMobileLog.d(TAG, "Sending PCI update invitationId = " + invitationId + " form title : " + currentForm.getFormTitle());
							LocalBroadcast.sendBroadcast(AmsConversations.BROADCAST_UPDATE_FORM_URL, bundle);
						}

						@Override
						public void onError(Throwable exception) {
							LPMobileLog.w(TAG, "an error during generating OTK");
							MessagingFactory.getInstance().getController().amsMessages.mFormsManager.getForm(invitationId).setFormStatus(Form.FormStatus.ERROR);
							MessagingFactory.getInstance().getController().amsMessages.setDeliveryStatusUpdateCommand(
									MessagingFactory.getInstance().getController().amsMessages.mFormsManager.getForm(invitationId), DeliveryStatus.ERROR
							);
						}
					}
			).execute();
		}
	}

	/**
	 * Send message after submitting a form to the AMS server
	 *
	 * @param invitationId
	 */
	@Override
	public void sendFormSubmissionMessageCommand(final String invitationId) {
		Form currentForm = Messaging.this.amsMessages.mFormsManager.getForm(invitationId);
		try {
			JSONObject jo = new JSONObject();
			jo.put(SUBMISSION_ID, currentForm.getSubmissionId());
			jo.put(INVITATION_ID, currentForm.getInvitationId());
			amsMessages.mFormsManager.updateForm(currentForm.getInvitationId(), currentForm.getSubmissionId());

			//todo maayan test it out
			Dialog dialog = amsDialogs.getDialogById(currentForm.getDialogId());
			MaskedMessage maskedMessage = getMessageValidator(dialog.getBrandId()).maskMessage(jo.toString(), false);
			if (maskedMessage == null) {
				currentForm.setFormStatus(Form.FormStatus.ERROR);
				updateMessage(currentForm.getInvitationId(), currentForm.getDialogId(), MessagingChatMessage.MessageType.AGENT_FORM , MessagingChatMessage.MessageState.ERROR);
				return;
			}
			maskedMessage.setServerMessage(jo.toString());

			jo.put(FORM_TITLE, currentForm.getFormTitle());
			maskedMessage.setDbMessage(jo.toString());
			new SendFormSubmissionMessageCommand(currentForm, maskedMessage, Messaging.this).execute();

			updateMessage(currentForm.getInvitationId(), currentForm.getDialogId(), MessagingChatMessage.MessageType.AGENT_FORM , MessagingChatMessage.MessageState.SUBMITTED);
		} catch (JSONException e) {
			e.printStackTrace();
		}
	}

	@Override
	public void updateMessage(final String invitationId, final String dialogId,
							  final MessagingChatMessage.MessageType messageType, final MessagingChatMessage.MessageState messageState) {
		final Form currentForm = Messaging.this.amsMessages.mFormsManager.getForm(invitationId);

		if (currentForm == null){
			LPMobileLog.i(TAG, "pci update message- form does not exists or closed.");
			return;
		}
		Dialog dialog = amsDialogs.getDialogById(dialogId);
		if (dialog == null || dialog.getState() == DialogState.CLOSE) {
			LPMobileLog.i(TAG, "pci update message- dialog does not exists or closed.");
			return;
		}
		ArrayList<String> l = new ArrayList<>();
		l.add(currentForm.getEventId());

		LPMobileLog.i(TAG, "pci update message- with eventID "+ currentForm.getEventId() + " to state: " +messageState);


		amsMessages.updateMessagesState(l, currentForm.getSiteId(), currentForm.getDialogId(), messageState);
	}

	@Override
	public void removeMultipleOlderImages(Context mContext, String targetId, String brandId) {
		LPMobileLog.d(TAG, "removeMultipleOlderFiles without service");

		// Remove redundant image files
		mFileSharingManager.removeMultipleOlderFiles(targetId, mMaxNumberOfStoredImageFiles, FileExtensionTypes.getImageExtensionsAsSqlString());

		// Remove redundant voice files
		mFileSharingManager.removeMultipleOlderFiles(targetId, mMaxNumberOfStoredVoiceFiles, FileExtensionTypes.getVoiceExtensionsAsSqlString());

		// Remove redundant Document files
		mFileSharingManager.removeMultipleOlderFiles(targetId, mMaxNumberofStoredDocumentFiles, FileExtensionTypes.getDocumentExtensionsAsSqlString());

	}

	@Override
	public void registerPusher(String brandId, String appId, String token, LPAuthenticationParams authenticationParams,final ICallback<Void, Exception> registrationCompletedCallback) {

		new RegisterPusherCommand(this, brandId, appId, token,authenticationParams, registrationCompletedCallback).execute();
	}

	@Override
	public void updateTokenInBackground(String brandId, LPAuthenticationParams authenticationParams) {

		LPMobileLog.d(TAG, "updateTokenInBackground: Clearing token from account");
		//mAccountsController.setToken(brandId, null);
		connect(brandId, authenticationParams, null, true, true);
	}

	@Override
	public void unregisterPusher(String brandId, String appId, ICallback<Void, Exception> unregisteringCompletedCallback, boolean immediately) {
		new UnregisterPusherCommand(mAccountsController, amsUsers, brandId, appId, unregisteringCompletedCallback, immediately).execute();
	}

	@Override
	public void getNumUnreadMessages(String brandId, String appId, final ICallback<Integer, Exception> callback) {
		new GetUnreadMessagesCountCommand(this, brandId, appId, new ICallback<Integer, Exception>() {
			@Override
			public void onSuccess(Integer value) {
				callback.onSuccess(value);
			}

			@Override
			public void onError(Exception exception) {
				callback.onError(exception);
			}
		}).execute();
	}

	@Override
	public void getUnreadMessagesCount(String brandId, String appId, final ICallback<Integer, Exception> callback) {
		new GetCombinedUnreadMessagesCountCommand(this, brandId, appId, new ICallback<Integer, Exception>() {
			@Override
			public void onSuccess(Integer value) {
				callback.onSuccess(value);
			}

			@Override
			public void onError(Exception exception) {
				callback.onError(exception);
			}
		}).execute();
	}

	/**
	 * A wrapper around ChangeChatStateCommand
	 *
	 * @param state
	 */
	@Override
	public ActionFailureReason changeChatState(String targetId, String brandId, ChatState state) {
		ActionFailureReason failedReason = getConversationActionFailedReason(targetId, brandId);
		if (failedReason != null) {
			return failedReason;
		}
		new ChangeChatStateCommand(amsDialogs, targetId, mAccountsController.getConnectionUrl(brandId), state).execute();
		return null;
	}

	/**
	 * A wrapper around ResolveConversationCommand
	 */
	@Override
	public ActionFailureReason resolveConversation(String targetId, String brandId) {
		ActionFailureReason failedReason = getConversationActionFailedReason(targetId, brandId);
		if (failedReason != null) {
			return failedReason;
		}

		new ResolveConversationCommand(amsConversations, targetId, mAccountsController.getConnectionUrl(brandId)).execute();
		return null;
	}

	public ActionFailureReason closeCurrentDialog() {
		ActionFailureReason actionFailureReason;
		Dialog activeDialog = amsDialogs.getActiveDialog();
		if (activeDialog == null) {
			LPMobileLog.e(TAG, "There's no active dialog. Aborting from closing dialog");
			actionFailureReason = ActionFailureReason.NOT_ACTIVE;
		} else {
			actionFailureReason = closeDialog(activeDialog.getBrandId());
		}

		return actionFailureReason;
	}

	/**
	 *
	 * Wraps the CloseDialogCommand to execute this operation.
	 */
	@Override
	public ActionFailureReason closeDialog(final String brandId) {
		ActionFailureReason actionFailureReason = getConversationActionFailedReason(brandId, brandId);
		if (actionFailureReason != null) {
			return actionFailureReason;
		} // Socket is open and there's an open conversation....

		final Dialog activeDialog = amsDialogs.getActiveDialog();
		boolean isDialogOpen = activeDialog != null && activeDialog.isOpen();
		if (isDialogOpen && AmsDialogs.isUmsSupportingDialogs()) {
			final CloseDialogCommand closeDialogCommand = new CloseDialogCommand(amsDialogs, activeDialog.getDialogId(), mAccountsController.getConnectionUrl(brandId));
			closeDialogCommand.setCallback(new ICallback<Integer, Exception>() {
				@Override
				public void onSuccess(Integer value) {
					// Cleanup
					closeDialogCommand.setCallback(null);
				}

				@Override
				public void onError(Exception exception) {
					if (Integer.parseInt(exception.getMessage(), -1) == 400) {
						LPMobileLog.e(TAG, LPMobileLog.FlowTags.DIALOGS, "Failed to close dialog due to an error (with code 400), closing the whole conversation.");
						new ResolveConversationCommand(amsConversations, brandId, mAccountsController.getConnectionUrl(brandId)).execute();
					}
					// Cleanup
					closeDialogCommand.setCallback(null);
				}
			});

			closeDialogCommand.execute();
		} else {
			// There's no open dialog (but the conversation is open) / The current UMS doesn't support multi dialogs
			new ResolveConversationCommand(amsConversations, brandId, mAccountsController.getConnectionUrl(brandId)).execute();
		}

		return null;
	}

	/**
	 * A wrapper around markConversationAsUrgent, it prepopulate it with TTRType.URGENT
	 */
	@Override
	public ActionFailureReason markConversationAsUrgent(String targetId, String brandId) {
		ActionFailureReason failedReason = getConversationActionFailedReason(targetId, brandId);
		if (failedReason != null) {
			return failedReason;
		}

		Dialog activeDialog = amsDialogs.getActiveDialog();
		ActionFailureReason changeTTRFailedReason = getDialogChangeTTRActionFailedReason(activeDialog);
		if (changeTTRFailedReason != null){
			return changeTTRFailedReason;
		}

		new ChangeConversationTTRCommand(amsConversations, targetId, mAccountsController.getConnectionUrl(brandId), TTRType.URGENT).execute();

		return null;
	}

    /**
     * A wrapper around markConversationAsUrgent, it prepopulate it with TTRType.NORMAL
     */
    @Override
    public ActionFailureReason markConversationAsNormal(String targetId, String brandId) {
        ActionFailureReason failedReason = getConversationActionFailedReason(targetId, brandId);
        if (failedReason != null) {
            return failedReason;
        }

		Dialog activeDialog = amsDialogs.getActiveDialog();
		ActionFailureReason changeTTRFailedReason = getDialogChangeTTRActionFailedReason(activeDialog);
        if (changeTTRFailedReason != null) {
            return changeTTRFailedReason;
        }

        new ChangeConversationTTRCommand(amsConversations, targetId, mAccountsController.getConnectionUrl(brandId), TTRType.NORMAL).execute();

        return null;
    }

	@Nullable
	private ActionFailureReason getConversationActionFailedReason(String targetId, String brandId) {
		if (!mConnectionController.isSocketReady(brandId)) {
			LPMobileLog.d(TAG, "Socket is not open");
			return ActionFailureReason.NO_NETWORK;
		}
		if (!amsConversations.isConversationActive(targetId)) {
			LPMobileLog.d(TAG, "There's no active dialog");
			return ActionFailureReason.NOT_ACTIVE;
		}

		return null;
	}

	@Nullable
	private ActionFailureReason getDialogChangeTTRActionFailedReason(Dialog activeDialog) {
		ActionFailureReason failureReason = null;

		if (activeDialog == null || !activeDialog.isOpen()) {
			failureReason = ActionFailureReason.NOT_ACTIVE;

		} else if (activeDialog.getDialogType() == DialogType.POST_SURVEY) {
			failureReason = ActionFailureReason.POST_SURVEY_IN_PROGRESS;
		}

		return failureReason;
	}

	public boolean canActiveDialogChangeTTR() {

		Dialog activeDialog = amsDialogs.getActiveDialog();

		ActionFailureReason dialogFailedReason = getDialogChangeTTRActionFailedReason(activeDialog);
		if (dialogFailedReason != null) {
			return false;
		}

		return true;
	}

	@Override
	public void sendCSAT(String brandId, String conversationID, int mStarsNumber, int yesNoValue) {
		new SendCsatCommand(mAccountsController.getConnectionUrl(brandId), conversationID, mStarsNumber, yesNoValue).execute();
	}

	@Override
	public void sendUserProfile(String brandId, UserProfileBundle userProfileBundle) {
		new SendSetUserProfileCommand(this, brandId, userProfileBundle).execute();
	}

	@Override
	public void setCallback(LivePersonCallback listener) {
		mEventsProxy.setCallback(listener);
	}

	@Override
	public void removeCallback() {
		mEventsProxy.removeCallback();
	}

	@Override
	public void checkActiveConversation(String targetId, final ICallback<Boolean, Exception> callback) {
		amsConversations.getActiveConversation(targetId).setPostQueryOnBackground(new DataBaseCommand.QueryCallback<Conversation>() {
			@Override
			public void onResult(Conversation data) {
				boolean conversationOpen = false;
				if (data != null) {
					conversationOpen = true;
				}
				callback.onSuccess(conversationOpen);
			}
		}).execute();
	}

	public void queryActiveDialog(String targetId,final ICallback<Dialog, Exception> callback) {
		amsDialogs.queryActiveDialog(targetId).setPostQueryOnBackground(new DataBaseCommand.QueryCallback<Dialog>() {
			@Override
			public void onResult(Dialog data) {
				if (data != null) {
					callback.onSuccess(data);
				} else {
					callback.onError(new Exception("Error: No active dialog"));
				}
			}
		}).execute();
	}

//	public void getActiveConversation(String targetId, final ICallback<Conversation, Exception> callback) {
//		amsConversations.getActiveConversation(targetId).setPostQueryOnBackground(new DataBaseCommand.QueryCallback<Conversation>() {
//			@Override
//			public void onResult(Conversation data) {
//				if (data != null) {
//					callback.onSuccess(data);
//				} else {
//					callback.onError(new Exception("Error: No active conversation"));
//				}
//			}
//		}).execute();
//	}

	@Override
	public void checkConversationIsMarkedAsUrgent(String targetId, final ICallback<Boolean, Exception> callback) {
		amsConversations.getActiveConversation(targetId).setPostQueryOnBackground(new DataBaseCommand.QueryCallback<Conversation>() {
			@Override
			public void onResult(Conversation data) {
				boolean conversationUrgent = false;
				if (data != null) {
					if (data.getConversationTTRType() == TTRType.URGENT &&
							data.isConversationOpen()) {
						conversationUrgent = true;
					}
				}
				callback.onSuccess(conversationUrgent);
			}
		}).execute();
	}

	@Override
	public void checkAgentID(String brandId, final ICallback<AgentData, Exception> callback) {
		amsDialogs.queryActiveDialog(brandId).setPostQueryOnBackground(new DataBaseCommand.QueryCallback<Dialog>() {
			@Override
			public void onResult(Dialog data) {
				if (data != null) {

					Context context = Infra.instance.getApplicationContext();
					if (context == null) {
						callback.onSuccess(null);
						return;
					}
					boolean updateAlways = Configuration.getBoolean(R.bool.send_agent_profile_updates_when_conversation_closed);
					if (updateAlways || data.isOpen()) {
						String agentOriginatorId = data.getAssignedAgentId();
						if (TextUtils.isEmpty(agentOriginatorId)) {
							callback.onSuccess(null);
						} else {
							amsUsers.getUserById(agentOriginatorId).setPostQueryOnBackground(new DataBaseCommand.QueryCallback<MessagingUserProfile>() {
								@Override
								public void onResult(MessagingUserProfile data) {
									if (data != null) {
										AgentData agentData = new AgentData();
										agentData.mFirstName = data.getFirstName();
										agentData.mLastName = data.getLastName();
										agentData.mAvatarURL = data.getAvatarUrl();
										agentData.mEmployeeId = data.getDescription();
										agentData.mNickName = data.getNickname();
										callback.onSuccess(agentData);
										return;
									}
									callback.onSuccess(null);
								}
							}).execute();
						}
					} else {
						callback.onSuccess(null);
					}
				} else {
					callback.onSuccess(null);
				}
			}
		}).execute();
	}

	/**
	 * Clear all messages and conversations of the given targetId.
	 * This method will clear only if there is no open conversation active.
	 *
	 * @param targetId
	 * @return <code>true</code> if messages cleared, <code>false</code> if messages were not cleared (due to open conversation, or no current brand)
	 */
	@Override
	public boolean clearHistory(final String targetId) {

		// Get active conversation
		boolean conversationActive = amsConversations.isConversationActive(targetId);
		if (conversationActive) {

			LPMobileLog.w(TAG, "clearHistory: There is an open conversation. Cannot clear history");

			return false;
		}

		// Remove all messages
		amsMessages.clearMessagesOfClosedConversations(targetId).setPostQueryOnBackground(new DataBaseCommand.QueryCallback<Integer>() {
			@Override
			public void onResult(Integer data) {
				LPMobileLog.d(TAG, "clearHistory: Removed " + data + " messages");
			}
		}).execute();

		// Remove all dialogs
		amsDialogs.clearClosedDialogs(targetId).setPostQueryOnBackground(new DataBaseCommand.QueryCallback<Integer>() {
			@Override
			public void onResult(Integer data) {
				LPMobileLog.d(TAG, "clearHistory: Removed " + data + " dialogs");
			}
		}).execute();

		// Remove all conversations
		amsConversations.clearClosedConversations(targetId).setPostQueryOnBackground(new DataBaseCommand.QueryCallback<Integer>() {
			@Override
			public void onResult(Integer data) {
				LPMobileLog.d(TAG, "clearHistory: Removed " + data + " conversations");
			}
		}).execute();

		return true;

	}

	@Override
	public void clearAllConversationData(final String targetId) {
		LPMobileLog.d(TAG, "clearAllConversationData");

		amsMessages.clearAllMessages(targetId).setPostQueryOnBackground(new DataBaseCommand.QueryCallback<Integer>() {
			@Override
			public void onResult(Integer data) {
				amsMessages.clear();
				amsDialogs.clearAllDialogs(targetId).setPostQueryOnBackground(new DataBaseCommand.QueryCallback<Integer>() {
					@Override
					public void onResult(Integer data) {
						amsDialogs.clear();
						amsConversations.clearAllConversations(targetId).setPostQueryOnBackground(new DataBaseCommand.QueryCallback<Integer>() {
							@Override
							public void onResult(Integer data) {
								amsConversations.clear();
							}
						}).execute();
					}
				}).execute();
			}
		}).execute();
	}

	public String getOriginatorId(String targetId) {
		return amsUsers.getConsumerId(targetId);
	}

	public void onAgentDetailsChanged(final MessagingUserProfile userProfile, boolean isDialogOpen) {
		Context context = Infra.instance.getApplicationContext();
		if (context == null) {
			return;
		}
		boolean updateAlways = Configuration.getBoolean(R.bool.send_agent_profile_updates_when_conversation_closed);
		if (updateAlways || isDialogOpen) {
			//update
			// If user profile is not provided we need to send agent details null
			AgentData agentData = null;
			if (userProfile != null) {
				agentData = new AgentData();
				agentData.mFirstName = userProfile.getFirstName();
				agentData.mLastName = userProfile.getLastName();
				agentData.mAvatarURL = userProfile.getAvatarUrl();
				agentData.mEmployeeId = userProfile.getDescription();
				agentData.mNickName = userProfile.getNickname();
			}

			mEventsProxy.onAgentDetailsChanged(agentData);
		}
	}

	/**
	 * Shut down Messaging.
	 * @param listener to shut down completion.
	 */
	@Override
	public void shutDown(ShutDownCompletionListener listener) {
		LPMobileLog.d(TAG, "Shutting down...");
		messagingShutDown(listener);
	}

	/**
	 * Shut down Messaging when upper module (MessagingUI) exists. with it's own ShutDownProcess.
	 * If Messaging is being used without upper module of UI, use {@link #shutDown(ShutDownCompletionListener)}
	 *
	 * ShutDown flow :
	 *
	 * MessagingUi shutdown -> ShutDownCompleted -> Messaging shutdown -> ShutDownCompleted -> Infra shutdown.
	 *
	 * Shut down get MessagingUI's ShutDownProcess, that runs before shutting down Messaging - since
	 * MessagingUI module is based on Messaging module.
	 *
	 * This method call Infra.shutDown with Messaging's ShutDownProcess and will call it before Infra ShutDownProcess.
	 *
	 * Infra's ShutDownProcess will run only when Messaging's ShutDownProcess completed - since
	 * Messaging module is based on Infra module.
	 *
	 * @param process shut down process of upper module (MessagingUI)
	 */
	@Override
	public void shutDown(final ShutDownProcess process) {

		//Sending Infra how to shutdown Messaging module.
		Infra.instance.shutDown(new ShutDownProcess() {

			// get Infra's ShutDownCompletionListener. when Messaging will finish shut down-
			// we will notify Infra with this listener, and it'll run Infra's shutdown process.
			@Override
			public void shutDown(final ShutDownCompletionListener listener) {

				//first, call MessagingUI shut down process.
				process.shutDown(new ShutDownCompletionListener() {

					@Override
					public void shutDownCompleted() {
						//when MessagingUI's completed- run messaging's shutdown.
						LPMobileLog.d(TAG, "Shutting down...");
						messagingShutDown(listener);
					}

					@Override
					public void shutDownFailed() {

					}

				});
			}
		});
	}

	private void messagingShutDown(final ShutDownCompletionListener listener) {
		ShutDownCompletionListener messagingShutDownListener = new ShutDownCompletionListener() {
			@Override
			public void shutDownCompleted() {
				listener.shutDownCompleted();
			}

			@Override
			public void shutDownFailed() {
				listener.shutDownFailed();
			}
		};
		mConnectionController.shutDown(messagingShutDownListener);
		amsReadController.shutDown();
		amsMessages.shutDown();
		amsDialogs.shutDown();
		amsConversations.shutDown();
	}

	/**
	 * Logout from messaging. removing all user data
	 * @param context                  - app Context
	 * @param initData                 - brand id to logout , host app id
	 * @param logoutLivePersonCallBack - logout callback for completion
	 */
	@Override
	public void logout(Context context, final MessagingInitData initData, final LogoutLivePersonCallBack logoutLivePersonCallBack) {
		logout(context, initData, new LogoutProcess() {
			@Override
			public void initForLogout() {}
			@Override
			public void preLogout(PreLogoutCompletionListener listener) {
				listener.preLogoutCompleted();
			}
			@Override
			public void shutDownForLogout(ShutDownCompletionListener listener) {
				listener.shutDownCompleted();
			}
			@Override
			public void logout() {}
			@Override
			public LogoutLivePersonCallBack getLogoutCallback() {
				return logoutLivePersonCallBack;
			}
		});
	}


	/**
	 * Logout and removing all user data from messaging when upper module (MessagingUI) exists.
	 * If Messaging is being used without upper module of UI, use {@link #logout(Context, MessagingInitData, LogoutLivePersonCallBack)}
	 * @param context       - app Context
	 * @param initData      - brand id to logout , host app id
	 * @param logoutProcess - logout process of upper module (MessagingUI)
	 */
	@Override
	public void logout(final Context context, final MessagingInitData initData, final LogoutProcess logoutProcess) {
		Infra.instance.logout(context, initData, new LogoutProcess() {

			@Override
			public void initForLogout() {
				initMessaging(context, initData);
				logoutProcess.initForLogout();
			}

			/**
			 * un-registering pusher before logging out.
			 * @param listener
			 */
			@Override
			public void preLogout(final PreLogoutCompletionListener listener) {
				logoutProcess.preLogout(new PreLogoutCompletionListener() {
					@Override
					public void preLogoutCompleted() {
						unregisterPusher(initData.getBrandId(), initData.getAppId(), new ICallback<Void, Exception>() {
							@Override
							public void onSuccess(Void value) {
								listener.preLogoutCompleted();
							}

							@Override
							public void onError(Exception exception) {
								listener.preLogoutFailed(exception);
							}
						}, true);
					}

					@Override
					public void preLogoutFailed(Exception e) {
						listener.preLogoutFailed(e);
					}
				});
			}

			@Override
			public LogoutLivePersonCallBack getLogoutCallback() {
				return logoutProcess.getLogoutCallback();
			}

			@Override
			public void shutDownForLogout(final ShutDownCompletionListener listener) {
				logoutProcess.shutDownForLogout(new ShutDownCompletionListener() {
					@Override
					public void shutDownCompleted() {
						messagingShutDown(listener);
					}

					@Override
					public void shutDownFailed() {

					}
				});
			}

			@Override
			public void logout() {
				//first delete upper level than us.
				logoutProcess.logout();
				clear();
			}
		});
	}

	@Override
	public void clear() {
		amsConversations.clear();
		amsDialogs.clear();
		amsMessages.clear();
		amsUsers.clear();
		mAccountsController.clear();
		mConnectionController.clear();
		mClientProperties.clear();
		FileUtils.deleteAllSharedFiles();
		mMessageValidator = null;
	}
	@Override
	public boolean isSocketOpen(String brandId) {
		return mConnectionController.isSocketOpen(brandId);
	}

	/**
	 * Check if initialized based on lower module- Infra.
	 * @return if Messaging is Initialized.
	 */
	@Override
	public boolean isInitialized() {
		return Infra.instance.isInitialized();
	}

	/**
	 * Get a comma separated string with all rowIds of messages that are currently uploading images
	 * @return
	 */
	public String getInProgressUploadMessageRowIdsString(){

		LPMobileLog.d(TAG, "getInProgressUploadMessageRowIdsString: direct call (no service)");
		return mFileSharingManager.getInProgressUploadMessageRowIdsString();

	}


	public FileSharingManager getFileSharingManager() {
		return mFileSharingManager;
	}

	public boolean isEnableStructuredContent() {
		return mEnableStructuredContent;
	}

	/**
	 * Set a pending intent action for the image foreground service notification when clicked
	 *
	 * @param imageServicePendingIntent
	 */
	public void setImageServicePendingIntent(PendingIntent imageServicePendingIntent) {
		mImageServicePendingIntent = imageServicePendingIntent;
	}

	public PendingIntent getImageServicePendingIntent() {
		return mImageServicePendingIntent;
	}

	public Notification.Builder getImageForegroundServiceUploadNotificationBuilder() {
		return mImageForegroundServiceUploadNotificationBuilder;
	}

	public Notification.Builder getImageForegroundServiceDownloadNotificationBuilder() {
		return mImageForegroundServiceDownloadNotificationBuilder;
	}

	public void setImageForegroundServiceUploadNotificationBuilder(Notification.Builder imageForegroundServiceNotificationBuilder) {
		mImageForegroundServiceUploadNotificationBuilder = imageForegroundServiceNotificationBuilder;
	}

	public void setImageForegroundServiceDownloadNotificationBuilder(Notification.Builder imageForegroundServiceNotificationBuilder) {
		mImageForegroundServiceDownloadNotificationBuilder = imageForegroundServiceNotificationBuilder;
	}

	public void sendDeliveryStatusUpdateCommand(String brandId, String dialogId, String conversationId, int sequence , DeliveryStatus deliveryStatus, DeliveryStatusUpdateInfo info){
		new DeliveryStatusUpdateCommand(mAccountsController.getConnectionUrl(brandId), brandId, dialogId, conversationId, sequence, deliveryStatus, info).execute();
	}

	public ConversationViewParams getConversationViewParams() {
		return mConversationViewParams;
	}

	public void setConversationViewParams(ConversationViewParams conversationViewParams) {
        LPMobileLog.d(TAG, "Setting conversation view params : " + conversationViewParams);
        mConversationViewParams = conversationViewParams;
    }

	public MessagingEventSubscriptionManager getMessagingEventSubscriptionManager() {
		return mMessagingEventSubscriptionManager;
	}

	public LPAudioUtils getAudioUtils() {
		return mAudioUtils;
	}

	public boolean isStillBusyFetching() { return isStillBusyFetching; }

	public void setStillBusyFetching(boolean stillBusyFetching) {
		isStillBusyFetching = stillBusyFetching;
	}

	public void setContext(Context context) {
		mContext = context;
	}
}
