package com.liveperson.messaging.model;

import android.content.Context;
import android.database.Cursor;
import android.os.Bundle;
import android.text.TextUtils;

import com.liveperson.infra.controller.DBEncryptionHelper;
import com.liveperson.infra.database.tables.MessagesTable;
import com.liveperson.infra.database.tables.UsersTable;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.messaging.R;
import com.liveperson.infra.utils.DateUtils;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.messaging.structuredcontent.model.elements.SimpleElement;
import com.liveperson.messaging.structuredcontent.parsers.ElementParser;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * Created by shiranr on 11/3/16.
 */
public class FullMessageRow {

	public static final String TAG = "FullMessageRow";

    public static final String EXTRA_AGENT_AVATAR = "EXTRA_AGENT_AVATAR";
    private boolean isStructuredContentEmpty;
    MessagingChatMessage mMessagingChatMessage;
    String mAgentAvatar;
    String mAgentNickName;
    FileMessage mFileMessage;
	// Structured content layout tree
	private SimpleElement mRootElement;

    public FullMessageRow(MessagingChatMessage messagingChatMessage, String agentAvatar, FileMessage fileMessage) {
        mMessagingChatMessage = messagingChatMessage;
        mAgentAvatar = agentAvatar;
        mFileMessage = fileMessage;
    }

    public FullMessageRow (Cursor cursor) {

        long messageId = cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_ID));
        String originatorId = cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_ORIGINATOR_ID));
        String message = cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_TEXT));
        long timeStamp = cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_TIMESTAMP));
        int typeInt = cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_MESSAGE_TYPE));
        MessagingChatMessage.MessageType type = MessagingChatMessage.MessageType.values()[typeInt];
        MessagingChatMessage.MessageState state = MessagingChatMessage.MessageState.values()[cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_STATUS))];
        String dialogId = cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_DIALOG_ID));
        String eventId = cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_EVENT_ID));
        int serverSequence = cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_SERVER_SEQUENCE));

        EncryptionVersion messageEncryptionVersion = EncryptionVersion.fromInt(cursor.getInt(cursor.getColumnIndex(MessagesTable.ENCRYPTION_VERSION_CURSOR_AS_VALUE)));

        EncryptionVersion userEncryptionVersion = EncryptionVersion.fromInt(cursor.getInt(cursor.getColumnIndex(UsersTable.ENCRYPTION_VERSION_CURSOR_AS_VALUE)));
        mAgentAvatar = cursor.getString(cursor.getColumnIndex(UsersTable.KEY_PROFILE_IMAGE));

        mAgentAvatar = DBEncryptionHelper.decrypt(userEncryptionVersion, mAgentAvatar);
        mAgentNickName = cursor.getString(cursor.getColumnIndex(UsersTable.KEY_NICKNAME));
        mAgentNickName = DBEncryptionHelper.decrypt(userEncryptionVersion, mAgentNickName);
        mMessagingChatMessage = new MessagingChatMessage(originatorId, message, timeStamp, dialogId, eventId, type, state, messageEncryptionVersion);
        mMessagingChatMessage.setServerSequence(serverSequence);
        mMessagingChatMessage.setMessageId(messageId);

        mFileMessage = FileMessage.fromCursor(cursor);
    }

    public FileMessage getFileMessage() {
        return mFileMessage;
    }

    public MessagingChatMessage getMessagingChatMessage() {
        return mMessagingChatMessage;
    }

    public String getAgentAvatar() {
        return mAgentAvatar;
    }

    public String getAgentNickName() {
        return mAgentNickName;
    }

    /**
     *
     * @param newMessageRow
     * @return 0 if it's the same item.
     *         1 if newMessageRow is a new item older than this current msg.
     *         -1 if newMessageRow is a new item newer than this current msg.
     */
    public int newerThan(FullMessageRow newMessageRow) {
        boolean isSameSequence = mMessagingChatMessage.getServerSequence() == newMessageRow.getMessagingChatMessage().getServerSequence();
        boolean isSameDialogId = TextUtils.equals(mMessagingChatMessage.getDialogId(),newMessageRow.getMessagingChatMessage().getDialogId());
        boolean needToCheckSeq = true;
        if (!TextUtils.isEmpty(mMessagingChatMessage.getEventId()) && !TextUtils.isEmpty(newMessageRow.getMessagingChatMessage().getEventId()) ){
            //if both event ids are not null, we don't need to check sequence
            //we can base the equation on the event id only.
            needToCheckSeq = false;
        }
        boolean isSameEventId = TextUtils.equals(mMessagingChatMessage.getEventId(),newMessageRow.getMessagingChatMessage().getEventId());
        boolean isNewerTime = mMessagingChatMessage.getTimeStamp() > newMessageRow.getMessagingChatMessage().getTimeStamp();

        // if we have the same event id or (we need to check sequence and it's the same conv id and sequence - its the same message.
        if ( isSameEventId || (needToCheckSeq && (isSameSequence && isSameDialogId))){
            return 0;
        }
        if (isNewerTime){
            return 1;
        }
        return -1;
    }

    /**
     * Updating current message, and creating diff bundle to update ui.
     * @param newMessageRow
     * @return
     */
    public Bundle update(FullMessageRow newMessageRow) {

        Bundle bundle = new Bundle();
        bundle.putAll(mMessagingChatMessage.update(newMessageRow.getMessagingChatMessage()));
        if (mFileMessage != null) {
            bundle.putAll(mFileMessage.update(newMessageRow.getFileMessage()));
        }else if (newMessageRow.getFileMessage() != null){
            bundle.putAll(newMessageRow.getFileMessage().getAllFileChanges());
            mFileMessage = newMessageRow.getFileMessage();
        }

        if (!TextUtils.equals(mAgentAvatar,newMessageRow.getAgentAvatar())){
            mAgentAvatar = newMessageRow.getAgentAvatar();
            bundle.putString(EXTRA_AGENT_AVATAR, mAgentAvatar);
        }

        return (bundle.isEmpty() ? null : bundle);
    }

    public static FullMessageRow generateUnreadIndicatorMessage(Context context, int numUnreadMessages, long timeStamp) {
        long messageId = -1;
        String originatorId = null;
        String unreadMessagesBase = context.getResources().getQuantityString(R.plurals.lp_unread_message, numUnreadMessages);
        String message = String.format(unreadMessagesBase, numUnreadMessages);
        MessagingChatMessage.MessageType type = MessagingChatMessage.MessageType.UNREAD_INDICATOR;
        MessagingChatMessage.MessageState state = MessagingChatMessage.MessageState.RECEIVED;
        String convId = "";
        String eventId = "";
        int serverSequence = -1;

        MessagingChatMessage messagingChatMessage = new MessagingChatMessage(originatorId, message, timeStamp, convId, eventId, type, state, EncryptionVersion.NONE);
        messagingChatMessage.setServerSequence(serverSequence);
        messagingChatMessage.setMessageId(messageId);

        return new FullMessageRow(messagingChatMessage, null, null);
    }

    public static Bundle getUnreadIndicatorMessageDiff(FullMessageRow newUnreadIndicatorMessage) {
        Bundle bundle = new Bundle();
        bundle.putString(MessagingChatMessage.EXTRA_MESSAGE_TEXT, newUnreadIndicatorMessage.getMessagingChatMessage().getMessage());
        return bundle;
    }


    public static FullMessageRow getLoadingIndicatorMessage(Context context, long timeStamp, boolean setText) {
        long messageId = -1;
        String originatorId = null;
        String message = setText ? context.getString(R.string.lp_still_loading_message) : "" ;// at the beginning there is no text of loading. only after 10 seconds we'll show text with loading
        MessagingChatMessage.MessageType type = MessagingChatMessage.MessageType.LOADING;
        MessagingChatMessage.MessageState state = MessagingChatMessage.MessageState.RECEIVED;
        String dialogId = "";
        String eventId = "";
        int serverSequence = -1;

        MessagingChatMessage messagingChatMessage = new MessagingChatMessage(originatorId, message, timeStamp, dialogId, eventId, type, state, EncryptionVersion.NONE);
        messagingChatMessage.setServerSequence(serverSequence);
        messagingChatMessage.setMessageId(messageId);

        return new FullMessageRow(messagingChatMessage, null, null);
    }


    public static Bundle getLoadingIndicatorMessageDiff(Context context) {
        Bundle bundle = new Bundle();
        // at the beginning there is no text of loading. only after 10 seconds we'll show text with loading
        bundle.putString(MessagingChatMessage.EXTRA_MESSAGE_TEXT, context.getString(R.string.lp_still_loading_message));
        return bundle;
    }


    public static FullMessageRow generateAgentTypingIndicatorMessage(String avatarUrl) {
        long messageId = -1;
        String originatorId = null;
        MessagingChatMessage.MessageType type = MessagingChatMessage.MessageType.AGENT_IS_TYPING_INDICATOR;
        MessagingChatMessage.MessageState state = MessagingChatMessage.MessageState.READ;
        String convId = "";
        String eventId = "";
        int serverSequence = -1;

        long oneSecondToMidnight = DateUtils.getEndOfToday() - 1000; // So the indicator will always appear in the end of the messages list
        MessagingChatMessage messagingChatMessage = new MessagingChatMessage(originatorId, "...", oneSecondToMidnight, convId, eventId, type, state, EncryptionVersion.NONE);
        messagingChatMessage.setServerSequence(serverSequence);
        messagingChatMessage.setMessageId(messageId);

        return new FullMessageRow(messagingChatMessage, avatarUrl, null);
    }

    /**
     * At the beginning there is no text of loading. only after 10 seconds we'll show text with loading
     *
     * @param message Still loading messaging.
     */
    public static FullMessageRow getLoadingHistoryIndicatorMessage(long timeStamp, String message) {
        long messageId = -1;
        String originatorId = null;
        MessagingChatMessage.MessageType type = MessagingChatMessage.MessageType.LOADING;
        MessagingChatMessage.MessageState state = MessagingChatMessage.MessageState.RECEIVED;
        String convId = "";
        String eventId = "";
        int serverSequence = -1;

        MessagingChatMessage messagingChatMessage = new MessagingChatMessage(originatorId, message, timeStamp, convId, eventId, type, state, EncryptionVersion.NONE);
        messagingChatMessage.setServerSequence(serverSequence);
        messagingChatMessage.setMessageId(messageId);

        return new FullMessageRow(messagingChatMessage, null, null);
    }

	/**
	 * Generate FullMessageRow object with AGENT_QUICK_REPLIES type that contains the given quickReplyJsonString.
	 * This create a message with empty dialogId, eventId, serverSequence, messageId since it is a temporary message that needs to be added to the
	 * conversation and then removed. We add dialogId only when Quick reply question is of PCS survey since we want to send metadata to UMS. (See MessagesAsListLoader, case AGENT_QUICK_REPLIES:)
	 * The timestamp of the generated message is one millisecond after the given relatedAgentMessageTimestamp since we want this message
	 * to appear right after the related agent message
	 * @param quickReplyJsonString the required QuickReplies JSON string
	 * @param relatedAgentMessageTimestamp the related agent message's timestamp
     * @param dialogId currently active dialog's id. This is Empty unless PCS dialog.
	 * @return
	 */
	public static FullMessageRow generateAgentQuickReplyMessage(String quickReplyJsonString, long relatedAgentMessageTimestamp, String dialogId) {
		MessagingChatMessage.MessageType type = MessagingChatMessage.MessageType.AGENT_QUICK_REPLIES;
		MessagingChatMessage.MessageState state = MessagingChatMessage.MessageState.READ;

		long quickReplyMessageTimestamp = relatedAgentMessageTimestamp + 1; // Put the QR message right after the related agent message
		MessagingChatMessage messagingChatMessage = new MessagingChatMessage(null, quickReplyJsonString, quickReplyMessageTimestamp, dialogId, "", type, state, EncryptionVersion.NONE);
		messagingChatMessage.setServerSequence(-1);
		messagingChatMessage.setMessageId(-1);

		FullMessageRow message = new FullMessageRow(messagingChatMessage, null, null);

		LPMobileLog.d(TAG, "generateAgentQuickReplyMessage: add the quick replies json to the message");
		SimpleElement rootElement = null;
		try {
			JSONObject root = new JSONObject(quickReplyJsonString);

			rootElement = ElementParser.parse(root);
		} catch (JSONException e) {
			LPMobileLog.d(TAG, "parse: there is a problem parsing the structured content json");
			LPMobileLog.w(TAG, e);
		}
		message.setRootElement(rootElement);

		return message;
	}

	public SimpleElement getRootElement() {
		return mRootElement;
	}

	public void setRootElement(SimpleElement layoutElement) {
		mRootElement = layoutElement;
	}

    public boolean isStructuredContentEmpty() {
        return isStructuredContentEmpty;
    }

    public void setStructuredContentEmpty(boolean structuredContentEmpty) {
        isStructuredContentEmpty = structuredContentEmpty;
    }
}
