package com.liveperson.messaging.network;

import android.os.Message;
import android.os.Process;

import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.utils.DispatchQueue;
import com.liveperson.infra.utils.HandleMessageCallback;
import com.liveperson.messaging.network.http.MessageTimeoutListener;

/**
 * Created by nirni on 12/27/15.
 * <p/>
 * This class is used to hold pending messages for a specific timeout in order to track delivery.
 * The messages are stored by the requestId.
 * When an ack for a message is received it should call the remove() method which removes the message
 * from the queue.
 */
public class MessageTimeoutQueue implements HandleMessageCallback {

    private static final String TAG = MessageTimeoutQueue.class.getSimpleName();
    private static final int PUBLISH_TIMEOUT = 40000;
    private static final int KEEP_ALIVE_TIMEOUT = 15000;
    private final MessageTimeoutListener mMessageTimeoutListener;
    private DispatchQueue mMessagesQueue;

    public enum MessageType {KEEP_ALIVE, PUBLISH}

    public MessageTimeoutQueue(MessageTimeoutListener messageTimeoutListener) {
        mMessageTimeoutListener = messageTimeoutListener;
        mMessagesQueue = new DispatchQueue(TAG, Process.THREAD_PRIORITY_BACKGROUND);
        mMessagesQueue.setHandleMessageCallback(this);
    }

    /**
     * Add a message to the queue
     *
     * @param requestId
     * @param brandId
     * @param dialogId
     */
    public void add(MessageType type, int requestId, String brandId, String dialogId, String eventId) {
        add(type, requestId, brandId, dialogId, eventId, 0);
    }

    public void add(MessageType type, int requestId, String brandId, String dialogId, int delay) {
        add(type, requestId, brandId, dialogId, null, delay);
    }

    public void add(MessageType type, int requestId, String brandId, String dialogId, String eventId, int delay) {
        LPMobileLog.d(TAG, "Adding message. requestId: " + requestId + " brandId: " + brandId + " dialogId: " + dialogId);
        Message msg = Message.obtain();
        msg.what = requestId;
        msg.obj = new BrandConversationObj(brandId, dialogId, type, eventId);

        if (type == MessageType.PUBLISH) {
            mMessagesQueue.sendMessage(msg, PUBLISH_TIMEOUT + delay);
        } else if (type == MessageType.KEEP_ALIVE) {
            remove(requestId);
            mMessagesQueue.sendMessage(msg, KEEP_ALIVE_TIMEOUT + delay);
        }
    }

    public void handleMsgImmediately(MessageType type, int requestId, String brandId, String conversationId, String eventId) {
        if (!remove(requestId)) {
            //only if its waiting to be handled send it immediately, otherwise- ignore.
            return;
        }
        LPMobileLog.d(TAG, "Sending message now. requestId: " + requestId + " brandId: " + brandId + " conversationId: " + conversationId);
        Message msg = Message.obtain();
        msg.what = requestId;
        msg.obj = new BrandConversationObj(brandId, conversationId, type, eventId);

        if (type == MessageType.PUBLISH) {
            mMessagesQueue.sendMessage(msg, 0);
        } else if (type == MessageType.KEEP_ALIVE) {
            remove(requestId);
            mMessagesQueue.sendMessage(msg, 0);
        }
    }


    /**
     * Removed a message from the queue
     *
     * @param requestId
     */
    public boolean remove(int requestId) {
        LPMobileLog.d(TAG, "Remove message from queue. requestId: " + requestId);
        return mMessagesQueue.removeMessage(requestId);
    }

    /**
     * Removed all messages from the queue
     */
    public void removeAll() {
        LPMobileLog.d(TAG, "Remove all messages from queue.");
        mMessagesQueue.cleanupQueue();
    }

    @Override
    public void onHandleMessage(Message msg) {
        int requestId = msg.what;
        BrandConversationObj brandConversationObj = (BrandConversationObj) msg.obj;

        LPMobileLog.d(TAG, "Timeout expired on messages. Set state to Error. requestId: " + requestId + " brandId: " + brandConversationObj.brandId + " conversationId: " + brandConversationObj.dialogId);

        if (brandConversationObj.type == MessageType.PUBLISH) {
            // Message timeout expired. Set the message status to Error
            mMessageTimeoutListener.onPublishMessageTimeout(
                    brandConversationObj.brandId,
                    brandConversationObj.eventId,
                    brandConversationObj.dialogId);
        }

        //if a message has failed to sent we need to close the socket and re open
        mMessageTimeoutListener.onMessageTimeout(brandConversationObj.brandId);
    }

    /**
     * A class to hold brandId and conversationId
     */
    private static class BrandConversationObj {

        String brandId;
        String dialogId;
        MessageType type;
        String eventId;

        public BrandConversationObj(String brandId, String dialogId, MessageType type, String eventId) {
            this.brandId = brandId;
            this.dialogId = dialogId;
            this.type = type;
            this.eventId = eventId;
        }
    }
}
