package com.liveperson.messaging.commands;

import android.text.TextUtils;

import com.liveperson.infra.Command;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.Infra;
import com.liveperson.infra.InternetConnectionService;
import com.liveperson.infra.controller.DBEncryptionHelper;
import com.liveperson.infra.database.DataBaseCommand;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.network.http.requests.BadgeCounterRequest;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.TaskType;
import com.liveperson.messaging.utils.TokenUtils;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.model.AmsAccount;
import com.liveperson.messaging.model.Dialog;
import com.liveperson.messaging.model.SynchronizedInternetConnectionCallback;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

import javax.net.ssl.SSLPeerUnverifiedException;


/**
 * Created by ofira on 9/14/17.
 */
// todo Perry, verify with Pusher team that the unread badge is for the conversation and not dialog.
public class GetUnreadMessagesCountCommand implements Command {

    public static final String TAG = "GetUnreadMessagesCountCommand";

    private static final String PUSHER_BADGE_URL = "https://%s/api/account/%s/device/unread-messages-count?appId=%s&lpId=%s";

    private static final String BADGE_SEQUENCE = "badge_sequence";
    private static final String BADGE_COUNT = "badge_count";
    private static final String BADGE_TIMESTAMP = "badge_timestamp";
    private static final String BADGE_CONVERSATION_ID = "badge_conversation_id";
    private static final String ERROR_UNABLE_TO_MAKE_REQUEST = "Unable to make request.";
    private static final String ERROR_SERVER_ERROR = "Server error: ";

    private static final long BADGE_LAST_TIMESTAMP_THRESHOLD = 1000 * 10; // 10 seconds

    private final Messaging mController;
    private String mBrandId;
    private ICallback<Integer, Exception> mCallback;
    private String mAppId;
    private String mPusherDomain;
    private PreferenceManager mPreferenceManager;
    private String mLocalToken;

    public GetUnreadMessagesCountCommand(Messaging messagingController, String brandId, String appId, ICallback<Integer, Exception> callback) {
        mController = messagingController;
        mBrandId = brandId;
        mAppId = appId;
        mCallback = callback;
        mPreferenceManager = PreferenceManager.getInstance();
    }

    @Override
    public void execute() {

        //Checking if the time threshold was passed
        if (!isBadgeThresholdWasPassed(mBrandId)) {
            LPMobileLog.d(TAG, "Time threshold was not passed yet. Return cached data");
            returnCachedDetails();

        } else {

            LPMobileLog.d(TAG, "Time threshold was passed");


            // If the AmsAccount does not exist for the given brandId (e.g. the app was killed and no conversation started yet),
            // we need to get the token from shared preferences since it's still does not exist in the account.
            if (mController.mAccountsController.getAccount(mBrandId) == null) {
                mLocalToken = getTokenFromSharedPreferences();
                if (TokenUtils.isJwtExpired(mLocalToken)) {
                    sendTokenExpired();
                    return;
                }
            } else if (mController.mAccountsController.isTokenExpired(mBrandId)) { // Check token from the account
                sendTokenExpired();
                return;
            }


            LPMobileLog.d(TAG, "JWT is valid - send request to Pusher");
            // Create pusher unregister URL
            mPusherDomain = mController.mAccountsController.getServiceUrl(mBrandId, ConnectionParamsCache.CSDS_PUSHER_DOMAIN_KEY);
            if (TextUtils.isEmpty(mPusherDomain)) {
                mPusherDomain = PreferenceManager.getInstance().getStringValue(ConnectionParamsCache.CSDS_PUSHER_DOMAIN_KEY, mBrandId, null);
                if (TextUtils.isEmpty(mPusherDomain)) {
                    notifyError(new Exception(ERROR_UNABLE_TO_MAKE_REQUEST + " Error: Missing Domain"));
                    return;
                }
            }

            if (TextUtils.isEmpty(mAppId)) {
                mCallback.onError(new Exception(ERROR_UNABLE_TO_MAKE_REQUEST + " Error: Missing appID"));
                return;
            }

            // Get consumerId from memory
            final String consumerId = mController.amsUsers.getConsumerId(mBrandId);

            // If consumerId is not available in memory try to get from DB
            if (TextUtils.isEmpty(consumerId)) {
                LPMobileLog.d(TAG, "execute: consumerId is not available. Trying to get from DB...");
                mController.amsUsers.getConsumerByBrandIDFromDB(mBrandId).setPostQueryOnBackground(new DataBaseCommand.QueryCallback<String>() {
                    @Override
                    public void onResult(String returnedConsumerId) {
                        if (returnedConsumerId != null && !TextUtils.isEmpty(returnedConsumerId)) {
                            validateStateAndSendRequest(mPusherDomain, returnedConsumerId);
                            LPMobileLog.d(TAG, "onResult: got  consumerId from DB (" + returnedConsumerId + "). get the badge counter with it...");

                        } else {
                            LPMobileLog.w(TAG, "onResult: Cannot get user profile from DB. Quit get badge counter");
                            mCallback.onError(new Exception(ERROR_UNABLE_TO_MAKE_REQUEST + " Error: Missing consumerID"));
                        }
                    }

                }).execute();
            } else {
                validateStateAndSendRequest(mPusherDomain, consumerId);
            }

        }
    }

    /**
     * Get the token that is stored in the shared preferences
     *
     * @return
     */
    private String getTokenFromSharedPreferences() {

        String token;
        String decryptedToken = mPreferenceManager.getStringValue(AmsAccount.KEY_ACCOUNT_TOKEN_ENC, mBrandId, null);
        if (TextUtils.isEmpty(decryptedToken)) {
            token = mPreferenceManager.getStringValue(AmsAccount.KEY_ACCOUNT_TOKEN, mBrandId, null);
            mPreferenceManager.remove(AmsAccount.KEY_ACCOUNT_TOKEN, mBrandId);
        } else {
            token = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1, decryptedToken);
        }

        return token;
    }

    /**
     * Send token expired callback and onError callback for this command
     */
    private void sendTokenExpired() {
        LPMobileLog.d(TAG, "JWT is expired - calling to onTokenExpired callback");
        mController.mEventsProxy.onTokenExpired();
        mCallback.onError(new Exception(ERROR_UNABLE_TO_MAKE_REQUEST + " Error: Token expired, refresh the token and try again"));
    }


    /**
     * Check if we have this conversation in the local DB:
     * Yes: check if the local sequence in lower than what pusher has -> notify: pusher counter
     * No: return 0 (the SDK is update with the latest messages) -> notify: 0
     *
     * @param convId
     * @param pushSequence
     * @param pushBadgeCounter
     */
    private void validateCounterWithSequence(final String convId, final int pushSequence, final int pushBadgeCounter) {

        LPMobileLog.d(TAG, "Conversation ID: " + convId + " Sequence " + pushSequence + " counter " + pushBadgeCounter);

		mController.amsDialogs.queryOpenDialogsOfConversation(convId).setPostQueryOnBackground(new DataBaseCommand.QueryCallback<ArrayList<Dialog>>() {
			@Override
			public void onResult(ArrayList<Dialog> dialogs) {

				if ((dialogs != null) && (dialogs.size() > 0)) {
					// Get the first dialog
					Dialog dialog = dialogs.get(0);
					LPMobileLog.d(TAG, "validateCounterWithSequence: found open dialog (" + dialog.getDialogId() + ") for conversion " + convId);
					if(dialog.getLastServerSequence() > pushSequence) {
						LPMobileLog.d(TAG, "Local opened dialog sequence: " + dialog.getLastServerSequence() + " remote sequence: " + pushSequence);
						updateCachedDetails(convId, dialog.getLastServerSequence(), 0);
						notifySuccess(0);
						return;
					}
					else {
						LPMobileLog.d(TAG, "Local opened dialog sequence: " + dialog.getLastServerSequence() + " remote sequence: " + pushSequence + ": returning unread counter = " + pushBadgeCounter);
						updateCachedDetails(convId, pushSequence, pushBadgeCounter);
						notifySuccess(pushBadgeCounter);
						return;
					}
				}

				LPMobileLog.d(TAG, "validateCounterWithSequence: no active dialog for the conversation " + convId + ". Set counter to 0");
				updateCachedDetails(convId, pushSequence, 0);
				notifySuccess(0);
			}
		}).execute();
    }

    /**
     * Notify in case of a success with the relevant counter
     *
     * @param counter
     */
    private void notifySuccess(final int counter) {
        if (mCallback != null) {
            Infra.instance.postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    mCallback.onSuccess(counter);
                }
            });
        }
    }

    /**
     * Notify in case of an error
     *
     * @param exception
     */
    private void notifyError(final Exception exception) {
        if (mCallback != null) {
            Infra.instance.postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    if(exception instanceof SSLPeerUnverifiedException){
                        mController.mEventsProxy.onError(TaskType.INVALID_CERTIFICATE, exception.getMessage());

                    }
                    mCallback.onError(exception);
                }
            });
        }
    }

    /**
     * Checking if we have an active internet connection and send the request to Pusher
     *
     * @param consumerId
     */
    private void validateStateAndSendRequest(final String pusherDomain, final String consumerId) {

        if (InternetConnectionService.isNetworkAvailable()) {
            sendRequest(pusherDomain, consumerId);

        } else {
            new SynchronizedInternetConnectionCallback(new Runnable() {
                @Override
                public void run() {
                    sendRequest(pusherDomain, consumerId);
                }
            }).execute();
        }
    }

    private String buildPusherURL(String pusherDomain, String consumerId) {

        String pusherURL = String.format(PUSHER_BADGE_URL, pusherDomain, mBrandId, mAppId, consumerId);

        return pusherURL;
    }


    /**
     * Send a request to Pusher
     *
     * @param consumerId
     * @param pusherURL
     */
    private void sendRequest(String pusherURL, String consumerId) {
        pusherURL = buildPusherURL(pusherURL, consumerId);
        List<String> certificates = mController.mAccountsController.getCertificatePinningKeys(mBrandId);
        String token = mController.mAccountsController.getToken(mBrandId);
        token = (token == null) ? mLocalToken : token;

        if (token == null || pusherURL == null) {
            notifyError(new Exception(ERROR_UNABLE_TO_MAKE_REQUEST + " Error: Authorization failed. Token is missing or is invalid"));
            return;
        }

        new BadgeCounterRequest(pusherURL, token, certificates, new ICallback<String, Exception>() {
            @Override
            public void onSuccess(String json) {
                if (!TextUtils.isEmpty(json)) {
                    try {
                        LPMobileLog.i(TAG, "Unread response " + json);
                        JSONObject badgeObj = new JSONObject(json);
                        String conversationId = badgeObj.optString("conversationId");
                        int sequence = badgeObj.optInt("sequence");
                        int counter = badgeObj.optInt("badge");
                        validateCounterWithSequence(conversationId, sequence, counter);
                        updateTimeStamp();
                    } catch (JSONException e) {
                        notifyError(e);
                    }
                }
            }

            @Override
            public void onError(Exception exception) {
                LPMobileLog.e(TAG, "Error response", exception);
                notifyError(new Exception(ERROR_SERVER_ERROR + exception.getMessage()));
            }
        }).execute();
    }

    /**
     * Updating the conversation id, counter and sequence
     *
     * @param conversationId
     * @param counter
     * @param sequence
     */
    private void updateCachedDetails(String conversationId, int sequence, int counter) {
        PreferenceManager.getInstance().setStringValue(BADGE_CONVERSATION_ID, mBrandId, conversationId);
        PreferenceManager.getInstance().setIntValue(BADGE_COUNT, mBrandId, counter);
        PreferenceManager.getInstance().setIntValue(BADGE_SEQUENCE, mBrandId, sequence);
    }

    /**
     * Update time stamp value with current time
     */
    private void updateTimeStamp() {
        PreferenceManager.getInstance().setLongValue(BADGE_TIMESTAMP, mBrandId, System.currentTimeMillis());
    }


    /**
     * Return cached details - conversation id, sequence and counter
     */
    private void returnCachedDetails() {
        LPMobileLog.d(TAG, "Return cached badge counter");
        String conversationId = PreferenceManager.getInstance().getStringValue(BADGE_CONVERSATION_ID, mBrandId, "");
        int sequence = PreferenceManager.getInstance().getIntValue(BADGE_SEQUENCE, mBrandId, 0);
        int counter = PreferenceManager.getInstance().getIntValue(BADGE_COUNT, mBrandId, 0);
        validateCounterWithSequence(conversationId, sequence, counter);
    }


    /**
     * Checking if the threshold was passed
     *
     * @param brandId
     * @return true - threshold was passed, false otherwise
     */
    private boolean isBadgeThresholdWasPassed(String brandId) {

        long lastTimestamp = PreferenceManager.getInstance().getLongValue(BADGE_TIMESTAMP, brandId, 0L);

        long delta = System.currentTimeMillis() - lastTimestamp;

        // If delta is greater than threshold or lastTimestamp never set
        if ((lastTimestamp == 0) || delta > BADGE_LAST_TIMESTAMP_THRESHOLD) {
            return true;
        }
        return false;
    }
}
