package com.liveperson.messaging.network.http;

import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.liveperson.api.response.model.UserProfile;
import com.liveperson.infra.Command;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.database.DataBaseCommand;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.network.http.HttpHandler;
import com.liveperson.infra.network.http.request.HttpGetRequest;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.model.AcUserData;
import com.liveperson.messaging.model.Dialog;
import com.liveperson.messaging.model.MessagingUserProfile;

/**
 * Created by ofira on 3/17/16.
 */
public class AgentProfileRequest implements Command {

    private static final String TAG = AgentProfileRequest.class.getSimpleName();

    private static final String AC_USERS_URL = "https://%s/api/account/%s/configuration/le-users/users/%s";
    private static final int AGENT_REQUEST_TIMEOUT = 30000;

    protected final Messaging mController;

    private String mBrand;
    private String mUserId;
    private String mDialogId;
    private boolean mShouldUpdateUI;
    private ICallback<MessagingUserProfile, Exception> callback;

    public AgentProfileRequest(Messaging controller, String brand, String userId, String dialogId, boolean shouldUpdateUi) {
        mBrand = brand;
        mUserId = userId;
        mDialogId = dialogId;
        mShouldUpdateUI = shouldUpdateUi;
        mController = controller;
    }

    @Override
    public void execute() {
        String acCdnDomain = mController.mAccountsController.getServiceUrl(mBrand, ConnectionParamsCache.CSDS_AC_CDN_DOMAIN_KEY);

        if (!TextUtils.isEmpty(acCdnDomain)) {
            String cSDSRequestUrl = String.format(AC_USERS_URL, acCdnDomain, mBrand, mUserId);

            LPMobileLog.d(TAG, "Getting agent details url " + cSDSRequestUrl);
            HttpGetRequest httpGetRequest = new HttpGetRequest(cSDSRequestUrl);
            httpGetRequest.setTimeout(AGENT_REQUEST_TIMEOUT);
            httpGetRequest.setCallback( new ICallback<String, Exception>() {
                @Override
                public void onSuccess(String profile) {
                    if (!TextUtils.isEmpty(profile)) {
                        LPMobileLog.d(TAG, "onSuccess with agent details " + profile);
                        onRequestCompleted();

                        AcUserData agent = new AcUserData(profile);
                        final MessagingUserProfile userProfile = new MessagingUserProfile(agent.getFirstName(), agent.getLastName(), UserProfile.UserType.AGENT);
                        userProfile.setNickname(agent.getNickName());
                        userProfile.setBrandID(mBrand);
                        userProfile.setDescription(agent.getEmployeeId());
                        userProfile.setOriginatorID(mUserId);
                        userProfile.setAvatarUrl(agent.getPictureUrl());
                        userProfile.setEmail(agent.getEmail());

                        mController.amsUsers.updateUserProfile(userProfile);

                        if(callback != null){
                            callback.onSuccess(userProfile);
                        }

                        if (TextUtils.isEmpty(mDialogId)){
                            LPMobileLog.d(TAG, "onResult: updating agent details. without conversation id.");
                            return;
                        }
                        mController.amsDialogs.queryDialogById(mDialogId)
                                .setPostQueryOnBackground(new DataBaseCommand.QueryCallback<Dialog>() {
                                    @Override
                                    public void onResult(@Nullable Dialog data) {
                                        if (data != null && TextUtils.equals(data.getAssignedAgentId(), mUserId)) {
                                            LPMobileLog.d(TAG, "onResult: Calling agent details callback");
                                            mController.onAgentDetailsChanged(userProfile, data.isOpen());
                                        }
                                        //update all massages that filtered by brand id.
                                        //if we have open conversation with the assigned agent that we just got his profile -
                                        //update also messages that filtered by conversationID.
                                        LPMobileLog.d(TAG, "got user details (" + mUserId + ") related to dialog ID: " + mDialogId);
                                        //if we finished fetchHistory we are updated. any agent details after that moment should update ui.
                                        if (mShouldUpdateUI /* || mController.mConnectionController.getConnection(mBrand).isUpdated()*/) {
                                            LPMobileLog.d(TAG, "Updating ui with agent details! ");
                                            mController.amsMessages.updateAgentDetailsUpdated(mBrand, mDialogId);
                                        }
                                    }
                                }).execute();
                    }else{
                        if(callback != null){
                            callback.onError(new Exception("Empty response"));
                        }
                    }
                }

                @Override
                public void onError(Exception exception) {
                    LPMobileLog.e(TAG, "JSONException", exception);
                    if(callback != null){
                        callback.onError(exception);
                    }
                }
            });
            HttpHandler.execute(httpGetRequest);

            onRequestSent();
        }
    }


    private void onRequestSent() {
        if (!TextUtils.isEmpty(mDialogId)){
            mController.amsDialogs.addUpdateRequestInProgress(mDialogId);
        }

    }

    private void onRequestCompleted() {
        if (!TextUtils.isEmpty(mDialogId)){
            mController.amsDialogs.removeUpdateRequestInProgress(mDialogId);
        }
    }

    public ICallback<MessagingUserProfile, Exception> getCallback() {
        return callback;
    }

    public AgentProfileRequest setCallback(ICallback<MessagingUserProfile, Exception> callback) {
        this.callback = callback;
        return this;
    }
}
