package com.liveperson.messaging.network.http;

import android.os.Build;
import android.support.annotation.NonNull;
import android.text.TextUtils;

import com.liveperson.infra.BuildConfig;
import com.liveperson.infra.Command;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.auth.LPAuthenticationParams;
import com.liveperson.infra.auth.LPAuthenticationType;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.network.http.HttpHandler;
import com.liveperson.infra.network.http.body.HttpRequestBody;
import com.liveperson.infra.network.http.body.LPJSONObjectBody;
import com.liveperson.infra.network.http.request.HttpPostRequest;
import com.liveperson.messaging.TaskType;
import com.liveperson.messaging.commands.tasks.IdpTask;
import com.liveperson.messaging.model.AmsAccount;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

import javax.net.ssl.SSLPeerUnverifiedException;

/**
 * Created by nirni on 1/4/16.
 * <p/>
 * A request command to get a token from the IDP service
 */
public class IdpRequest implements Command {

    public static final String TAG = IdpRequest.class.getSimpleName();
    protected static final int IDP_REQUEST_TIMEOUT = 30000;

    protected static final String SIGNUP = "signup";
    protected static final String AUTHENTICATE = "authenticate";
    protected static final String DEFAULT_REDIRECT_URI = "https://liveperson.net";
    public static final String USER_EXPIRED_ERROR = "2001";
    private static final String IDP_URL = "https://%s/api/account/%s/app/default/%s?v=2.0";
    protected final AmsAccount mAccount;

    protected String mIdpDomain;
    protected String mBrandId;
    protected IdpTask.IDPExceptionICallback mCallback;
    protected LPAuthenticationParams mLPAuthenticationParams = null;
    protected String mHostVersion;
    protected List<String> mCertificates;

    /**
     * @param account
     * @param idpDomain
     * @param brandId
     * @param callback
     */
    public IdpRequest(AmsAccount account, String idpDomain, String brandId,LPAuthenticationParams lpAuthenticationParams,  String hostVersion, IdpTask.IDPExceptionICallback callback, List<String> certificates) {
        mIdpDomain = idpDomain;
        mBrandId = brandId;
        mCallback = callback;
        mHostVersion = hostVersion;
        mCertificates = certificates;
        mAccount = account;
        mLPAuthenticationParams = lpAuthenticationParams;
    }

    /**
     * Send the request and handle the response
     */
    @Override
    public void execute() {

        LPAuthenticationType authType = LPAuthenticationType.UN_AUTH;
        if (mLPAuthenticationParams != null){
            authType = mLPAuthenticationParams.getAuthType();
        }
        switch (authType) {
            case AUTH:
                //use idp version #2
                String hostAppJWT = mLPAuthenticationParams.getHostAppJWT();
                String authKey = mLPAuthenticationParams.getAuthKey();

                // If both jwt and authKey were not provided
				if (TextUtils.isEmpty(hostAppJWT) && TextUtils.isEmpty(authKey)) {
					LPMobileLog.e(TAG, "execute: both hostAppJWT and authKey are empty. Cannot continue");
					sendErrorCallback(new Exception("No JWT nor authKey was provided to AUTH authentication. Cannot authenticate"));
					return;
				}

                sendGeneralRequest(getHttpPostRequestForIDPV2(hostAppJWT, authKey));
                break;
            case UN_AUTH:
                //Handled from IdpTask -> UnAuthRequest
                break;
            default: // SIGN_UP
                //use sign up request = idp version #1
                if (BuildConfig.DEBUG) {
                    LPAuthenticationParams.printSignupDeprecationNotice();
                }
                sendGeneralRequest(getHttpPostRequestForSignUp());
                break;
        }
    }

    protected void sendErrorCallback(Exception exception) {
        LPMobileLog.e(TAG, "Sending error callback.", exception);
        if(exception instanceof SSLPeerUnverifiedException){
            mCallback.onError(TaskType.INVALID_CERTIFICATE, exception);
        }else {
            mCallback.onError(TaskType.IDP, exception);
        }
    }

    protected void sendGeneralRequest(HttpPostRequest httpPostRequest) {
        if (httpPostRequest != null){
            //Setting the certificate pinning
            httpPostRequest.setCertificatePinningKeys(mCertificates);

            LPMobileLog.d(TAG, "IDP request url : " + httpPostRequest.getUrl());

            httpPostRequest.setTimeout(IDP_REQUEST_TIMEOUT);
            // Add headers
            addHeaders(httpPostRequest);

            // Execute the request
            HttpHandler.execute(httpPostRequest);
        }
    }


    @NonNull
    private HttpPostRequest getHttpPostRequestForIDPV2(String hostAppJWT,String authKey) {
        JSONObject jsonBody = null;

        if (!TextUtils.isEmpty(hostAppJWT)) {
            jsonBody = getBodyForIDPJwtFlow(hostAppJWT);

        } else if (!TextUtils.isEmpty(authKey)) {
            jsonBody = getBodyForIDPAuthCodeFlow(authKey, mLPAuthenticationParams.getHostAppRedirectUri());

        }
        LPMobileLog.d(TAG, "Idp json body: " + jsonBody);

        String idpUrl = String.format(IDP_URL, mIdpDomain, mBrandId, AUTHENTICATE);
        final HttpPostRequest httpPostRequest = new HttpPostRequest(idpUrl);

        // Add the body if exist
        HttpRequestBody body = new LPJSONObjectBody(jsonBody);
        httpPostRequest.setBody(body);

        httpPostRequest.setCallback(new ICallback<String, Exception>() {

            @Override
            public void onSuccess(String idpResponse) {
                LPMobileLog.d(TAG, "onSuccess " + idpResponse);

                if (!TextUtils.isEmpty(idpResponse)) {

                    try {
                        JSONObject idpJson = new JSONObject(idpResponse);
                        String token = idpJson.getString("token");
                        mCallback.onSuccess(token);
                    } catch (JSONException je) {
                        LPMobileLog.d(TAG, "JSONException: ", je);
                        String errorMsg = "idp url = " + httpPostRequest.getUrl() + ". Exception " + je.getMessage();
                        sendErrorCallback(new Exception(errorMsg));
                    }
                }
            }

            @Override
            public void onError(Exception e) {
                LPMobileLog.d(TAG, "Error: idp url = " + httpPostRequest.getUrl() + ". Exception ", e);
                sendErrorCallback(e);
            }
        });


        return httpPostRequest;
    }


    @NonNull
    private JSONObject getBodyForIDPAuthCodeFlow(String authKey, String hostAppRedirectUri) {
        JSONObject jsonBody = new JSONObject();
        try {
            jsonBody.put("code", authKey);
            jsonBody.put("redirect_uri", TextUtils.isEmpty(hostAppRedirectUri) ? DEFAULT_REDIRECT_URI : hostAppRedirectUri);

        } catch (JSONException e) {
        }
        return jsonBody;
    }

    @NonNull
    protected JSONObject getBodyForIDPJwtFlow(String jwt) {

        JSONObject jsonBody = new JSONObject();
        try {
            jsonBody.put("id_token", jwt);
        } catch (JSONException e) {
        }
        return jsonBody;
    }

    @NonNull
    private HttpPostRequest getHttpPostRequestForSignUp() {

        String IDP_URL = "https://%s/api/account/%s/%s?v=1.0";
        String idpUrl = String.format(IDP_URL, mIdpDomain, mBrandId, SIGNUP);

        final HttpPostRequest httpPostRequest = new HttpPostRequest(idpUrl);
        httpPostRequest.setCallback(new ICallback<String, Exception>() {

            @Override
            public void onSuccess(String idpResponse) {
                LPMobileLog.d(TAG, "onSuccess " + idpResponse);

                if (!TextUtils.isEmpty(idpResponse)) {

                    try {
                        JSONObject idpJson = new JSONObject(idpResponse);
                        String token = idpJson.getString("jwt");
                        mCallback.onSuccess(token);
                    } catch (JSONException je) {
                        LPMobileLog.d(TAG, "JSONException: ", je);
                        String errorMsg = "idp url = " + httpPostRequest.getUrl() + ". Exception " + je.getMessage();
                        sendErrorCallback(new Exception(errorMsg));
                    }
                }
            }

            @Override
            public void onError(Exception e) {
                LPMobileLog.d(TAG, "Error: idp url = " + httpPostRequest.getUrl() + ". Exception ", e);
                sendErrorCallback(e);
            }
        });

        return httpPostRequest;

    }


    protected void addHeaders(HttpPostRequest httpPostRequest) {
        httpPostRequest.addHeader("sdkVersion", mHostVersion);
        httpPostRequest.addHeader("platform", "Android");
        httpPostRequest.addHeader("platformVer", String.valueOf(Build.VERSION.SDK_INT));
        httpPostRequest.addHeader("device", Build.MODEL);
        httpPostRequest.addHeader("applicationId", mBrandId.replace("\n", ""));
    }

}
