package com.liveperson.messaging.commands.tasks;

import android.text.TextUtils;

import com.liveperson.infra.Infra;
import com.liveperson.infra.auth.LPAuthenticationParams;
import com.liveperson.infra.auth.LPAuthenticationType;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.messaging.TaskType;
import com.liveperson.messaging.controller.AccountsController;
import com.liveperson.monitoring.MonitoringFactory;
import com.liveperson.monitoring.cache.MonitoringParamsCache;
import com.liveperson.monitoring.model.LPMonitoringIdentity;
import com.liveperson.monitoring.sdk.callbacks.EngagementCallback;
import com.liveperson.monitoring.sdk.callbacks.MonitoringErrorType;
import com.liveperson.monitoring.sdk.responses.LPEngagementResponse;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;

/**
 * Created by nirni on 1/4/16.
 * <p/>
 * A task to get the token from the IDP service
 */
public class UnAuthGetEngagementTask extends BaseAmsAccountConnectionTask {

    public static final String TAG = "UnAuthGetEngagementTask";
    private AccountsController mAccountsController;


    public UnAuthGetEngagementTask(AccountsController accountsController){
        mAccountsController = accountsController;
    }

    /**
     *
     */
    @Override
    public void execute() {

        LPAuthenticationParams auth = mAccountsController.getLPAuthenticationParams(mBrandId);
        LPMobileLog.d(TAG, "Running GetEngagement task, relevant only if auth type is UnAuth. current type: " + auth.getAuthType());
        if (auth.getAuthType() != LPAuthenticationType.UN_AUTH){
            //nothing to do here..
            mCallback.onTaskSuccess();
            return;
        }

        if (!MonitoringFactory.INSTANCE.isInitialized()) {
            //THIS SHOULD NOT HAPPEN! the only case this can be is if the host app didn't insert APP install ID - than we won't initialized monitoring.
            Exception exception = new Exception(mBrandId + ": UnAuth flow, monitoring isn't initialized. Are you missing 'app install id' in LivePerson.init? ");
            LPMobileLog.e(TAG, "MonitoringFactory is not initialized.", exception);
            mCallback.onTaskError(TaskType.IDP, exception);
            return;
        }


        if (!isConnectorIdEmpty()) {
            //Already have connector id, nothing to do here..
            LPMobileLog.d(TAG, "Type is UnAuth and we already have connector id");
            mCallback.onTaskSuccess();
            return;
        }


        MonitoringFactory.INSTANCE.getEngagement(Infra.instance.getApplicationContext(), Collections.singletonList(new LPMonitoringIdentity()), null, new EngagementCallback() {
            @Override
            public void onSuccess(@NotNull LPEngagementResponse lpEngagementResponse) {
                try {
                    String connectorId = lpEngagementResponse.getEngagementDetailsList().get(0).getConnectorId();
                     mAccountsController.getAccount(mBrandId).setConnectorId(connectorId);
                     mCallback.onTaskSuccess();
                } catch (NullPointerException e) {
                    mCallback.onTaskError(TaskType.IDP, e);
                }
            }

            @Override
            public void onError(@NotNull MonitoringErrorType errorType, @Nullable Exception exception) {
                mCallback.onTaskError(TaskType.IDP, exception);
            }
        });
    }


    private boolean isConnectorIdEmpty() {

        String connectorId = null;
        MonitoringParamsCache mMonitoringParamsCache = MonitoringFactory.INSTANCE.getMonitoring().getParamsCache();
        if (mMonitoringParamsCache != null) {
            connectorId = mMonitoringParamsCache.getConnectorId();
        }
        if (connectorId == null) {
            connectorId = mAccountsController.getAccount(mBrandId).getConnectorId();
            return TextUtils.isEmpty(connectorId);
        }
        mAccountsController.getAccount(mBrandId).setConnectorId(connectorId);
        return false;
    }

    @Override
    public String getName() {
        return TAG;
    }


}
