package com.liveperson.messaging.structuredcontent.parsers;


import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.messaging.structuredcontent.model.elements.SimpleElement;
import com.liveperson.messaging.structuredcontent.model.elements.basic.ButtonElement;
import com.liveperson.messaging.structuredcontent.model.elements.basic.ImageElement;
import com.liveperson.messaging.structuredcontent.model.elements.basic.MapElement;
import com.liveperson.messaging.structuredcontent.model.elements.basic.TextElement;
import com.liveperson.messaging.structuredcontent.model.elements.complex.CarouselElement;
import com.liveperson.messaging.structuredcontent.model.elements.complex.LayoutElement;
import com.liveperson.messaging.structuredcontent.model.elements.complex.QuickRepliesElement;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Created by nirni on 2/2/17.
 * An element parser. This parses basic elements from a json object:
 * Text,
 * Button,
 * Link,
 * Image,
 * Map,
 * Quick Reply
 */
public class ElementParser {

	public static final String TAG = ElementParser.class.getSimpleName();

	/**
	 * Parse an element from the given json
	 * @param jsonObject
	 * @return a SimpleElement of null if received json for unknown element type
	 * @throws JSONException
	 */
	public static SimpleElement parse(JSONObject jsonObject) {

		// element
		SimpleElement elementToAdd;

		try {
			String type = jsonObject.getString(ElementType.TYPE);
			switch (type) {
				case ElementType.VERTICAL:
					elementToAdd = new LayoutElement(jsonObject);
					break;
				case ElementType.HORIZONTAL:
					elementToAdd = new LayoutElement(jsonObject);
					break;
					case ElementType.TEXT:
					elementToAdd = new TextElement(jsonObject);
					break;
				case ElementType.BUTTON:
					elementToAdd = new ButtonElement(jsonObject);
					break;
				case ElementType.IMAGE:
					elementToAdd = new ImageElement(jsonObject);
					break;
				case ElementType.MAP:
					elementToAdd = new MapElement(jsonObject);
					break;
				case ElementType.CAROUSEL:
					elementToAdd = new CarouselElement(jsonObject);
					break;
				case ElementType.QUICK_REPLY:
					elementToAdd = new QuickRepliesElement(jsonObject);
					break;
//				case ElementType.LINK:
//					elementToAdd = new LinkElement(jsonObject); // Not in phase 1
//					break;
//				case ElementType.WEBVIEW:
//					elementToAdd = new WebViewElement(jsonObject); // Not in phase 1
//					break;
				default:
					return null;
			}
		} catch (JSONException e) {
			LPMobileLog.w(TAG, "parse: Error parsing json. Either missing mandatory element or incorrect json. jsonObject: " + jsonObject, e);
			return null;
		}

		return elementToAdd;
	}

	public static boolean isStructuredContentEmpty(JSONObject jsonObject) {
		try {
			String type = jsonObject.getString(ElementType.TYPE);
			if (ElementType.VERTICAL.equals(type) || ElementType.HORIZONTAL.equals(type)) {
				JSONArray elements = jsonObject.getJSONArray(ElementType.ELEMENTS);
				if (elements.length() == 0) {
					return true;
				}
			}
		} catch (JSONException e) {
			LPMobileLog.w(TAG, "parse: Error parsing json. Either missing mandatory element or incorrect json. jsonObject: " + jsonObject, e);
		}
		return false;
	}
}
