package com.liveperson.messaging.network.socket.requests;

import android.text.TextUtils;

import com.liveperson.api.request.ReqBody;
import com.liveperson.api.response.events.ContentEventNotification;
import com.liveperson.api.request.QueryMessages;
import com.liveperson.infra.database.DataBaseCommand;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.network.socket.BaseResponseHandler;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.SocketTaskType;
import com.liveperson.messaging.commands.tasks.BaseAmsSocketConnectionCallback;
import com.liveperson.api.request.BaseAMSSocketRequest;
import com.liveperson.messaging.model.Conversation;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * Created by Ilya Gazman on 11/19/2015.
 */
public class QueryMessagesRequest extends BaseAMSSocketRequest<QueryMessages.Response, QueryMessagesRequest> {

    private static final String TAG = QueryMessagesRequest.class.getSimpleName();
    private final String mBrandId;
    private final Messaging mController;
    private String mDialogId;
    private int mMaxSize = -1;
    private int mNewerThanSequenceId;
    private int mOlderThanSequenceId = -1;
    private String mTargetId;
    private BaseAmsSocketConnectionCallback mConnectionCallback;
    private boolean mShouldUpdateUI = true;

    public QueryMessagesRequest(Messaging controller, String brandId, String targetId, String dialogId, int newerThanSequenceId) {
        super(controller.mAccountsController.getConnectionUrl(brandId));

        mController = controller;
        mBrandId = brandId;
        mTargetId = targetId;
        mDialogId = dialogId;
        mNewerThanSequenceId = newerThanSequenceId;
    }

    public void setResponseCallBack(BaseAmsSocketConnectionCallback callBack) {
        mConnectionCallback = callBack;
    }

    public QueryMessagesRequest setOlderThanSequenceId(int olderThanSequenceId) {
        mOlderThanSequenceId = olderThanSequenceId;
        return this;
    }

    public String getDialogId() {
        return mDialogId;
    }

    public QueryMessagesRequest setMaxSize(int maxSize) {
        mMaxSize = maxSize;
        return this;
    }

    public void setShouldUpdateUI(boolean shouldUpdateUI) {
        mShouldUpdateUI = shouldUpdateUI;
    }

    @Override
    protected String getData() {
        onRequestSent();
        return new QueryMessages(mDialogId, mMaxSize, mOlderThanSequenceId, mNewerThanSequenceId + 1).toJsonString(getRequestId());
    }

    private void onRequestSent() {
        //mController.amsConversations.addUpdateRequestInProgress(mDialogId);
        mController.amsDialogs.addUpdateRequestInProgress(mDialogId);

    }
    private void onRequestCompleted() {
        //mController.amsConversations.removeUpdateRequestInProgress(mConversationId);
        mController.amsDialogs.removeUpdateRequestInProgress(mDialogId);
    }

    @Override
    protected String getRequestName() {
        return TAG;
    }


    @Override
    protected BaseResponseHandler<QueryMessages.Response, QueryMessagesRequest> getResponseHandler() {
        return new BaseResponseHandler<QueryMessages.Response, QueryMessagesRequest>() {

            @Override
            public BaseResponseHandler getResponseByExpectedType(String expectedType) {
                BaseResponseHandler supportedResponseHandler = getSupportedResponseHandler(expectedType);
                if (supportedResponseHandler != null){
                    return supportedResponseHandler;
                }
                return this;
            }

            @Override
            public String getAPIResponseType() {
                return QueryMessages.Response.QUERY_MESSAGES_RESPONSE_TYPE;
            }

            @Override
            protected BaseResponseHandler getSupportedResponseHandler(String messageType) {
                if (TextUtils.equals(messageType, ReqBody.StringResp.REQ_BODY_RESPONSE_TYPE)){
                    return new BaseResponseHandler<ReqBody.StringResp, NewConversationRequest>() {

                        @Override
                        public String getAPIResponseType() {
                            return ReqBody.StringResp.REQ_BODY_RESPONSE_TYPE;
                        }

                        @Override
                        protected ReqBody.StringResp parse(JSONObject jsonObject) throws JSONException {
                            return new ReqBody.StringResp(jsonObject);
                        }

                        @Override
                        protected boolean handle(ReqBody.StringResp response) {
                            LPMobileLog.d(TAG, "Received String response (" + response.code + ").");
                            if (response.code > 201) {
                                LPMobileLog.w(TAG, "Received bad query response (" + response.code + ").");
                                if (mConnectionCallback != null){
                                    mConnectionCallback.onTaskError(SocketTaskType.QUERY_MESSAGES, new Exception("response = " + response));
                                }
                                return true;
                            }
                            return false;
                        }
                    };
                }
                return super.getSupportedResponseHandler(messageType);
            }

            @Override
            protected void giveUp() {
                super.giveUp();
                LPMobileLog.i(TAG, getRequestId() + ": Request lost (socket closed) for query request.");
                if (mConnectionCallback != null){
                    mConnectionCallback.onTaskError(SocketTaskType.QUERY_MESSAGES, new Exception("CloseSocket"));
                }

            }

            @Override
            protected QueryMessages.Response parse(JSONObject jsonObject) throws JSONException {
                return new QueryMessages.Response(jsonObject);
            }

            @Override
            protected boolean handle(final QueryMessages.Response response) {
                LPMobileLog.d(TAG, "Received query messages response");

                final String originatorId = mController.getOriginatorId(mTargetId);
                final long clockDiff = mController.mConnectionController.getClockDiff(mBrandId);

                if (response.getBody().size() == 0){
                    LPMobileLog.d(TAG, "No messages in query response.");
                    //updating last server sequence on conversation
                    if (mOlderThanSequenceId == -1){
                        //empty conversation. need to update last sequence in DB to 0.
                        DataBaseCommand<Boolean> command = mController.amsDialogs.
                                updateLastServerSequenceByDialogId(mDialogId, 0, 1, mShouldUpdateUI);
                        if (command != null){
                            command.execute();
                        }

                    }
                    onHandleResponseCompletedSuccessfully();
                    return true;
                }

                //get last message from response for its sequence.
                ContentEventNotification lastReceivedEvent = response.getBody().get(response.getBody().size() - 1);
                if (lastReceivedEvent != null){
                    mOlderThanSequenceId = lastReceivedEvent.sequence;
                }

                LPMobileLog.d(TAG, getRequestId() + " Last sequence event received in query messages response - "+ mOlderThanSequenceId);

                //updating last server sequence on conversation
                DataBaseCommand<Boolean> command = mController.amsDialogs.
                        updateLastServerSequenceByDialogId(mDialogId, mOlderThanSequenceId, mOlderThanSequenceId - mNewerThanSequenceId, mShouldUpdateUI);
                if (command == null) {
                    LPMobileLog.d(TAG, "Got unexpected QueryMessages!! query results last sequence = " + mOlderThanSequenceId + ". ignoring query results!");
                    onHandleResponseCompletedFailed();
                } else {
                    command.setPostQueryOnBackground(new DataBaseCommand.QueryCallback<Boolean>() {
                        @Override
                        public void onResult(Boolean isEventExpected) {
                            if (isEventExpected) {
                                //adding messages only if sequence are matching
                              /*  mController.amsMessages.addMultipleMessages(response,
                                        originatorId, mBrandId, mTargetId, mConversationId, clockDiff, mShouldUpdateUI).setPostQueryOnBackground(new DataBaseCommand.QueryCallback<Void>() {
                                    @Override
                                    public void onResult(Void data) {
                                        onHandleResponseCompletedSuccessfully();
                                    }
                                }).execute();*/

                                //sending update on messages status.
                                mController.amsMessages.sendReadAckOnMessages(mBrandId, mTargetId, originatorId);

                            } else {
                                LPMobileLog.d(TAG, "Got unexpected QueryMessages!! query results last sequence = " + mOlderThanSequenceId + ". ignoring query results!");
                                onHandleResponseCompletedFailed();

                            }
                        }
                    });
                    command.execute();
                }
                return true;
            }
        };
    }

    private void onHandleResponseCompletedFailed() {
        onRequestCompleted();
        if (mConnectionCallback != null){
            mConnectionCallback.onTaskError(SocketTaskType.QUERY_MESSAGES, new Exception("unexpected QueryMessages"));
        }
    }

    private void onHandleResponseCompletedSuccessfully() {
        onRequestCompleted();
        if (mConnectionCallback != null){
            mConnectionCallback.onTaskSuccess();
        }
    }
}

