package com.liveperson.messaging.commands;

import com.liveperson.api.response.types.CloseReason;
import com.liveperson.infra.Command;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.network.socket.SocketManager;
import com.liveperson.messaging.MessagingFactory;
import com.liveperson.messaging.model.AmsConversations;
import com.liveperson.messaging.model.Conversation;
import com.liveperson.messaging.model.ConversationData;
import com.liveperson.messaging.model.Dialog;
import com.liveperson.messaging.network.socket.requests.ResolveConversationRequest;

import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;

/**
 * Created by shiranr on 18/11/2015.
 */
public class ResolveConversationCommand implements Command {
    private static final String TAG = "ResolveConversationCommand";
    private final AmsConversations mAmsConversations;
    //indicate if to send resolve conversation request, or just close the conversation in DB without sending resolve request on socket.
    private boolean mOfflineMode = false;

    private String mTargetId;
    private String mSocketUrl;
    private ICallback<String, Throwable> callback;

    /**
     * Send request to close Conversation.
     * @param amsConversations
     * @param targetId
     * @param socketUrl - accept null only when Offline mode is enabled (see #setOfflineMode)
     */
    public ResolveConversationCommand(AmsConversations amsConversations, String targetId, @Nullable String socketUrl) {
        mAmsConversations = amsConversations;
        mSocketUrl = socketUrl;
        mTargetId = targetId;
    }

    /**
     * Use this in case you want to just close the conversation in DB without sending resolve request on socket.
     */
    public void setOfflineMode(boolean offline) {
        mOfflineMode = offline;
    }

    @Override
    public void execute() {
        if (!mAmsConversations.isConversationActive(mTargetId)) {
            LPLog.INSTANCE.w(TAG, "No open conversation found. aborting resolve conversation command");
            return;
        }
        final Conversation conversation = mAmsConversations.getConversation(mTargetId);

        if (mOfflineMode) {
            final ConversationData conversationData = new ConversationData(mTargetId, conversation);
            conversationData.closeReason = CloseReason.CONSUMER;
            conversationData.endTs = System.currentTimeMillis();
            final ArrayList<Dialog> dialogs = new ArrayList<>();

            mAmsConversations.updateClosedConversation(conversationData, false).setPreQueryOnBackground(() -> {
                ArrayList<Dialog> _dialogs = MessagingFactory.getInstance().getController().amsDialogs.queryDialogsByConversationId(conversationData.conversationId).executeSynchronously();
                dialogs.addAll(_dialogs);
                conversationData.setDialogs(dialogs);
            }).setPostQueryOnBackground(conversationResult -> {
                //return null only if dialog was already closed.
                if (conversationResult != null) {
                    for (final Dialog dialog : dialogs) {
                        if (!dialog.isOpen()) { // No need to close an already closed dialog
                            continue;
                        }

                        // We're doing this synchronously since we're already in a background thread
                        Dialog dialogResult = MessagingFactory.getInstance().getController().amsDialogs.updateClosedDialog(conversationData, dialog, false).executeSynchronously();
	                    LPLog.INSTANCE.d(TAG, "Updating closed dialog. " + dialogResult.getDialogId());
                    }
                }
                if (callback != null) {
                    callback.onSuccess("");
                }
            }).execute();

        } else {
            ResolveConversationRequest resolveConversationRequest = new ResolveConversationRequest(mSocketUrl, conversation.getConversationId());
            resolveConversationRequest.setResponseCallBack(callback);
            SocketManager.getInstance().send(resolveConversationRequest);
        }
    }

	public void setCallback(ICallback<String, Throwable> callback) {
		this.callback = callback;
	}

}
