package com.liveperson.messaging.model;

import android.content.ContentValues;
import android.database.Cursor;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.liveperson.api.response.types.CSAT;
import com.liveperson.api.response.types.CloseReason;
import com.liveperson.api.response.types.ConversationState;
import com.liveperson.api.response.types.TTRType;
import com.liveperson.infra.Clearable;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.database.BaseDBRepository;
import com.liveperson.infra.database.DataBaseCommand;
import com.liveperson.infra.database.DataBaseExecutor;
import com.liveperson.infra.database.tables.ConversationsTable;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.messaging.R;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDown;
import com.liveperson.infra.utils.LocalBroadcast;
import com.liveperson.infra.utils.Queue;
import com.liveperson.messaging.Messaging;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * Created by Ilya Gazman on 11/10/2015.
 * <p/>
 * Conversation model
 */
public class AmsConversations extends BaseDBRepository implements ShutDown, Clearable {

    private static final String TAG = "AmsConversations";

    private static final String BROADCAST_UPDATE_CONVERSATION = "BROADCAST_UPDATE_CONVERSATION";
    public static final String BROADCAST_UPDATE_CSAT_CONVERSATION = "BROADCAST_UPDATE_CSAT_CONVERSATION";
    public static final String BROADCAST_UPDATE_CONVERSATION_CLOSED = "BROADCAST_UPDATE_CONVERSATION_CLOSED";
    private static final String BROADCAST_UPDATE_NEW_CONVERSATION_MSG = "BROADCAST_UPDATE_NEW_CONVERSATION_MSG";
    private static final String BROADCAST_UPDATE_UNREAD_MSG = "BROADCAST_UPDATE_UNREAD_MSG";
    public static final String BROADCAST_UPDATE_CONVERSATION_TTR = "BROADCAST_UPDATE_CONVERSATION_TTR";
    public static final String BROADCAST_UPDATE_FORM_URL = "BROADCAST_UPDATE_FORM_URL";
    private static final String BROADCAST_CONVERSATION_FRAGMENT_CLOSED = "BROADCAST_CONVERSATION_FRAGMENT_CLOSED";

    public static final String DELAY_TILL_WHEN = "DELAY_TILL_WHEN";
    public static final String DELAY_TILL_WHEN_CHANGED = "DELAY_TILL_WHEN_CHANGED";
    public static final String BROADCAST_UPDATE_CONVERSATION_OFF_HOURS = "BROADCAST_UPDATE_CONVERSATION_OFF_HOURS";

    public static final String KEY_CONVERSATION_ID = "CONVERSATION_ID";
    public static final String KEY_CONVERSATION_TTR_TIME = "CONVERSATION_TTR_TIME";
    public static final String KEY_CONVERSATION_TARGET_ID = "CONVERSATION_TARGET_ID";
    public static final String KEY_CONVERSATION_STATE = "CONVERSATION_STATE";
    public static final String KEY_CONVERSATION_SHOWED_CSAT = "CONVERSATION_SHOWED_CSAT";
    public static final String KEY_CONVERSATION_ASSIGNED_AGENT = "CONVERSATION_ASSIGNED_AGENT";

    public static final String KEY_WELCOME_CONVERSATION_ID = "WELCOME_CONVERSATION_ID";

    protected final Messaging mController;
    private Map<String, Conversation> mConversationsByAccountId = new HashMap<>();
    private Map<String, Conversation> mConversationsByServerId = new HashMap<>();
    private Queue<Long> pendingConversationRequestId;

    /**
     * Creates new AmsConversations
     *
     * @param controller
     */
    public AmsConversations(Messaging controller) {
        super(ConversationsTable.TABLE_NAME);
        pendingConversationRequestId = new Queue<>();
        mController = controller;
    }

    /**
     * @return active conversation
     */
    public Conversation getConversation(String brandId) {
        return getConversationFromTargetIdMap(brandId);
    }

    public DataBaseCommand<Conversation> getActiveConversation(final String targetId) {
        return new DataBaseCommand<>(() -> {
            Conversation conversation = mConversationsByAccountId.get(targetId);
            if (conversation != null) {
                if (conversation.getState() == ConversationState.OPEN || conversation.getState() == ConversationState.PENDING){
                    return conversation;
                }
                return null;
            }
            Cursor cursor = getDB().query(
                    null,
                    ConversationsTable.Key.TARGET_ID + "=? and " + ConversationsTable.Key.STATE + " in (?, ?) " +
                            "order by _id desc limit 1",
                    new String[]{targetId, String.valueOf(ConversationState.OPEN.ordinal()), String.valueOf(ConversationState.PENDING.ordinal())}, null, null, null);

            return readConversation(cursor);
        });
    }

    /**
     * Get conversation from the DB or from the local Map based on the conversation ID
     * @param targetId - brand id
     * @param convId - conversation id that defines that conversation
     * @return
     */
    public DataBaseCommand<Conversation> getConversationById(final String targetId, final String convId) {
        return new DataBaseCommand<>(() -> {
            Conversation conversation = mConversationsByServerId.get(convId);
            if (conversation != null) {
                return conversation;
            }
            Cursor cursor = getDB().query(null,
                    ConversationsTable.Key.TARGET_ID + "=? and " + ConversationsTable.Key.CONVERSATION_ID + "=? ",
                    new String[]{targetId, convId}, null, null, null);

            return readConversation(cursor);
        });
    }


    /**
     * Set active conversation
     *
     * @param conversation
     */
    private void setConversation(String brandId, Conversation conversation) {
        if (getConversationFromTargetIdMap(brandId) == null && conversation == null) {
            return;
        }
        addConversationToMaps(brandId, conversation);
        sendUpdateStateIntent(conversation);
    }

    /**
     * @return If there is conversation and it's open
     */
    public boolean isConversationActive(String targetId) {
        final Conversation conversation = getConversationFromTargetIdMap(targetId);
        return (conversation != null && conversation.getState() == ConversationState.OPEN);
    }

    /**
     * Load the last active conversation
     *
     * @param targetId
     */
    public void loadOpenConversationForBrand(final String targetId) {
        DataBaseExecutor.execute(() -> {
            Cursor cursor = getDB().query(
                    null,
                    ConversationsTable.Key.TARGET_ID + "=? and " + ConversationsTable.Key.STATE + " in (?, ?) " +
                            "order by _id desc limit 1",
                    new String[]{targetId, String.valueOf(ConversationState.OPEN.ordinal()), String.valueOf(ConversationState.PENDING.ordinal())}, null, null, null);

            Conversation conversation = readConversation(cursor);

            if (conversation != null) {
                LPLog.INSTANCE.d(TAG, "Setting current conversation for " + targetId + ". conversation id = " + conversation.getConversationId());
                setConversation(targetId, conversation);
            }
        });
    }

    private static Conversation readConversation(Cursor cursor) {
        if (cursor != null) {
            try {
                if (cursor.moveToFirst()) {
                    return new Conversation(cursor);
                }
            } finally {
                cursor.close();
            }
        }
        return null;
    }

    /**
     * Update conversation
     *
     * @param data a bundle of properties to update
     */
    public void updateCurrentConversation(final ConversationData data) {
        final Conversation conversation = getConversationFromTargetIdMap(data.targetId);
        final ContentValues conversationValues = new ContentValues();
        if (conversation.getState() != data.state) {
            conversation.setState(data.state);
            conversationValues.put(ConversationsTable.Key.STATE, (data.state != null ? data.state.ordinal() : -1));
        }
        if (conversation.getConversationTTRType() != data.conversationTTRType) {
            conversation.setConversationTTRType(data.conversationTTRType);
            switch (data.conversationTTRType) {
                case URGENT:
                    mController.mEventsProxy.onConversationMarkedAsUrgent();
                    break;
                case NORMAL:
                    mController.mEventsProxy.onConversationMarkedAsNormal();
                    break;
            }
            conversationValues.put(ConversationsTable.Key.TTR_TYPE, data.conversationTTRType.ordinal());
        }

        if (conversation.getRequestId() != data.requestId) {
            conversation.setRequestId(data.requestId);
            conversationValues.put(ConversationsTable.Key.REQUEST_ID, data.requestId);
        }
        if (!TextUtils.equals(conversation.getConversationId(), conversation.getConversationId())) {
            conversation.setConversationId(data.conversationId);
            conversationValues.put(ConversationsTable.Key.CONVERSATION_ID, data.conversationId);
        }
//        if (!TextUtils.equals(conversation.getAssignedAgentServerId(), data.assignedAgentServerId)) {
//            conversation.setAssignedAgentId(data.assignedAgentServerId);
//            conversationValues.put(ConversationsTable.Key.ASSIGNED_AGENT_ID, data.assignedAgentServerId);
//        }

        if (conversation.getUnreadMessages() != data.unreadMessages) {
            conversation.setUnreadMessages(data.unreadMessages);
            conversationValues.put(ConversationsTable.Key.UNREAD_MESSAGES_COUNT, data.unreadMessages);
            sendUpdateUnreadMsgIntent(conversation);
        }
        if (conversation.getStartTimestamp() != data.startTs) {
            conversation.setStartTimestamp(data.startTs);
            conversationValues.put(ConversationsTable.Key.START_TIMESTAMP, data.startTs);
        }
        //LastServerSequence should be updated only by updateLastServerSequence(int)

        if (conversationValues.size() > 0) {
            DataBaseExecutor.execute(() -> {
                getDB().update(conversationValues, ConversationsTable.Key.CONVERSATION_ID + "=?",
                        new String[]{String.valueOf(data.conversationId)});
                sendUpdateStateIntent(conversation);
            });
        }
    }


    public DataBaseCommand<Conversation> updateCurrentConversationServerID(final ConversationData data) {
        final Conversation conversation = getConversationFromServerIdMap(Conversation.TEMP_CONVERSATION_ID);
        removeTempConversationFromMaps(data.targetId);
        conversation.setState(data.state);
        conversation.setConversationTTRType(data.conversationTTRType);
        conversation.setRequestId(data.requestId);
        conversation.setConversationId(data.conversationId);
        addConversationToMaps(data.targetId, conversation);

        //LastServerSequence should be updated only by updateLastServerSequence(int)
        return new DataBaseCommand<>(() -> {
            getDB().insert(getContentValues(conversation));

            sendUpdateStateIntent(conversation);
	        LPLog.INSTANCE.d(TAG, "Finished updating conversation with server id");
            return conversation;
        });
    }

    public DataBaseCommand<Void> deleteTempConversationServerID() {
        return new DataBaseCommand<>(() -> {
            getDB().removeAll(ConversationsTable.Key.CONVERSATION_ID + "=?", new String[]{Conversation.TEMP_CONVERSATION_ID});
	        LPLog.INSTANCE.d(TAG, "Finished removing temp conversation");
            return null;
        });
    }

    /**
     * Creates a pending conversation
     *
     * @param targetId
     * @param requestId
     */
    public void createPendingConversation(final String targetId, final String brandId, final long requestId) {
        createConversation(targetId, brandId, ConversationState.PENDING, requestId);
    }

    /**
     * Creates a queued conversation
     *
     * @param targetId
     * @param requestId
     */
    public void createQueuedConversation(final String targetId, final String brandId, final long requestId) {
        createConversation(targetId, brandId, ConversationState.QUEUED, requestId);
    }

    private void createConversation(final String targetId, final String brandId, final ConversationState conversationState, final long requestId){
        final Conversation conversation = new Conversation(targetId, brandId);
        conversation.setConversationId(Conversation.TEMP_CONVERSATION_ID);
        conversation.setState(conversationState);
        conversation.setConversationTTRType(TTRType.NORMAL);
        conversation.setRequestId(requestId);
        addConversationToMaps(targetId, conversation);

        DataBaseExecutor.execute(() -> {
            ContentValues conversationValues = new ContentValues();
            conversationValues.put(ConversationsTable.Key.BRAND_ID, conversation.getBrandId());
            conversationValues.put(ConversationsTable.Key.TARGET_ID, conversation.getTargetId());
            conversationValues.put(ConversationsTable.Key.CONVERSATION_ID, conversation.getConversationId());
            conversationValues.put(ConversationsTable.Key.STATE, conversation.getState().ordinal());
            conversationValues.put(ConversationsTable.Key.TTR_TYPE, conversation.getConversationTTRType().ordinal());
            conversationValues.put(ConversationsTable.Key.REQUEST_ID, conversation.getRequestId());
            conversationValues.put(ConversationsTable.Key.UNREAD_MESSAGES_COUNT, -1);
            conversationValues.put(ConversationsTable.Key.START_TIMESTAMP, System.currentTimeMillis());
            getDB().insert(conversationValues);
	        LPLog.INSTANCE.d(TAG, "create New Pending Conversation - tempID = " + conversation.getConversationId());
            sendUpdateStateIntent(conversation);
        });
    }

    /**
     * Create new current conversation
     *
     * @param data a bundle of properties to update
     */
    public Conversation createNewCurrentConversation(final ConversationData data) {
        final Conversation conversation = new Conversation(data);
        addConversationToMaps(data.targetId, conversation);

        DataBaseExecutor.execute(() -> {
            getDB().insert(getContentValues(conversation));
	        LPLog.INSTANCE.i(TAG, "Create new current conversation - conversation Id = " + data.conversationId);
            //update(conversation.getBrandID());
            sendUpdateStateIntent(conversation);
            sendUpdateNewConversationIntent(conversation);
        });
        return conversation;
    }

    public DataBaseCommand<Conversation> updateClosedConversation(final ConversationData conversationData, boolean shouldUpdateUI) {
        final String brandId = conversationData.targetId;
        final Conversation conversation = getConversationFromTargetIdMap(brandId);

        shouldUpdateUI = shouldNotifyUIConversationClosed(conversationData.closeReason, conversationData.brandId, conversationData.endTs, shouldUpdateUI);
        final CSAT.CSAT_SHOW_STATUS isShowedCsat = calculateShowedCsat(conversationData, shouldUpdateUI);

        if (conversation != null && conversationData.conversationId.equals(conversation.getConversationId())) {
            //updating current conversation
            if (conversation.getState() != ConversationState.CLOSE) {
                LPLog.INSTANCE.i(TAG, "Closing conversation " + conversationData.conversationId +
                        ", close reason:" + conversationData.closeReason + ", close ts:" + conversationData.endTs);
                conversation.setState(ConversationState.CLOSE);
                conversation.getTTRManager().cancelAll();
                conversation.setCloseReason(conversationData.closeReason);
                conversation.setEndTimestamp(conversationData.endTs);
            }

            conversation.setShowedCSAT(isShowedCsat);
        }

        final boolean finalUpdateUI = shouldUpdateUI;
        return new DataBaseCommand<>(() -> {
            Cursor cursor = getDB().query(null, ConversationsTable.Key.CONVERSATION_ID + "=?", new String[]{conversationData.conversationId}, null, null, null);
            Conversation conversation1 = readConversation(cursor);

            if (conversation1 == null) {
	            LPLog.INSTANCE.i(TAG, "Old conversation " + conversationData.conversationId + " does not exist in DB. creating new one closed conversation"
                        + ", close reason:" + conversationData.closeReason + ", close ts:" + conversationData.endTs);
                conversation1 = new Conversation(conversationData);
                conversation1.setConversationId(conversationData.conversationId);
                conversation1.setConversationTTRType(conversationData.conversationTTRType);
                conversation1.setCloseReason(conversationData.closeReason);
                conversation1.setEndTimestamp(conversationData.endTs);
                conversation1.setState(ConversationState.CLOSE);
                conversation1.setShowedCSAT(isShowedCsat);
                getDB().insert(getContentValues(conversation1));

            } else {
                String assignedAgentId = conversationData.getAssignedAgentId();
                if (conversation1.getState() != ConversationState.CLOSE) {
	                LPLog.INSTANCE.d(TAG, "Closing current conversation.. ");
                    conversation1.setState(ConversationState.CLOSE);
                    conversation1.setCloseReason(conversationData.closeReason);
                    conversation1.setEndTimestamp(conversationData.endTs);
                    conversation1.setShowedCSAT(isShowedCsat);
                    getDB().update(getContentValues(conversation1), ConversationsTable.Key.CONVERSATION_ID + "=?",
                            new String[]{String.valueOf(conversation1.getConversationId())});

                    if (finalUpdateUI) {
                        sendUpdateCSATConversationIntent(conversation1, assignedAgentId);
                    }
                    // Send intent that we closing conversation
                    sendConversationClosedIntent(conversation1, assignedAgentId);
                } else {
                    if (conversation1.isShowedCSAT() != isShowedCsat){
                        conversation1.setShowedCSAT(isShowedCsat);
                        getDB().update(getContentValues(conversation1), ConversationsTable.Key.CONVERSATION_ID + "=?",
                                new String[]{String.valueOf(conversation1.getConversationId())});
                    }
                    //if conversation already exists & closed - return null
                    if (finalUpdateUI) {
                        sendUpdateCSATConversationIntent(conversation1, assignedAgentId);
                    }
                    sendConversationClosedIntent(conversation1, assignedAgentId);
                    return null;
                }

            }

            return conversation1;
        });
    }

    /**
     * Notify the ui via Intent if there is a closed conversation that we didn't show CSAT for.
     * Sending an intent only in case the CSAT expiration time didn't pass and only if it didn't auto closed.
     * @param brandId
     */
    public void notifyClosedConversationFromDB(final String brandId) {
        DataBaseExecutor.execute(() -> {
	        LPLog.INSTANCE.d(TAG, "notifyClosedConversationFromDB");

            //Select the newest closed conversation.
            Cursor c = getDB().rawQuery("select * from " + ConversationsTable.TABLE_NAME
                            + " where " +
                            ConversationsTable.Key.END_TIMESTAMP + " = ( SELECT MAX ( " +
                            ConversationsTable.Key.END_TIMESTAMP +
                            " ) from  " + ConversationsTable.TABLE_NAME +
                            " where " + ConversationsTable.Key.BRAND_ID + " = ? and " + ConversationsTable.Key.STATE + " = ? )"
                    , brandId, ConversationState.CLOSE.ordinal());


            if (c != null) {
                try {
                    Conversation conversation = readConversation(c);
                    //if we didn't show CSAT for that conversation.
                    if (conversation != null) {
	                    LPLog.INSTANCE.d(TAG, "notifyClosedConversationFromDB : " + conversation.getConversationId());
	                    LPLog.INSTANCE.d(TAG, "notifyClosedConversationFromDB conversation.isShowedCSAT(): " + conversation.isShowedCSAT());

                        String assignedAgentId ;

                        Dialog currentDialog = mController.amsDialogs.getActiveDialog();

                        if (currentDialog == null) {
                            ArrayList<Dialog> dialogs = mController.amsDialogs.getDialogsByConversationId(conversation.getConversationId());
                            if (dialogs.size() != 0) {
                                currentDialog = dialogs.get(0);
                            }
                        }

                        assignedAgentId = (currentDialog != null) ? currentDialog.getAssignedAgentId() : "";

                        if (conversation.isShowedCSAT() == CSAT.CSAT_SHOW_STATUS.NOT_SHOWN) {
                            //notify only in case the CSAT expiration time didn't pass and only if it didn't auto closed.
	                        LPLog.INSTANCE.d(TAG, "notifyClosedConversationFromDB : " + conversation.getConversationId());
                            if (shouldNotifyUIConversationClosed(conversation.getCloseReason(), brandId, conversation.getEndTimestamp(), true)) {
                                sendUpdateCSATConversationIntent(conversation, assignedAgentId);
                            }
                        }
                        sendConversationClosedIntent(conversation, assignedAgentId);
                    }
                } finally {
                    c.close();
                }
            }
        });
    }
    private boolean shouldNotifyUIConversationClosed(CloseReason closeReason, String brandId, long closeTS, boolean updateUI) {
        if (updateUI){
            //if close reason is by System (auto close) no need to update ui.
            if (isAutoClose(closeReason)){
                LPLog.INSTANCE.d(TAG, "Updating closed conversation. Close Reason = System. do not update UI.");
                updateUI = false;
            }else{
                // Check expiry :
                // if we got to a decision we need to update ui -
                // we need to check if the closed conversation didn't expired.
                updateUI = checkExpiry(brandId, closeTS, updateUI);
            }
        }
        return updateUI;
    }
    private boolean isAutoClose(CloseReason closeReason) {
        return closeReason == CloseReason.TIMEOUT || closeReason == CloseReason.SYSTEM;
    }

    private boolean checkExpiry(String brandId, long closeTS, boolean updateUI) {
        int expirationInMinutes = Configuration.getInteger(R.integer.csatSurveyExpirationInMinutes);
        if (expirationInMinutes != 0){ // 0 means no expiration
            final long clockDiff = mController.mConnectionController.getClockDiff(brandId);
            long now = System.currentTimeMillis();
            long endTime = clockDiff + closeTS;
            long expirationInMillis = TimeUnit.MINUTES.toMillis(expirationInMinutes);
            if (now - endTime > expirationInMillis){
                LPLog.INSTANCE.d(TAG, "Closing conversation- time expired for CSAT. endTime = " + endTime + " expirationInMinutes = " + expirationInMinutes);
                updateUI = false;
            }

        }
        return updateUI;
    }

    public static CSAT.CSAT_SHOW_STATUS calculateShowedCsat(ConversationData conversationData, boolean updateUI) {
        CSAT.CSAT_SHOW_STATUS isShowedCsat = CSAT.CSAT_SHOW_STATUS.NOT_SHOWN;

        if (updateUI) {
            if (conversationData.csat != null) {
                //update new data if exists, if not - keep the old value
                isShowedCsat = conversationData.csat.isShowedCsat();
            }
        } else {
            //if no need to update UI we need to disable the option to show CSAT.
           isShowedCsat = CSAT.CSAT_SHOW_STATUS.NO_NEED_TO_SHOW;
        }

        return isShowedCsat;
    }

    private static void sendConversationClosedIntent(Conversation conversation, String assignedAgentId) {
        Bundle extras = new Bundle();
        extras.putString(KEY_CONVERSATION_TARGET_ID, conversation.getTargetId());
        extras.putString(KEY_CONVERSATION_ID, conversation.getConversationId());
        extras.putString(KEY_CONVERSATION_ASSIGNED_AGENT, assignedAgentId);

        LPLog.INSTANCE.d(TAG, "Sending Conversation autoClosed update with : " + extras);
        LocalBroadcast.sendBroadcast(BROADCAST_UPDATE_CONVERSATION_CLOSED, extras);
    }

    private static void sendUpdateStateIntent(Conversation conversation) {
        Bundle extras = new Bundle();
        extras.putString(KEY_CONVERSATION_TARGET_ID, conversation.getTargetId());
        extras.putString(KEY_CONVERSATION_ID, conversation.getConversationId());
        extras.putInt(KEY_CONVERSATION_STATE, conversation.getState().ordinal());
//        extras.putString(KEY_CONVERSATION_ASSIGNED_AGENT, conversation.getAssignedAgentServerId());
        LPLog.INSTANCE.d(TAG, "Sending Conversation update with : " + extras);
        LocalBroadcast.sendBroadcast(BROADCAST_UPDATE_CONVERSATION, extras);
    }

    private static void sendUpdateCSATConversationIntent(Conversation conversation, String assignedAgentId) {
        Bundle extras = new Bundle();
        extras.putString(KEY_CONVERSATION_TARGET_ID, conversation.getTargetId());
        extras.putString(KEY_CONVERSATION_ID, conversation.getConversationId());
        extras.putInt(KEY_CONVERSATION_STATE, conversation.getState().ordinal());
        extras.putString(KEY_CONVERSATION_ASSIGNED_AGENT, assignedAgentId);
        extras.putInt(KEY_CONVERSATION_SHOWED_CSAT, conversation.isShowedCSAT().getValue());
        LPLog.INSTANCE.d(TAG, "Sending Conversation CSAT update with : " + extras);
        LocalBroadcast.sendBroadcast(BROADCAST_UPDATE_CSAT_CONVERSATION, extras);
    }

    private static void sendUpdateNewConversationIntent(Conversation conversation) {
        Bundle extras = new Bundle();
        extras.putString(KEY_CONVERSATION_TARGET_ID, conversation.getTargetId());
        extras.putString(KEY_CONVERSATION_ID, conversation.getConversationId());
        LPLog.INSTANCE.d(TAG, "Sending Conversation update with : " + extras);
        LocalBroadcast.sendBroadcast(BROADCAST_UPDATE_NEW_CONVERSATION_MSG, extras);
    }

    private static void sendUpdateUnreadMsgIntent(Conversation conversation) {
        Bundle extras = new Bundle();
        extras.putString(KEY_CONVERSATION_TARGET_ID, conversation.getTargetId());
        LPLog.INSTANCE.d(TAG, "Sending Conversation update with : " + extras);
        LocalBroadcast.sendBroadcast(BROADCAST_UPDATE_UNREAD_MSG, extras);
    }

    private static ContentValues getContentValues(Conversation conversation) {
        ContentValues conversationValues = new ContentValues();
        conversationValues.put(ConversationsTable.Key.CONVERSATION_ID, conversation.getConversationId());
        conversationValues.put(ConversationsTable.Key.BRAND_ID, conversation.getBrandId());
        conversationValues.put(ConversationsTable.Key.TARGET_ID, conversation.getTargetId());
        conversationValues.put(ConversationsTable.Key.STATE, (conversation.getState() != null ? conversation.getState().ordinal() : -1));
        conversationValues.put(ConversationsTable.Key.TTR_TYPE, (conversation.getConversationTTRType() != null ?conversation.getConversationTTRType().ordinal() : -1));
        conversationValues.put(ConversationsTable.Key.REQUEST_ID, conversation.getRequestId());
        conversationValues.put(ConversationsTable.Key.CLOSE_REASON, (conversation.getCloseReason() != null ? conversation.getCloseReason().ordinal() : -1));
        conversationValues.put(ConversationsTable.Key.START_TIMESTAMP, conversation.getStartTimestamp());
        conversationValues.put(ConversationsTable.Key.END_TIMESTAMP, conversation.getEndTimestamp());
        conversationValues.put(ConversationsTable.Key.CSAT_STATUS, conversation.isShowedCSAT().getValue());
        conversationValues.put(ConversationsTable.Key.UNREAD_MESSAGES_COUNT, conversation.getUnreadMessages());

        return conversationValues;
    }

    public void updateTTRType(final String targetId, final TTRType type, final long effectiveTTR) {

        final Conversation conversation = getConversationFromTargetIdMap(targetId);
        if (conversation != null) {
            conversation.getTTRManager().updateTTR(targetId, effectiveTTR);

            if (type != conversation.getConversationTTRType()) {
                conversation.setConversationTTRType(type);
                DataBaseExecutor.execute(() -> {
                    ContentValues conversationValues = new ContentValues();
                    conversationValues.put(ConversationsTable.Key.TTR_TYPE, type.ordinal());
                    getDB().update(conversationValues, ConversationsTable.Key.CONVERSATION_ID + "=?", new String[]{String.valueOf(conversation.getConversationId())});
                });
            }
        }
    }

	public long calculateEffectiveTTR(final String targetId, final long ttrValue, final long manualTTR, final long delayTTR, final long clockDiff) {

    	long effectiveTTR = -1;
		final Conversation conversation = getConversationFromTargetIdMap(targetId);
		if (conversation != null) {

			effectiveTTR = conversation.getTTRManager().calculateEffectiveTTR(targetId, ttrValue, manualTTR, delayTTR, clockDiff);
		}

		return effectiveTTR;
	}

	public void resetEffectiveTTR(final String targetId) {
		final Conversation conversation = getConversationFromTargetIdMap(targetId);
		if (conversation != null) {
			conversation.getTTRManager().resetEffectiveTTR();
		}
	}

	public void showTTR(final String targetId) {
		final Conversation conversation = getConversationFromTargetIdMap(targetId);
		if (conversation != null) {
			conversation.getTTRManager().showTTR(targetId);
		}
	}

	public void notifyOffHoursStatus(final String brandId) {
        Conversation conversation = getConversationFromBrandIdMap(brandId);
        if (conversation != null){
            conversation.getTTRManager().updateIfOffHours(brandId);
        }
    }

    // TODO Perry, Add logic here: Show CSAT if conversation is closed AND there where no PCS dialogs
    public void updateCSAT(final String brandId, final String conversationIdToUpdate) {
        final Conversation conversation = getConversationFromTargetIdMap(brandId);

        if (conversation != null && conversationIdToUpdate.equals(conversation.getConversationId())) {
            conversation.setShowedCSAT(CSAT.CSAT_SHOW_STATUS.SHOWN);
        }

        DataBaseExecutor.execute(() -> {
            ContentValues conversationValues = new ContentValues();
            conversationValues.put(ConversationsTable.Key.CSAT_STATUS, CSAT.CSAT_SHOW_STATUS.SHOWN.getValue());
            getDB().update(conversationValues, ConversationsTable.Key.CONVERSATION_ID + "=?", new String[]{String.valueOf(conversationIdToUpdate)});
        });
    }

    public DataBaseCommand<Conversation> queryConversationById(final String serverID) {
        return new DataBaseCommand<>(() -> {
            Cursor c = getDB().rawQuery("select * from " + ConversationsTable.TABLE_NAME + " where " + ConversationsTable.Key.CONVERSATION_ID + " = ?", serverID);
            Conversation conversation = null;
            if (c != null) {
                conversation = readConversation(c);
                // Cursor closed in readConversation(c)
            }
            return conversation;
        });
    }

    private void addConversationToMaps(String targetId, Conversation conversation) {
        mConversationsByAccountId.put(targetId, conversation);
        mConversationsByServerId.put(conversation.getConversationId(), conversation);

        // Store the conversation by the conversation id
        LPLog.INSTANCE.d(TAG, "Putting conversation in ConversationMap. Conversation Id: " + conversation.getConversationId() + " targetId: " + conversation.getTargetId());

    }

    private void removeTempConversationFromMaps(String targetId) {
        Conversation conversationByAccount = mConversationsByAccountId.get(targetId);
        if (conversationByAccount != null &&
                conversationByAccount.getConversationId().equals(Conversation.TEMP_CONVERSATION_ID)) {
            this.mConversationsByAccountId.remove(targetId);
        }
        this.mConversationsByServerId.remove(Conversation.TEMP_CONVERSATION_ID);

	    LPLog.INSTANCE.d(TAG, "Removing temp conversation Id: " + Conversation.TEMP_CONVERSATION_ID + " targetId: " + targetId);

    }

    private Conversation getConversationFromBrandIdMap(String brandId) {
        final Conversation conversation = mConversationsByAccountId.get(brandId);
        if (conversation != null) {
            final String conversationServerId = conversation.getConversationId();

            // If still not mapped by conversation server id, map it now
            if (conversationServerId != null && mConversationsByServerId.get(conversationServerId) == null) {
                mConversationsByServerId.put(conversationServerId, conversation);
            }
        }

        return conversation;
    }


    /**
     * Remove all conversations of the given targetId from the internal maps
     *
     * @param targetId
     */
    private void removeAllConversationsFromMaps(String targetId) {

        // Don't remove if there is an active conversation
        if (isConversationActive(targetId)) {
            LPLog.INSTANCE.w(TAG, "removeAllConversationsFromMaps: current conversation from brand " + targetId + " is active. Did not remove");
            return;
        }

        Conversation conversationByAccount = mConversationsByAccountId.get(targetId);

        if (conversationByAccount != null) {
            String conversationId = conversationByAccount.getConversationId();
            this.mConversationsByServerId.remove(conversationId);

            LPLog.INSTANCE.d(TAG, "removeAllConversationsFromMaps: Removing conversation ID" + conversationId);

            this.mConversationsByAccountId.remove(targetId);
            LPLog.INSTANCE.d(TAG, "Removed conversations of targetId: " + targetId);
        }
    }

    /**
     * Clear all conversations of the given targetId
     *
     * @param targetId
     * @return the number of messages removed
     */
    public DataBaseCommand<Integer> clearClosedConversations(final String targetId) {

        // Remove conversation from internal maps
        removeAllConversationsFromMaps(targetId);

        // Remove conversation from DB
        return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<Integer>() {

            //where targetId='xyz' and state=0;
            String whereString = ConversationsTable.Key.TARGET_ID + "=? and " + ConversationsTable.Key.STATE + "=?";
            String[] whereArgs = {targetId, String.valueOf(ConversationState.CLOSE.ordinal())}; // Closed conversations for targetId

            @Override
            public Integer query() {
                return getDB().removeAll(whereString, whereArgs);
            }
        });

    }

    public DataBaseCommand<Integer> clearAllConversations(final String targetId) {

        // Remove conversation from internal maps
        removeAllConversationsFromMaps(targetId);

        // Remove conversation from DB
        return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<Integer>() {

            //where targetId='xyz' and state=0;
            String whereString = ConversationsTable.Key.TARGET_ID + "=?";
            String[] whereArgs = {targetId}; // All conversations for targetId regardless of conversations state

            @Override
            public Integer query() {
                return getDB().removeAll(whereString, whereArgs);
            }
        });

    }

    public Conversation getConversationFromTargetIdMap(String targetId) {
        final Conversation conversation = mConversationsByAccountId.get(targetId);
        if (conversation != null) {
            final String conversationServerId = conversation.getConversationId();

            // If still not mapped by conversation server id, map it now
            if (conversationServerId != null && mConversationsByServerId.get(conversationServerId) == null) {
                mConversationsByServerId.put(conversationServerId, conversation);
            }
        }

        return conversation;
    }

    private Conversation getConversationFromServerIdMap(final String conversationServerId) {
        final Conversation conversation = mConversationsByServerId.get(conversationServerId);

        // In some cases, the Conversation is still not mapped by server id. Fetch it from thew brandId map and then map it also by server id
        if (conversation == null) {
            for (Map.Entry<String, Conversation> convEntry : mConversationsByAccountId.entrySet()) {
                final Conversation convEntryValue = convEntry.getValue();
                final String convEntryValueServerId = convEntryValue.getConversationId();
                if (conversationServerId.equals(convEntryValueServerId)) {
                    mConversationsByServerId.put(conversationServerId, convEntryValue);
                    return convEntryValue;
                }
            }
        }
        return conversation;
    }

    public Conversation getNewestClosedConversation(String targetId) {
        //return the newest closed conversation record of one conversation for specific target id

        Cursor c = getDB().rawQuery(
                "select * from " + ConversationsTable.TABLE_NAME +
                        " where " + ConversationsTable.Key.TARGET_ID + " = ? and " +
                        ConversationsTable.Key.STATE + " = ?" +
                        " order by " + ConversationsTable.Key.END_TIMESTAMP + " DESC limit 1"
                , targetId, ConversationState.CLOSE.ordinal());

        if (c != null && c.moveToFirst()) {
            try {
                if (c.getCount() == 1) {
                    return new Conversation(c);
                }
            } finally {
                c.close();
            }
        }
        return null;
    }

    /**
     * Set a number of updating process (query messages/ agent details) that currently updating specific conversation.
     * If we'll be interrupted we can recover by looking on this field.
     *
     * @param conversationId
     * @param numUpdateInProgress
     */
    public void setUpdateRequestInProgress(final String conversationId, final int numUpdateInProgress) {
        final Conversation conversation = getConversation(conversationId);

        if (conversation != null) {
            LPLog.INSTANCE.d(TAG, "adding update request in progress for conversation: " + conversationId + ", requests in progress: " + numUpdateInProgress);
            conversation.setUpdateInProgress(numUpdateInProgress);
        }
        DataBaseExecutor.execute(() -> {
	        LPLog.INSTANCE.d(TAG, "update request for conversation in DB: " + conversationId + ", numUpdateInProgress: " + numUpdateInProgress);

            ContentValues conversationValues = new ContentValues();
            conversationValues.put(ConversationsTable.Key.CONCURRENT_REQUESTS_COUNTER, numUpdateInProgress);
            getDB().update(conversationValues, ConversationsTable.Key.CONVERSATION_ID + "=?", new String[]{String.valueOf(conversationId)});
        });
    }

    /**
     * Increasing updating process (query messages/ agent details) that currently updating specific conversation.
     * If we'll be interrupted we can recover by looking on this field.
     *
     * @param conversationId
     */
    public void addUpdateRequestInProgress(final String conversationId) {
        final Conversation conversation = getConversation(conversationId);

        if (conversation != null) {
            int mUpdateInProgress = conversation.getUpdateInProgress() + 1;
            LPLog.INSTANCE.d(TAG, "adding update request in progress for conversation: " + conversationId + ", requests in progress: " + mUpdateInProgress);
            conversation.setUpdateInProgress(mUpdateInProgress);
        }
        updateRequestsInProgress(conversationId, 1);
    }

    /**
     * Decreasing updating process (query messages/ agent details) that currently updating specific conversation.
     * If we'll be interrupted we can recover by looking on this field.
     *
     * @param conversationId
     */
    public void removeUpdateRequestInProgress(final String conversationId) {
        final Conversation conversation = getConversation(conversationId);

        if (conversation != null) {
            int mUpdateInProgress = conversation.getUpdateInProgress() - 1;
            LPLog.INSTANCE.d(TAG, "removing update request for conversation: " + conversationId + ", requests in progress: " + mUpdateInProgress);
            conversation.setUpdateInProgress(mUpdateInProgress);
        }

        updateRequestsInProgress(conversationId, -1);
    }

    /**
     * update the number of updating process in db
     *
     * @param conversationId
     * @param addValue       new value
     */
    private void updateRequestsInProgress(final String conversationId, final int addValue) {
        DataBaseExecutor.execute(() -> {
            try (Cursor c = getDB().query(new String[]{ConversationsTable.Key.CONCURRENT_REQUESTS_COUNTER},
                    ConversationsTable.Key.CONVERSATION_ID + " = ? ", new String[]{conversationId}, null, null, null)) {
                int updateInProgress = 0;
                if (c != null && c.moveToFirst()) {
                    updateInProgress = c.getInt(c.getColumnIndex(ConversationsTable.Key.CONCURRENT_REQUESTS_COUNTER));
                }

	            LPLog.INSTANCE.d(TAG, "update request for conversation in DB: " + conversationId + ", requests in progress: " + updateInProgress + " added value = " + addValue);

                ContentValues conversationValues = new ContentValues();
                conversationValues.put(ConversationsTable.Key.CONCURRENT_REQUESTS_COUNTER, updateInProgress + addValue);
                getDB().update(conversationValues, ConversationsTable.Key.CONVERSATION_ID + "=?", new String[]{String.valueOf(conversationId)});
            }
        });
    }

	public void updateConversationState(String targetId, final String conversationId, final ConversationState state) {

		final Conversation conversation = getConversation(targetId);

		if (conversation != null) {
			LPLog.INSTANCE.d(TAG, "update conversation state, new state = " + state);
			conversation.setState(state);
		}

        DataBaseExecutor.execute(() -> {
	        LPLog.INSTANCE.d(TAG, "update new state for conversation in DB: " + conversationId + ", state: " + state);
            ContentValues conversationValues = new ContentValues();
            conversationValues.put(ConversationsTable.Key.STATE, state.ordinal());
            getDB().update(conversationValues, ConversationsTable.Key.CONVERSATION_ID + "=?", new String[]{String.valueOf(conversationId)});
        });
	}

    public DataBaseCommand<List<Conversation>> getNotUpdatedConversations(final String brandId) {
        return new DataBaseCommand<>(() -> {
            try (Cursor c = getDB().rawQuery(
                    "select * from " + ConversationsTable.TABLE_NAME +
                            " where " + ConversationsTable.Key.BRAND_ID + " = ? and " +
                            ConversationsTable.Key.CONCURRENT_REQUESTS_COUNTER + " > 0 ", brandId)) {
                if (c != null) {
                    List<Conversation> conversations = new ArrayList<>(c.getCount());
                    while (c.moveToNext()) {
                        conversations.add(new Conversation(c));
                    }
                    return conversations;
                }
            }
            return null;
        });

    }

	@Override
    public void clear() {
		// Clear all TTR stored data
		for (Conversation conversation : mConversationsByAccountId.values()) {
			conversation.getTTRManager().clear();
		}
        for (Conversation conversation : mConversationsByServerId.values()) {
            conversation.getTTRManager().clear();
        }

        //clearing all conversation maps
        mConversationsByAccountId.clear();
        mConversationsByServerId.clear();
	}

    @Override
    public void shutDown() {

        //removing all waiting ttr events
        for (Conversation conversation : mConversationsByAccountId.values()) {
            conversation.getTTRManager().shutDown();
        }
    }


    /**
     * update Conversations set status = closed
     * @param brandId
     */
    public void markAllPendingConversationsAsFailed(final String brandId) {
        final Conversation conversation = getConversationFromTargetIdMap(brandId);
        if (conversation != null && (conversation.getState() == ConversationState.PENDING || conversation.getState() == ConversationState.QUEUED)){
            conversation.setState(ConversationState.CLOSE);
            conversation.setEndTimestamp(System.currentTimeMillis());
        }
        DataBaseExecutor.execute(() -> {
            String where = ConversationsTable.Key.STATE + " =? AND " + ConversationsTable.Key.BRAND_ID + " = ?";
            ContentValues contentValues = new ContentValues();
            // The value to be changed
            contentValues.put(ConversationsTable.Key.STATE, ConversationState.CLOSE.ordinal());

            int result = getDB().update(contentValues, where, new String[]{String.valueOf(ConversationState.PENDING.ordinal()), brandId});
	        LPLog.INSTANCE.d(TAG, String.format(Locale.ENGLISH, "Updated %d pending conversation as Closed on DB", result));
        });
    }

    public void createDummyConversationForWelcomeMessage(final String brandId, final String dummyConversationId, final long startTime) {
        DataBaseExecutor.execute(() -> {
            ContentValues conversationValues = new ContentValues();
            conversationValues.put(ConversationsTable.Key.BRAND_ID, brandId);
            conversationValues.put(ConversationsTable.Key.TARGET_ID, brandId);
            conversationValues.put(ConversationsTable.Key.CONVERSATION_ID, dummyConversationId);
            conversationValues.put(ConversationsTable.Key.STATE, ConversationState.LOCKED.ordinal());
            conversationValues.put(ConversationsTable.Key.TTR_TYPE, TTRType.NORMAL.ordinal());
            conversationValues.put(ConversationsTable.Key.REQUEST_ID, -1);
            conversationValues.put(ConversationsTable.Key.UNREAD_MESSAGES_COUNT, -1);
            conversationValues.put(ConversationsTable.Key.START_TIMESTAMP, startTime);
            getDB().insert(conversationValues);
            LPLog.INSTANCE.d(TAG, "created dummy conversation for first message- startTime = " + startTime);
        });
    }

    public void enqueuePendingConversationRequestId(Long requestId) {
        pendingConversationRequestId.add(requestId);
    }

    @Nullable
    public Long dequeuePendingConversationRequestId() {
        return pendingConversationRequestId.poll();
    }

    public void notifyConversationFragmentClosedEvent() {
        LocalBroadcast.sendBroadcast(BROADCAST_CONVERSATION_FRAGMENT_CLOSED);
        mController.mEventsProxy.onConversationFragmentClosed();
    }
}
