package com.liveperson.messaging.commands.pusher

import android.text.TextUtils
import com.liveperson.infra.Command
import com.liveperson.infra.ICallback
import com.liveperson.infra.Infra
import com.liveperson.infra.InternetConnectionService
import com.liveperson.infra.auth.LPAuthenticationParams
import com.liveperson.infra.controller.DBEncryptionHelper
import com.liveperson.infra.log.LPLog.e
import com.liveperson.infra.managers.PreferenceManager
import com.liveperson.infra.network.http.HttpHandler
import com.liveperson.infra.network.http.request.HttpGetRequest
import com.liveperson.infra.utils.EncryptionVersion
import com.liveperson.messaging.Messaging
import com.liveperson.messaging.TaskType
import com.liveperson.messaging.controller.connection.ConnectionParamsCache
import com.liveperson.messaging.model.AmsAccount
import com.liveperson.messaging.model.SynchronizedAuthenticationCompletedCallback
import com.liveperson.messaging.model.SynchronizedInternetConnectionCallback
import org.json.JSONObject
import javax.net.ssl.SSLPeerUnverifiedException

class GetIsPusherRegisteredCommand(private val controller: Messaging,
                                   private val brandId: String,
                                   private val deviceToken: String,
                                   private val appId: String,
                                   private val authenticationParams: LPAuthenticationParams?,
                                   private val callback: ICallback<Boolean, Exception>) : Command {

    companion object {
        private const val TAG = "GetIsPusherRegisteredCommand"
        private const val URL = "https://%s/api/account/%s/device/isRegistered?appId=%s&consumerId=%s"
    }

    override fun execute() {
        if (TextUtils.isEmpty(appId)) {
            callback.onError(Exception("Unable to make request. Error: Missing appID"))
            notifyError(Exception("Unable to make request. Error: Missing appID"))
            return
        }
        validateNetworkState()
    }

    private fun validateNetworkState() {
        if (InternetConnectionService.isNetworkAvailable()) {
            authorizeAndSendRequest()
        } else {
            SynchronizedInternetConnectionCallback { authorizeAndSendRequest() }.execute()
        }
    }

    private fun authorizeAndSendRequest() {
        val isExecuting = SynchronizedAuthenticationCompletedCallback(controller.mAccountsController, brandId, object : ICallback<Void, Exception> {
            override fun onSuccess(value: Void?) {
                try {
                    var pusherDomain = controller.mAccountsController.getServiceUrl(brandId, ConnectionParamsCache.CSDS_PUSHER_DOMAIN_KEY)

                    if (TextUtils.isEmpty(pusherDomain)) {
                        pusherDomain = PreferenceManager.getInstance().getStringValue(ConnectionParamsCache.CSDS_PUSHER_DOMAIN_KEY, brandId, null);
                    }
                    if (TextUtils.isEmpty(pusherDomain)) {
                        notifyError(Exception("Error: Missing Domain"));
                    } else {
                        val token = getTokenFromSharedPreferences()
                        if (!TextUtils.isEmpty(controller.amsUsers.getConsumerId(brandId))) {
                            val consumerId = controller.amsUsers.getConsumerId(brandId)
                            val url = String.format(URL, pusherDomain, brandId, appId, consumerId)
                            sendRequest(url, token)
                        } else {
                            controller.amsUsers.getConsumerByBrandIDFromDB(brandId).setPostQueryOnBackground { consumerId: String? ->
                                if (!TextUtils.isEmpty(consumerId)) {
                                    val url = String.format(URL, pusherDomain, brandId, appId, consumerId)
                                    sendRequest(url, token)
                                } else {
                                    callback.onError(Exception("Error: Missing consumerID"))
                                    notifyError(java.lang.Exception("Error: Missing consumerID"))
                                }
                            }.execute()
                        }
                    }
                } catch (error: Exception) {
                    e(TAG, "sendRequest: Failed to obtain domain/consumerId/token to make request ", error)
                }
            }

            override fun onError(exception: Exception?) {
                exception?.let {
                    e(TAG, "authorizeAndSendRequest: Failed to authorize ", it)
                    notifyError(it)
                }
            }

        }).executeWithReturnValue()

        if (!isExecuting) {
            controller.connect(brandId, authenticationParams, null, true, false)
        }
    }

    private fun getTokenFromSharedPreferences(): String? {
        val token: String?
        val decryptedToken: String? = PreferenceManager.getInstance().getStringValue(AmsAccount.KEY_ACCOUNT_TOKEN_ENC, brandId, null)
        if (TextUtils.isEmpty(decryptedToken)) {
            token = PreferenceManager.getInstance().getStringValue(AmsAccount.KEY_ACCOUNT_TOKEN, brandId, null)
            PreferenceManager.getInstance().remove(AmsAccount.KEY_ACCOUNT_TOKEN, brandId)
        } else {
            token = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1, decryptedToken)
        }
        return token
    }

    private fun sendRequest(requestURL: String, token: String?) {
        val certificates = controller.mAccountsController.getCertificatePinningKeys(brandId)
        val request = HttpGetRequest(requestURL)
        request.addHeader("authorization", "bearer $token")
        request.certificatePinningKeys = certificates
        request.callback = object : ICallback<String, Exception?> {
            override fun onSuccess(response: String) {
                val jsonResponse = JSONObject(response)
                val isRegistered = jsonResponse.getBoolean("isRegistered")
                val registeredDeviceToken = jsonResponse.getString("deviceToken")

                // response can be boolean or String. Need to remove the first and last
                if (isRegistered && deviceToken == registeredDeviceToken) {
                    callback.onSuccess(true)
                } else {
                    callback.onSuccess(false)
                }
            }

            override fun onError(exception: Exception?) {
                callback.onError(exception)
            }
        }
        HttpHandler.execute(request)
    }

    private fun notifyError(exception: Exception) {
        Infra.instance.postOnMainThread {
            if (exception is SSLPeerUnverifiedException) {
                controller.mEventsProxy.onError(TaskType.INVALID_CERTIFICATE, exception.message)
            }
            callback.onError(exception)
        }
    }
}
