package com.liveperson.messaging.background;

import android.app.Notification;
import android.app.PendingIntent;
import android.app.Service;
import android.content.Intent;
import android.os.IBinder;
import android.support.annotation.Nullable;

import com.liveperson.infra.handler.NotificationHandler;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.messaging.R;
import com.liveperson.messaging.MessagingFactory;

/**
 * Created by nirni on 7/18/16.
 * This is a generic service that proxy the incoming action to a ServiceActioner implementation.
 * Current this supports only an image upload/download. If need to add another one, need to add support in onStartCommand() and in the listener.
 */
public class BackgroundActionsService extends Service {

	private static final String TAG = "BackgroundActionsService";

	public static final String EXTRA_ACTION_TYPE = "extra_action_type";
	public static final String EXTRA_FILE_TYPE = "extra_file_type";
	public static final int EXTRA_ACTION_TYPE_UPLOAD = 1;
	public static final int EXTRA_TYPE_ACTION_DOWNLOAD = 2;
	public static final int EXTRA_TYPE_ACTION_REUPLOAD = 3;
	private static final int ONGOING_NOTIFICATION_ID = 17;

	/**
	 * A ServiceActioner implementation that this service is talking to.
	 * If this service needs to support more actions, it should be added here
	 */
	private ServiceActioner mServiceActioner;

	// This is the listener that each action should receive and call upon finish its task
	private final ServiceActionCallbackListener mServiceActionCallbackListener = new ServiceActionCallbackListener() {
		@Override
		public void onSuccess(String brandId) {
			// If supporting more ServiceActioner implementations need to put them here
			if (mServiceActioner != null) {
				// If there are no more pending actions stop the service
				if (!mServiceActioner.isPendingActions()) {
					LPLog.INSTANCE.d(TAG, "onSuccess: all service actions are done. Stop service");
					stopSelf();
					serviceIsOff(brandId);
				}
				else {
					LPLog.INSTANCE.d(TAG, "onSuccess: there are still pending service actions. Service still running...");
				}
			}
		}

		@Override
		public void onFail(String brandId) {
			// If supporting more ServiceActioner implementations need to put them here
			if (mServiceActioner != null) {
				// If there are no more pending actions stop the service
				if (!mServiceActioner.isPendingActions()) {
					LPLog.INSTANCE.d(TAG, "onFail: all service actions are done. Stop service");
					stopSelf();
					serviceIsOff(brandId);
				}
				else {
					LPLog.INSTANCE.d(TAG, "onFail: there are still pending service actions. Service still running...");
				}
			}
		}
	};

	/**
	 * Start a non sticky service
	 * @param intent
	 * @param flags
	 * @param startId
	 * @return
	 */
	@Override
	public int onStartCommand(Intent intent, int flags, int startId) {

		String brandId;
        int type = intent.getIntExtra(EXTRA_ACTION_TYPE, -1);
        if (type == -1){
            LPLog.INSTANCE.e(TAG, "onStartCommand: Error getting type. aborting");
            return Service.START_NOT_STICKY;
        }

		mServiceActioner = MessagingFactory.getInstance().getController().getFileSharingManager();
		//noinspection ConstantConditions -- we want to verify it's the correct class, otherwise abort
		if (!(mServiceActioner instanceof ServiceActioner)) {
			LPLog.INSTANCE.e(TAG, "onStartCommand: FileSharingManager does not implement ServiceActioner");
			return Service.START_NOT_STICKY;
		}

		switch (type) {
            case EXTRA_ACTION_TYPE_UPLOAD:
				LPLog.INSTANCE.d(TAG, "onStartCommand: got new file upload command through service");
				// TODO: 2/25/18 this disabled due to a crash on host app compiled with SDK 27. Need to revert this when SDK is upgraded to compile with SDK 27
//				setForeground(true);
                break;
            case EXTRA_TYPE_ACTION_DOWNLOAD:
				LPLog.INSTANCE.d(TAG, "onStartCommand: got new file download command through service");
				// TODO: 2/25/18 this disabled due to a crash on host app compiled with SDK 27. Need to revert this when SDK is upgraded to compile with SDK 27
//				setForeground(false);
                break;
			case EXTRA_TYPE_ACTION_REUPLOAD:
				LPLog.INSTANCE.d(TAG, "onStartCommand: got new file re-upload command through service");
				// TODO: 2/25/18 this disabled due to a crash on host app compiled with SDK 27. Need to revert this when SDK is upgraded to compile with SDK 27
//				setForeground(true);
				break;
		}

		brandId = intent.getStringExtra(FileSharingManager.SERVICE_EXTRA_BRAND_ID);
		serviceIsUp(brandId);
		mServiceActioner.actionFromService(intent, mServiceActionCallbackListener);

		return Service.START_NOT_STICKY;
	}

	/**
	 * Set the service to be a foreground service.
	 *
	 * @param upload set the type of the foreground service notification content to be used. true - upload operation, false - download operation
	 */
	private void setForeground(boolean upload) {

		// Get the notification builder for the the foreground service
		Notification.Builder builder = getNotificationBuilder(upload);
		Notification notification = builder.build();

		startForeground(ONGOING_NOTIFICATION_ID, notification);
	}

	/**
	 * Get pending intent that was set for the ongoing notification. If no pending intent was set return null
	 * @return
	 */
	private PendingIntent getPendingIntent() {
		return MessagingFactory.getInstance().getController().getImageServicePendingIntent();
	}

	private Notification.Builder getNotificationBuilder(boolean upload) {
		Notification.Builder builder;

		// Get the notification that was set for upload or download by the host app.
		builder = upload ? MessagingFactory.getInstance().getController().getImageForegroundServiceUploadNotificationBuilder() :
				MessagingFactory.getInstance().getController().getImageForegroundServiceDownloadNotificationBuilder();

		if (builder == null) {
			builder = getDefaultNotificationBuilder(upload);
		}

		return builder;
	}

	/**
	 * Create a default notification builder
	 * */
	private Notification.Builder getDefaultNotificationBuilder(boolean upload){
		int icon;
		String title;

		if (upload) {
			icon = android.R.drawable.stat_sys_upload;
			title = getString(R.string.uploading_image);
		} else {
			icon = android.R.drawable.stat_sys_download;
			title = getString(R.string.downloading_image);
		}
		return new NotificationHandler(this, title, null)
				.setIconResourceId(icon)
				.setPendingIntent(getPendingIntent())
				.createForegroundServiceNotificationBuilder();
	}

	/**
	 * Inform the messaging (and the connection state machine) that the service is up
	 * @param brandId
	 */
	private void serviceIsUp(String brandId) {

		MessagingFactory.getInstance().getController().serviceStarted(brandId);
	}

	/**
	 * Inform the messaging (and the connection state machine) that the service is off
	 * @param brandId
	 */
	private void serviceIsOff(String brandId){
		MessagingFactory.getInstance().getController().serviceStopped(brandId);

	}

	/**
	 * This service is not meant to be bind
	 * @param intent
	 * @return
	 */
	@Nullable
	@Override
	public IBinder onBind(Intent intent) {
		LPLog.INSTANCE.w(TAG, "onBind: Service onBind. Should not be used");
		return null;
	}

	///////////////////// Interfaces /////////////////////////////////

	/**
	 * A listener that is passed to the ServiceActioner and is used as a callback to indicate the action has finished
	 */
	public interface ServiceActionCallbackListener{

		void onSuccess(String brandId);

		void onFail(String brandId);
	}

	/**
	 * An interface for actions from this service.
	 * Each action is sent to the actionFromService method.
	 */
	public interface ServiceActioner {

		/**
		 * Receive an intent with the description of the action and a listener for callbacks
		 * @param intent
		 * @param serviceActionCallbackListener
		 */
		void actionFromService(Intent intent, BackgroundActionsService.ServiceActionCallbackListener serviceActionCallbackListener);

		/**
		 * This should return whether there are any pending actions on the ServiceActioner
		 * @return <i>true</i> if there are any pending actions, <i>false</i> otherwise
		 */
		@SuppressWarnings("BooleanMethodIsAlwaysInverted")  //Currently method is better for understanding.
		boolean isPendingActions();
	}
}
