package com.liveperson.messaging.commands.pusher;

import android.text.TextUtils;

import com.liveperson.infra.Command;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.Infra;
import com.liveperson.infra.InternetConnectionService;
import com.liveperson.infra.auth.LPAuthenticationParams;
import com.liveperson.infra.auth.LPAuthenticationType;
import com.liveperson.infra.controller.DBEncryptionHelper;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.network.http.requests.BadgeCounterRequest;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.TaskType;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.model.AmsAccount;
import com.liveperson.messaging.model.SynchronizedAuthenticationCompletedCallback;
import com.liveperson.messaging.model.SynchronizedInternetConnectionCallback;
import com.liveperson.messaging.utils.TokenUtils;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.net.ssl.SSLPeerUnverifiedException;


/**
 * Created by ofira on 9/14/17.
 */
// todo Perry, verify with Pusher team that the unread badge is for the conversation and not dialog.
public class GetUnreadMessagesCountCommand implements Command {

    private static final String TAG = "GetUnreadMessagesCountCommand";

    private static final String PUSHER_BADGE_URL = "https://%s/api/account/%s/device/unread-messages-count?appId=%s&lpId=%s&v=2.0";

    private static final String BADGE_COUNT = "badge_count";
    private static final String BADGE_TIMESTAMP = "badge_timestamp";
    private static final String ERROR_UNABLE_TO_MAKE_REQUEST = "Unable to make request.";
    private static final String ERROR_SERVER_ERROR = "Server error: ";

    private static final long BADGE_LAST_TIMESTAMP_THRESHOLD = 1000 * 10; // 10 seconds

    private final Messaging mController;
    private String mBrandId;
    private ICallback<Integer, Exception> mCallback;
    private String mAppId;
    private PreferenceManager mPreferenceManager;
    private String mLocalToken;
    private LPAuthenticationParams authenticationParams;
    private static Map<String, Integer> unreadCountMapping;

    public GetUnreadMessagesCountCommand(Messaging messagingController, String brandId, String appId, LPAuthenticationParams authenticationParams, ICallback<Integer, Exception> callback) {
        mController = messagingController;
        mBrandId = brandId;
        mAppId = appId;
        mCallback = callback;
        mPreferenceManager = PreferenceManager.getInstance();
        this.authenticationParams = authenticationParams;
    }

    @Override
    public void execute() {

        //Checking if the time threshold was passed
        if (!isBadgeThresholdWasPassed(mBrandId)) {
            LPLog.INSTANCE.d(TAG, "Time threshold was not passed yet. Return cached data");
            returnCachedDetails();

        } else {

            LPLog.INSTANCE.d(TAG, "Time threshold was passed");

            // Backwards compatibility to support older API call from host app.
            if (authenticationParams == null || authenticationParams.getAuthType() != LPAuthenticationType.AUTH) {
                // If the AmsAccount does not exist for the given brandId (e.g. the app was killed and no conversation started yet),
                // we need to get the token from shared preferences since it's still does not exist in the account.
                if (mController.mAccountsController.getAccount(mBrandId) == null) {
                    mLocalToken = getTokenFromSharedPreferences();
                    if (TokenUtils.isJwtExpired(mLocalToken)) {
                        sendTokenExpired();
                        return;
                    }
                } else if (mController.mAccountsController.isTokenExpired(mBrandId)) { // Check token from the account
                    sendTokenExpired();
                    return;
                }

                LPLog.INSTANCE.d(TAG, "JWT is valid - send request to Pusher");

                if (TextUtils.isEmpty(mAppId)) {
                    notifyError(new Exception(ERROR_UNABLE_TO_MAKE_REQUEST + " Error: Missing appID"));
                    return;
                }
                fetchConsumerIdAndSendRequest();
            } else {
                validateNetworkState();
            }
        }
    }

    /**
     * Get consumer id from memory. If not available in memory, try to get it from database.
     */
    private void fetchConsumerIdAndSendRequest() {
        String consumerId = mController.amsUsers.getConsumerId(mBrandId);

        // If consumerId is not available in memory try to get from DB
        if (TextUtils.isEmpty(consumerId)) {
            LPLog.INSTANCE.d(TAG, "fetchConsumerIdAndSendRequest: consumerId is not available. Trying to get from DB...");
            mController.amsUsers.getConsumerByBrandIDFromDB(mBrandId).setPostQueryOnBackground(returnedConsumerId -> {
                if (returnedConsumerId != null && !TextUtils.isEmpty(returnedConsumerId)) {
                    LPLog.INSTANCE.d(TAG, "fetchConsumerIdAndSendRequest: got consumerId from DB (" + returnedConsumerId + "). get the unread message count with it...");
                    getPusherDomainAndSendRequest(returnedConsumerId);
                } else {
                    LPLog.INSTANCE.w(TAG, "fetchConsumerIdAndSendRequest: Cannot get user profile from DB. Quit get badge counter");
                    notifyError(new Exception(ERROR_UNABLE_TO_MAKE_REQUEST + " Error: Missing consumerID"));
                }
            }).execute();
        } else {
            getPusherDomainAndSendRequest(consumerId);
        }
    }

    /**
     * Get pusher domain from memory. If not available in memory, try to get it from shared preferences.
     * @param consumerId consumer id for which to make this request
     */
    private void getPusherDomainAndSendRequest(String consumerId) {
        String pusherDomain = mController.mAccountsController.getServiceUrl(mBrandId, ConnectionParamsCache.CSDS_PUSHER_DOMAIN_KEY);
        if (TextUtils.isEmpty(pusherDomain)) {
            LPLog.INSTANCE.d(TAG, "getPusherDomainAndSendRequest: pusher domain is not available. Trying to get from Shared preferences...");
            pusherDomain = PreferenceManager.getInstance().getStringValue(ConnectionParamsCache.CSDS_PUSHER_DOMAIN_KEY, mBrandId, null);
            if (TextUtils.isEmpty(pusherDomain)) {
                notifyError(new Exception(ERROR_UNABLE_TO_MAKE_REQUEST + " Error: Missing Domain"));
            } else {
                sendRequest(pusherDomain, consumerId);
            }
        } else {
            sendRequest(pusherDomain, consumerId);
        }
    }

    /**
     * Get the token that is stored in the shared preferences
     *
     * @return
     */
    private String getTokenFromSharedPreferences() {

        String token;
        String decryptedToken = mPreferenceManager.getStringValue(AmsAccount.KEY_ACCOUNT_TOKEN_ENC, mBrandId, null);
        if (TextUtils.isEmpty(decryptedToken)) {
            token = mPreferenceManager.getStringValue(AmsAccount.KEY_ACCOUNT_TOKEN, mBrandId, null);
            mPreferenceManager.remove(AmsAccount.KEY_ACCOUNT_TOKEN, mBrandId);
        } else {
            token = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1, decryptedToken);
        }

        return token;
    }

    /**
     * Send token expired callback and onError callback for this command
     */
    private void sendTokenExpired() {
        LPLog.INSTANCE.d(TAG, "JWT is expired - calling to onTokenExpired callback");
        mController.mEventsProxy.onTokenExpired();
        notifyError(new Exception(ERROR_UNABLE_TO_MAKE_REQUEST + " Error: Token expired, refresh the token and try again"));
    }

    /**
     * Notify in case of a success with the relevant counter
     *
     * @param counter
     */
    private void notifySuccess(final int counter) {
        if (mCallback != null) {
            Infra.instance.postOnMainThread(() -> mCallback.onSuccess(counter));
        }
    }

    /**
     * Notify in case of an error
     *
     * @param exception
     */
    private void notifyError(final Exception exception) {
        if (mCallback != null) {
            Infra.instance.postOnMainThread(() -> {
                if(exception instanceof SSLPeerUnverifiedException){
                    mController.mEventsProxy.onError(TaskType.INVALID_CERTIFICATE, exception.getMessage());
                }
                mCallback.onError(exception);
            });
        }
    }

    /**
     * Checking if we have an active internet connection and send the request to Pusher
     *
     */
    private void validateNetworkState() {

        if (InternetConnectionService.isNetworkAvailable()) {
            authorizeAndSendRequest();
        } else {
            new SynchronizedInternetConnectionCallback(this::authorizeAndSendRequest).execute();
        }
    }

    /**
     * Checking if we are already have domain and valid jwt to make a request otherwise we connect/refresh
     * jwt in background before making the request.
     */
    private void authorizeAndSendRequest() {

        boolean isExecuting = new SynchronizedAuthenticationCompletedCallback(mController.mAccountsController, mBrandId, new ICallback<Void, Exception> (){

            @Override
            public void onSuccess(Void value) {
                try {
                    mLocalToken = getTokenFromSharedPreferences();
                    fetchConsumerIdAndSendRequest();
                } catch (Exception error) {
                    LPLog.INSTANCE.e(TAG, "sendRequest: Failed to obtain domain/consumerId/token to make request ", error);
                }
            }
            @Override
            public void onError(Exception exception) {
                LPLog.INSTANCE.e(TAG, "authorizeAndSendRequest: Failed to authorize ", exception);
                notifyError(exception);
            }
        }).executeWithReturnValue();

        // If we have valid jwt we will execute the request. otherwise, we'll try to connect.
        if (!isExecuting && authenticationParams != null) {
            mController.connect(mBrandId, authenticationParams, null, true, false);
        }
    }

    private String buildPusherURL(String pusherDomain, String consumerId) {
        return String.format(PUSHER_BADGE_URL, pusherDomain, mBrandId, mAppId, consumerId);
    }


    /**
     * Send a request to Pusher
     */
    private void sendRequest(String pusherURL, String consumerId) {
        pusherURL = buildPusherURL(pusherURL, consumerId);
        List<String> certificates = mController.mAccountsController.getCertificatePinningKeys(mBrandId);
        String token = mController.mAccountsController.getToken(mBrandId);
        token = (token == null) ? mLocalToken : token;

        if (token == null) {
            notifyError(new Exception(ERROR_UNABLE_TO_MAKE_REQUEST + " Error: Authorization failed. Token is missing or is invalid"));
            return;
        }

        new BadgeCounterRequest(pusherURL, token, certificates, new ICallback<String, Exception>() {
            @Override
            public void onSuccess(String json) {
                if (!TextUtils.isEmpty(json)) {
                    try {
                        JSONArray jsonArray = new JSONArray(json);
                        int count = 0;
                        for (int i = 0; i < jsonArray.length(); i++) {
                            JSONObject obj;
                            obj = jsonArray.getJSONObject(i);
                            int idCount = obj.optInt("badge");
                            String conversationId = obj.optString("conversationId");
                            if (unreadCountMapping == null) {
                                unreadCountMapping = new HashMap<>();
                            }
                            unreadCountMapping.put(conversationId, idCount);
                            count += idCount;
                        }
                        updateTimeStamp();
                        PreferenceManager.getInstance().setIntValue(BADGE_COUNT, mBrandId, count);
                        notifySuccess(count);
                    } catch (Exception e) {
                        notifyError(e);
                    }
                } else {
                    LPLog.INSTANCE.e(TAG, "Returned an empty response from pusher");
                }
            }

            @Override
            public void onError(Exception exception) {
                if (exception != null) {
                    try {
                        String mess = parseException(exception.getMessage());
                        JSONObject error = new JSONObject(mess);
                        String errorString = error.getString("error");
                        if (!TextUtils.isEmpty(errorString)) {
                            JSONObject errorBody = new JSONObject(errorString);
                            String statusCode = errorBody.getString("statusCode");
                            String internalCode = errorBody.getString("internalCode");

                            // Ignore the error and return 0 If requested an unread message count for a user that does not exists in pusher.
                            if (!statusCode.equals("404") && !internalCode.equals("23")) {
                                LPLog.INSTANCE.e(TAG, "Failed to send get unread message count to pusher pusher.", exception);
                                notifyError(new Exception(ERROR_SERVER_ERROR + exception.getMessage()));
                            } else {
                                notifySuccess(0);
                            }
                        }

                    } catch (Exception e) {
                        LPLog.INSTANCE.e(TAG, "Failed to parse unread message count exception from pusher.", e);
                        notifyError(e);
                    }
                }
            }
        }).execute();
    }

    /**
     * Extract actual error object from Exception.
     * @param exception
     * @return
     */
    private String parseException(String exception) {
        return exception.substring(exception.indexOf('{'));
    }
    /**
     * Update time stamp value with current time
     */
    private void updateTimeStamp() {
        PreferenceManager.getInstance().setLongValue(BADGE_TIMESTAMP, mBrandId, System.currentTimeMillis());
    }


    /**
     * Return cached details - conversation id, sequence and counter
     */
    private void returnCachedDetails() {
        LPLog.INSTANCE.d(TAG, "Return cached badge counter");
        int count = PreferenceManager.getInstance().getIntValue(BADGE_COUNT, mBrandId, 0);
        notifySuccess(count);
    }


    /**
     * Checking if the threshold was passed
     *
     * @param brandId
     * @return true - threshold was passed, false otherwise
     */
    private boolean isBadgeThresholdWasPassed(String brandId) {
        long lastTimestamp = PreferenceManager.getInstance().getLongValue(BADGE_TIMESTAMP, brandId, 0L);
        long delta = System.currentTimeMillis() - lastTimestamp;
        // If delta is greater than threshold or lastTimestamp never set
        return (lastTimestamp == 0) || delta > BADGE_LAST_TIMESTAMP_THRESHOLD;
    }

    /**
     * Internal only. Fetch unread message count data and store it in the hash map
     */
    public void fetchUnreadCountMapping() {
        if (unreadCountMapping == null) {
            unreadCountMapping = new HashMap<>();
            fetchConsumerIdAndSendRequest();
        } else {
            Integer sum = 0;
            for (String key: unreadCountMapping.keySet()) {
                sum += unreadCountMapping.get(key);
            }
            notifySuccess(sum);
        }
    }

    /**
     * Return mapped unread messages count data.
     * @return
     */
    public static Map<String, Integer> getUnreadCountMapped() {
        return unreadCountMapping;
    }

    /**
     * Clear list of all conversation ids - count data when user moves to background.
     */
    public static void clearMappedUnreadCount() {
        LPLog.INSTANCE.d(TAG, "Removing all mapped unread count data");
        unreadCountMapping = null;
    }
}
