package com.liveperson.messaging.commands.pusher;

import android.text.TextUtils;

import com.liveperson.infra.Command;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.InternetConnectionService;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.network.http.requests.UnregisterPushRequest;
import com.liveperson.messaging.controller.AccountsController;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.model.AmsUsers;
import com.liveperson.messaging.model.SynchronizedInternetConnectionCallback;

import java.util.List;

/**
 * Command to unregister from push notifications for a given brandId.
 * <p>
 * Created by nirni on 1/27/16.
 */
public class UnregisterPusherCommand implements Command {

    private static final String TAG = "UnregisterPusherCommand";

    private static final String PUSHER_UNREGISTER_URL = "https://%s/api/account/%s/device/unregister?v=2.0";
    private final AccountsController mAccountsController;
    private final AmsUsers mAmsUsers;

    private String mBrandId;
    private String mAppId;
    private ICallback<Void, Exception> listener;
    private boolean runImmediately;

    public UnregisterPusherCommand(AccountsController accountsController, AmsUsers amsUsers, String brandId, String appId, ICallback<Void, Exception> listener, boolean immediately) {
        mAccountsController = accountsController;
        mAmsUsers = amsUsers;
        mBrandId = brandId;
        mAppId = appId;

        this.listener = listener;
        runImmediately = immediately;
    }


    @Override
    public void execute() {

        LPLog.INSTANCE.i(TAG, "execute unregister for brandId " + mBrandId);

        // Create pusher unregister URL
        String pusherDomain = mAccountsController.getServiceUrl(mBrandId, ConnectionParamsCache.CSDS_PUSHER_DOMAIN_KEY);
        if (TextUtils.isEmpty(pusherDomain)) {
            pusherDomain = PreferenceManager.getInstance().getStringValue(ConnectionParamsCache.CSDS_PUSHER_DOMAIN_KEY, mBrandId, null);
            if (TextUtils.isEmpty(pusherDomain)) {
                LPLog.INSTANCE.w(TAG, "pusherDomain does not exists. Quit unregister push");
                notifySuccess();
                return;
            }
        }
        String url = String.format(PUSHER_UNREGISTER_URL, pusherDomain, mBrandId);

        try {
            // Use older version of unregister api to remove UnAuth user from pusher database on sign out.
            if (runImmediately && (mAccountsController.getAccount(mBrandId) != null && !mAccountsController.getAccount(mBrandId).isAuthenticated())) {
                url = url.substring(0, url.indexOf('?'));
            }
        } catch (Exception e) {
            // If we failed to get account details, ignore and make unregister request with v 2.0
        }
        final String pusherURL = url;

        // Get consumerId from memory
        String consumerId = mAmsUsers.getConsumerId(mBrandId);

        // If consumerId is not available in memory try to get from DB
        if (TextUtils.isEmpty(consumerId)) {
			LPLog.INSTANCE.d(TAG, "execute: consumerId is not available. Trying to get from DB...");
            mAmsUsers.getConsumerByBrandIDFromDB(mBrandId).setPostQueryOnBackground(dbConsumerId -> {
                if (!TextUtils.isEmpty(dbConsumerId)) {
                    runUnregisterPush(dbConsumerId, pusherURL);
					LPLog.INSTANCE.d(TAG, "onResult: got  consumerId from DB (" + dbConsumerId + "). Unregister push with it...");
                } else {
	                LPLog.INSTANCE.w(TAG, "onResult: Cannot get user profile from DB. Quit unregister push");
                }

            }).execute();
        } else {
            runUnregisterPush(consumerId, pusherURL);
        }
    }

    private void notifySuccess() {
        if (listener != null) {
            listener.onSuccess(null);
        }
    }

    private void runUnregisterPush(final String consumerId, final String pusherURL) {
        final List<String> certificates = mAccountsController.getCertificatePinningKeys(mBrandId);
        if (runImmediately) {
            LPLog.INSTANCE.d(TAG, "run: Unregister push immediately");
            if (InternetConnectionService.isNetworkAvailable()) {
                // If available execute immediately
                new UnregisterPushRequest(pusherURL, consumerId, mAppId, certificates).setCallback(listener).execute();
            } else {
                if (listener != null) {
                    listener.onError(new Exception("No network available"));
                }
            }
        } else {
            new SynchronizedInternetConnectionCallback(() -> {
	            LPLog.INSTANCE.d(TAG, "run: Unregister push for consumerId: " + LPLog.INSTANCE.mask(consumerId));
                new UnregisterPushRequest(pusherURL, consumerId, mAppId, certificates).setCallback(listener).execute();
            }).execute();
        }
    }
}
