package com.liveperson.messaging.model;

import android.os.Bundle;
import android.text.TextUtils;

import com.liveperson.infra.Infra;
import com.liveperson.infra.auth.LPAuthenticationParams;
import com.liveperson.infra.auth.LPAuthenticationType;
import com.liveperson.infra.controller.DBEncryptionHelper;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.messaging.R;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.infra.utils.LocalBroadcast;
import com.liveperson.messaging.utils.TokenUtils;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.controller.connection.IConnectionParamsCache;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Created by Ilya Gazman on 11/10/2015.
 * <p/>
 * Accounts model
 */
public class AmsAccount {

    private static final String TAG = "AmsAccount";

    public static final String KEY_ACCOUNT_TOKEN = "account_token";
    public static final String KEY_ACCOUNT_TOKEN_ENC = "account_token_enc";
    private static final String KEY_ACCOUNT_NON_AUTH_ENC = "account_non_auth_enc";
    //BE CAREFUL !! THIS CODE IS ALSO IN MONITORING MODULE - DON'T FORGET TO CHANGE IT ALSO THERE
    private static final String KEY_CONSUMER_ID_ENC = "account_original_consumer_id_enc";
    private static final String KEY_CONNECTOR_ID_ENC = "account_connector_id_enc";

    private static final String CERTIFICATE_PUBLIC_KEYS_ENC = "certificate_public_keys_enc";

    private static final String QA_CSDS = "hc1n.dev.lprnd.net"; // QA
    private static final String PROD_CSDS = "adminlogin.liveperson.net"; // Production--> its in branding

    public static final String BROADCAST_KEY_AUTH_COMPLETED_ACTION = "BROADCAST_KEY_AUTH_COMPLETED_ACTION";

    private PreferenceManager mPreferenceManager;
	private IConnectionParamsCache mConnectionParamsCache;

    private String mBrandId;
    private String mCSDSDomain;
    private String mToken;
    private LPAuthenticationParams mLPAuthenticationParams;
    private String mAppId;
    private String mNonAuthCode;
    private String mOriginalConsumerId;
    private String mConnectorId;


    public AmsAccount(String brandId) {
        mPreferenceManager = PreferenceManager.getInstance();


        if (brandId.startsWith("qa") || brandId.startsWith("le")) {
            mCSDSDomain = QA_CSDS;
        } else {
            mCSDSDomain = Infra.instance.getApplicationContext().getResources().getString(R.string.csds_url);
        }

        mBrandId = brandId;

        //restore token/jwt
        String decryptedToken = mPreferenceManager.getStringValue(KEY_ACCOUNT_TOKEN_ENC, mBrandId, null);
        if (TextUtils.isEmpty(decryptedToken)){
            mToken = mPreferenceManager.getStringValue(KEY_ACCOUNT_TOKEN, mBrandId, null);
            mPreferenceManager.remove(KEY_ACCOUNT_TOKEN, mBrandId);
            saveToken();
        }else{
            mToken = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1, decryptedToken);
        }

        //restore non-auth code
        String decryptedNonAuth = mPreferenceManager.getStringValue(KEY_ACCOUNT_NON_AUTH_ENC, mBrandId, null);
        if (!TextUtils.isEmpty(decryptedNonAuth)){
            mNonAuthCode = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1, decryptedNonAuth);
        }

        //restore original consumerId
        //BE CAREFUL !! THIS CODE IS ALSO IN MONITORING MODULE - DON'T FORGET TO CHANGE IT ALSO THERE
        String decryptedConsumerId = mPreferenceManager.getStringValue(KEY_CONSUMER_ID_ENC, mBrandId, null);
        if (!TextUtils.isEmpty(decryptedConsumerId)){
            mOriginalConsumerId = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1, decryptedConsumerId);
        }

		 //restore ConnectorID
        String decryptedConnectorId = mPreferenceManager.getStringValue(KEY_CONNECTOR_ID_ENC, mBrandId, null);
        if (!TextUtils.isEmpty(decryptedConnectorId)){
            mConnectorId = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1, decryptedConnectorId);
        }

		mConnectionParamsCache = new ConnectionParamsCache(mBrandId);
        LPLog.INSTANCE.d(TAG, "restoring mOriginalConsumerId = " + LPLog.INSTANCE.mask(mOriginalConsumerId) + ", mConnectorId = " + mConnectorId);
        LPLog.INSTANCE.d(TAG, "restoring data: mNonAuthCode = " + LPLog.INSTANCE.mask(mNonAuthCode));
        LPLog.INSTANCE.d(TAG, "restoring data: mToken = " + LPLog.INSTANCE.mask(mToken));

	}

    public void setToken(String token) {
        mToken = token;
        saveToken();
    }

    private void saveToken() {
        LPLog.INSTANCE.d(TAG, "setToken: storing token in preferences - " + LPLog.INSTANCE.mask(mToken));
        String encryptedToken = DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, mToken);
        mPreferenceManager.setStringValue(KEY_ACCOUNT_TOKEN_ENC, mBrandId, encryptedToken);
    }

    public String getToken() {
        return mToken;
    }

    public void sendAuthenticationCompletedStatus() {
        Bundle connectionBundle = new Bundle();
        connectionBundle.putString(AmsConnection.BROADCAST_KEY_BRAND_ID, mBrandId);
        LocalBroadcast.sendBroadcast(BROADCAST_KEY_AUTH_COMPLETED_ACTION, connectionBundle);
    }


    public void setNonAuthCode(String nonAuthCode) {
        this.mNonAuthCode = nonAuthCode;
        LPLog.INSTANCE.d(TAG, "setNonAuthCode: storing mNonAuthCode in preferences "+ LPLog.INSTANCE.mask(nonAuthCode));
        String encryptedToken = DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, mNonAuthCode);
        mPreferenceManager.setStringValue(KEY_ACCOUNT_NON_AUTH_ENC, mBrandId, encryptedToken);
    }

    public String getNonAuthCode() {
        return mNonAuthCode;
    }

    public LPAuthenticationParams getLPAuthenticationParams() {
        return mLPAuthenticationParams;
    }

    public void setLPAuthenticationParams(LPAuthenticationParams lpAuthenticationParams) {
        mLPAuthenticationParams = lpAuthenticationParams;
        savePublicKey();
    }

    private void savePublicKey(){
        if(mLPAuthenticationParams != null) {
            List<String> decryptedKeys = mLPAuthenticationParams.getCertificatePinningKeys();
            if (decryptedKeys != null && decryptedKeys.size() > 0) {
                Set<String> encryptedKeys = new HashSet<>(decryptedKeys.size());
                for (String key : decryptedKeys) {
                    String encryptedKey = DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, key);
                    encryptedKeys.add(encryptedKey);
                }
                mPreferenceManager.setStringsSet(CERTIFICATE_PUBLIC_KEYS_ENC, mBrandId, encryptedKeys);
            }
        }
    }

    private List<String> getCachedPublicKey(){
        Set<String> encryptedKeys = mPreferenceManager.getStringSet(CERTIFICATE_PUBLIC_KEYS_ENC, mBrandId, null);
        List<String> decryptedKeys = null;
        if(encryptedKeys != null){
            decryptedKeys = new ArrayList<>(encryptedKeys.size());
            for(String key : encryptedKeys){
                String decryptedKey = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1, key);
                decryptedKeys.add(decryptedKey);
            }
        }
        return decryptedKeys;
    }

    public boolean isAuthenticated(){
       return mLPAuthenticationParams.isAuthenticated();
    }

    public boolean isAuthenticatedCompleted(){
        return !TextUtils.isEmpty(mToken);
    }


    /**
     * Getting the certificate keys from memory, or from the cached value as a fallback
     * @return
     */
    public List<String> getCertificatePinningKeys(){
        if (mLPAuthenticationParams != null) {
            return mLPAuthenticationParams.getCertificatePinningKeys();
        }
        return getCachedPublicKey();
    }

    public String getDomain() {
        return mCSDSDomain;
    }

	/**
	 * Set all domains on the account storage
	 * @param csdsUrls
	 * @return true - if one of the relevant values is new, false - non is new
	 */
	public boolean setCSDSMap(HashMap<String, String> csdsUrls) {

		return mConnectionParamsCache.updateCsdsDomains(csdsUrls);
	}

	/**
	 * Check if CSDS data exist in persistent memory
	 * @return true if exist, false if not
	 */
    public boolean isCsdsDataMissing() {

		return !mConnectionParamsCache.isCsdsFilled();
    }

    public String getServiceUrl(String serviceName) {

		if (!mConnectionParamsCache.isCsdsFilled()) {
			return null;
		}

		return mConnectionParamsCache.getServiceDomain(serviceName);
	}

    public void setAppId(String appId) {
        mAppId = appId;
    }

    public String getAppId() {
        return mAppId;
    }

	public IConnectionParamsCache getConnectionParamsCache() {
		return mConnectionParamsCache;
	}

    /**
     * @return true if token exists and expired.
     */
    public boolean isHostAppJWTExpired() {
        String jwt =  mLPAuthenticationParams.getHostAppJWT();

        return TokenUtils.isJwtExpired(jwt);

    }
    /**
     * @return true if token exists and expired.
     */
    public boolean isTokenExpired() {

        String jwt = getToken();

        return TokenUtils.isJwtExpired(jwt);
    }

	/**
	 * Return whether AutoMessages is enabled
	 * @return
	 */
	public boolean isAutoMessagesEnabled(){

		return mConnectionParamsCache.isAutoMessagesFeatureEnabled();
	}


    //BE CAREFUL !! THIS CODE IS ALSO IN MONITORING MODULE - DON'T FORGET TO CHANGE IT ALSO THERE
    public void setOriginalConsumerId(String originalConsumerId) {
        this.mOriginalConsumerId = originalConsumerId;
        String encryptedConsumerId = DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, originalConsumerId);
        mPreferenceManager.setStringValue(KEY_CONSUMER_ID_ENC, mBrandId, encryptedConsumerId);
    }

    public String getOriginalConsumerId() {
        return mOriginalConsumerId;
    }

    public void setConnectorId(String connectorId) {
        this.mConnectorId = connectorId;
        String encryptedConnectorId = DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, connectorId);
        mPreferenceManager.setStringValue(KEY_CONNECTOR_ID_ENC, mBrandId, encryptedConnectorId);
    }

    public String getConnectorId() {
        return mConnectorId;
    }

    public boolean isInUnAuthMode(){
        LPAuthenticationType authType = getLPAuthenticationParams().getAuthType();
	    return authType == LPAuthenticationType.UN_AUTH;
    }
}
