package com.liveperson.messaging.network.socket

import android.os.Bundle
import com.liveperson.api.response.events.MessagingEventNotification
import com.liveperson.api.response.model.Event
import com.liveperson.infra.ICallback
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.model.types.ChatState
import com.liveperson.infra.network.socket.BaseResponseHandler
import com.liveperson.infra.network.socket.BaseSocketRequest
import com.liveperson.infra.utils.LocalBroadcast
import com.liveperson.messaging.Messaging
import com.liveperson.messaging.model.Dialog
import org.json.JSONException
import org.json.JSONObject

/**
 * Created by shiranr on 2/1/18.
 */

class MessagingEventNotificationHandler(internal var mController: Messaging) : BaseResponseHandler<MessagingEventNotification, BaseSocketRequest<*, *>>() {

    companion object {

        private const val TAG = "MessagingEventNotificationHandler"

        const val AGENT_TYPING_ACTION = "agent_typing"
        const val AGENT_TYPING_ACTION_IS_TYPING_EXTRA = "is_typing"
        const val ORIGINATOR_ID_EXTRA = "originator_id"
    }

    override fun getAPIResponseType(): String = "ms.MessagingEventNotification"

    /**
     * Parse the incoming event into the OnlineEventDistribution data object
     *
     * @param jsonObject
     * @return
     * @throws JSONException
     */
    @Throws(JSONException::class)
    override fun parse(jsonObject: JSONObject): MessagingEventNotification? {
        var responseBody: MessagingEventNotification? = null

        try {
            responseBody = MessagingEventNotification(jsonObject)
        } catch (e: JSONException) {
            LPLog.e(TAG, "Error parsing JSON", e)
        }

        return responseBody
    }

    /**
     * Handle the incoming event. The body of the incoming event is ContentEventNotification on which we act upon.
     * This handles 3 event types:
     * AcceptStatusEvent - indicates whether the agent received or read the message the consumer sent it
     * ChatStateEvent - indicates whether the agent is typing or not
     * ContentEvent - an incoming message from the agent
     *
     *
     * Note that on AcceptStatusEvent and on ContentEvent, the server sends an echo message, so we need to ignore
     * an event that is sent with the same originator ID as ours
     *
     * @param event
     */
    override fun handle(event: MessagingEventNotification): Boolean {

        if (event.body.isEmpty()){
            return true
        }
        val dialogId = event.body[0].dialogId

        val firstNotification = mController.messagingEventSubscriptionManager.onReceivedEvent(dialogId)

        val shouldUpdateUi = mController.messagingEventSubscriptionManager.shouldUpdateUI(dialogId)

        mController.amsDialogs.saveMessagesResult(dialogId, event.body, firstNotification, shouldUpdateUi, object : ICallback<Dialog, Exception>{
            override fun onSuccess(dialog: Dialog?) {

                // Get the consumerID from the model
                val consumerId = mController.getOriginatorId(dialog!!.targetId)
                event.body.forEach {
                    LPLog.d(TAG, "saving message in dialog $dialogId")
                    val eventType = it.event.type
                    val originatorId = it.originatorId

                    when (eventType) {
                    // Event whether the agent is typing of not
                        Event.Types.ChatStateEvent -> {

                            // If this message is not from myself
                            if (originatorId != consumerId) {

                                val isTyping = when (it.event.chatState) {
                                    ChatState.COMPOSING -> true
                                    else -> false
                                }

                                // Send a broadcast on the typing status
                                val agentBundle = Bundle()
                                agentBundle.putBoolean(AGENT_TYPING_ACTION_IS_TYPING_EXTRA, isTyping)
                                agentBundle.putString(ORIGINATOR_ID_EXTRA, originatorId)
                                LocalBroadcast.sendBroadcast(AGENT_TYPING_ACTION, agentBundle)

                                mController.mEventsProxy.onAgentTyping(isTyping)

                            }
                        }
                        else -> { /* No-Op */ }
                    }
                }
            }

            override fun onError(exception: Exception?) {

            }
        })

        return true
    }
}
