package com.liveperson.api.response.model;

import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.liveperson.api.response.types.CloseReason;
import com.liveperson.api.response.types.DialogState;
import com.liveperson.api.response.types.DialogType;
import com.liveperson.infra.log.LPLog;
import com.liveperson.messaging.model.ConversationData;
import com.liveperson.messaging.model.Dialog;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Set;

import static com.liveperson.infra.errors.ErrorCode.ERR_0000005D;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000005E;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000005F;

/**
 * Created by oferd on 8/10/15.
 */
public class DialogData extends MultiDialog {
    private static final String TAG = "DialogData";

    public DialogState state;
    public String assignedAgentId;
    public String dialogId;
    public String conversationId;
    @Deprecated
    private String[] participants;
    public Participants participantsDetails; // TODO Perry, replacing the strings array by a Participants object: https://nation.slack.com/archives/C9SU8QRT2/p1526390943000223
    public DialogType dialogType;
    public String closedCause;
    public long creationTs;
    public long endTs;
    public boolean isOpen;

    /**
     * The server calls it 'closedBy' but it's the exact same enum as 'CloseReason'
     */
    public CloseReason closeReason;

    public DialogData(JSONObject jsonObject) {
        super(jsonObject);

        /*
        {"dialogId":"bf54241d-17f6-47ae-bc8c-cf8ab7fd365c","participants":["985b11bdedf5865345dcb413c5fad46a17841cdd6daf3e0949cc340e5d0311b2"],"dialogType":"MAIN","channelType":"MESSAGING","state":"OPEN","creationTs":1521738198961,"metaDataLastUpdateTs":1521738198961}
        {"dialogId":"bf54241d-17f6-47ae-bc8c-cf8ab7fd365c","participants":["985b11bdedf5865345dcb413c5fad46a17841cdd6daf3e0949cc340e5d0311b2"],"dialogType":"MAIN","channelType":"MESSAGING","state":"OPEN","creationTs":1521738198961,"metaDataLastUpdateTs":1521738198961}
        */

        JSONArray participantsArray = jsonObject.optJSONArray("participants");
        if (participantsArray == null) {
            participantsArray = new JSONArray();
        }
        this.participants = new String[participantsArray.length()];
        for (int i = 0; i < participantsArray.length(); i++) {
            String participantId = participantsArray.optString(i);
            if (!TextUtils.isEmpty(participantId)) {
                participants[i] = participantId;
            }
        }

        parseParticipantsDetails(jsonObject);

        dialogId = jsonObject.optString("dialogId");
        closeReason = CloseReason.parse(jsonObject.optString("closedBy"));
        //jsonObject.optLong("metaDataLastUpdateTs");
        creationTs = jsonObject.optLong("creationTs", -1);
        endTs = jsonObject.optLong("endTs", -1);

        closedCause = jsonObject.optString("closedCause");
        dialogType = DialogType.parse(jsonObject.optString("dialogType"));
        channelType = ChannelType.parse(jsonObject.optString("channelType"));
        state = DialogState.parse(jsonObject.optString("state"));
        isOpen = state == DialogState.OPEN;
    }

    protected DialogData(JSONObject jsonObject, @Nullable String conversationId) {
        this(jsonObject);

        if (TextUtils.isEmpty(conversationId)) {
            conversationId = jsonObject.optString("convId");
            if (TextUtils.isEmpty(conversationId)) {
                conversationId = dialogId;
            }
        }

        this.conversationId = conversationId;

        if (TextUtils.isEmpty(dialogId)) {
            dialogId = this.conversationId;
        }
    }

    /**
     * Constructs a legacy dialog. <br/>
     * Basically so we should't need to call this constructor and build a dialog's data from conversation's data.
     * @param conversationData The conversation's data that will supply the dialog's data
     */
    @Deprecated
    protected DialogData(ConversationData conversationData) {
        // todo Perry: In the new UMS architecture we expect to get the dialog inside the conversation data
        this(new JSONObject(), conversationData.conversationId);

        participants = conversationData.participants.extractAllParticipantsIds();
        participantsDetails = conversationData.participants;
        dialogType = DialogType.MAIN;
        channelType = ChannelType.MESSAGING;
        closeReason = conversationData.closeReason;
        setState(DialogState.parse(conversationData.state));
        assignedAgentId = conversationData.getAssignedAgentId();
    }

    public DialogData(Dialog dialog) {
        assignedAgentId = dialog.getAssignedAgentId();
        participantsDetails = dialog.getParticipants();
        participants = participantsDetails.extractAllParticipantsIds();

        channelType = dialog.getChannelType();
        dialogType = dialog.getDialogType();

        conversationId = dialog.getConversationId();
        dialogId = dialog.getDialogId();

        //closedCause = dialog.getCloseReason();
        //closedCause = jsonObject.optString("closedCause");
        closeReason = dialog.getCloseReason();

        endTs = dialog.getEndTimestamp();
        creationTs = dialog.getStartTimestamp();

        state = dialog.getState();
        isOpen = state == DialogState.OPEN;
    }

    private void parseParticipantsDetails(JSONObject jsonObject) {
        JSONArray participantsDetailsJson = jsonObject.optJSONArray("participantsDetails");
        if (participantsDetailsJson != null && participantsDetailsJson.length() > 0) {
            this.participantsDetails = new Participants();
            HashMap<Participants.ParticipantRole, ArrayList<String>> participantsDetailsMap = new HashMap<>();
            for (int i = 0; i < participantsDetailsJson.length(); i++) {
                JSONObject participantDetailsJson = participantsDetailsJson.optJSONObject(i);
                if (participantDetailsJson == null) continue;

                String participantId = participantDetailsJson.optString("id");
                String participantRole = participantDetailsJson.optString("role").toUpperCase();
                if (!TextUtils.isEmpty(participantId)) {
                    Participants.ParticipantRole role = Participants.ParticipantRole.parse(participantRole);
                    if (role != null) {
                        ArrayList<String> ids = participantsDetailsMap.get(role);
                        if (ids == null) {
                            ids = new ArrayList<>();
                            participantsDetailsMap.put(role, ids);
                        }
                        participantsDetailsMap.get(role).add(participantId);
                    }
                }
            }

            Set<Participants.ParticipantRole> roles = participantsDetailsMap.keySet();
            for (Participants.ParticipantRole role: roles) {
                participantsDetails.add(participantsDetailsMap.get(role), role);
            }
        } else {
            LPLog.INSTANCE.e(TAG, ERR_0000005D, "The 'participantsDetails' is empty / missing, is the server's version too old?");
        }
    }

    public static DialogData[] extractDialogsData(ConversationData conversationData) {
        DialogData[] dialogsData;
        if (conversationData.dialogs == null || conversationData.dialogs.length == 0) {
            if (conversationData.dialogs == null) {
                LPLog.INSTANCE.e(TAG, ERR_0000005E, "Missing dialogs array! Creating a new one from the conversation's data.");
            } else {
                LPLog.INSTANCE.e(TAG, ERR_0000005F, "Dialogs array is empty! Creating a new one from the conversation's data.");
            }

            // Constructing legacy dialog
            dialogsData = new DialogData[]{new DialogData(conversationData)};
        } else {
            dialogsData = conversationData.dialogs;
        }

        return dialogsData;
    }

    /**
     * Extracts the dialog Id from the dialog inside the data
     * @param conversationData
     * @return
     */
    public static String extractDialogId(ConversationData conversationData) {
        String dialogId;
        // todo Perry, with the new architecture we expect to get the dialog inside the conversation data
        if (conversationData.dialogs != null && conversationData.dialogs.length > 0 && conversationData.dialogs[0] != null) {
            dialogId = conversationData.dialogs[0].dialogId;
        } else {
            // todo Perry, this handles only the old architecture:
            dialogId = conversationData.conversationId;
        }

        return dialogId;
    }

    @Override
    public String toString() {
        return "{ dialogId: " + dialogId + ", creationTs: " + creationTs + ", endTs: " + endTs + " }";
    }

    public void setState(DialogState dialogState) {
        state = dialogState;
        isOpen = state == DialogState.OPEN;
    }
}
