package com.liveperson.messaging.commands;

import android.text.TextUtils;

import com.liveperson.infra.Command;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.network.socket.SocketManager;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.model.MessagingUserProfile;
import com.liveperson.messaging.model.SynchronizedAmsConnectionCallback;
import com.liveperson.messaging.model.UserProfile;
import com.liveperson.messaging.network.socket.requests.SetUsersProfileRequest;

import static com.liveperson.infra.errors.ErrorCode.ERR_000000E5;

/**
 * Send a setUserProfile request to the server
 */
public class SendSetUserProfileCommand implements Command {

    private static final String TAG = "SendSetUserProfileCommand";

    private final Messaging mController;

    private String mBrandId;
    private UserProfile mUserProfile;

    public SendSetUserProfileCommand(Messaging messagingController, String brandId, UserProfile userProfile) {
        mController = messagingController;
        mBrandId = brandId;
        mUserProfile = userProfile;
    }

    @Override
    public void execute() {
        LPLog.INSTANCE.i(TAG, "Running SetUserProfile request...");
        new SynchronizedAmsConnectionCallback(mController.mConnectionController, mBrandId, () -> {
            if (TextUtils.isEmpty(mBrandId)) {
                LPLog.INSTANCE.e(TAG, ERR_000000E5, "execute: BrandId is empty. Cannot proceed with set user profile");
                return;
            }

            String consumerId = mController.amsUsers.getConsumerId(mBrandId);
            LPLog.INSTANCE.d(TAG, "Consumer id: " + LPLog.INSTANCE.mask(consumerId + "\n" + mUserProfile));

            MessagingUserProfile msgUserProfile = new MessagingUserProfile(
                    mUserProfile.getFirstName(), mUserProfile.getLastName(),
                    com.liveperson.api.response.model.UserProfile.UserType.CONSUMER);
            msgUserProfile.setAvatarUrl(mUserProfile.getAvatarUrl());
            msgUserProfile.setNickname(mUserProfile.getNickname());
            msgUserProfile.setDescription(mUserProfile.getPhoneNumber());
            msgUserProfile.setPrivateData(
                    new com.liveperson.api.response.model.UserProfile.PrivateData("", ""));
            //TODO: do we need the empty PrivateData?
            msgUserProfile.setMobileNumber(mUserProfile.getPhoneNumber());
            msgUserProfile.setOriginatorID(consumerId);

            // Send the request
            SetUsersProfileRequest setUsersProfileRequest = new SetUsersProfileRequest(
                    mController.amsUsers, mController.mAccountsController.getConnectionUrl(mBrandId), mBrandId, msgUserProfile);
            SocketManager.getInstance().send(setUsersProfileRequest);
        }).execute();
    }
}
