package com.liveperson.messaging.commands.pusher;

import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.liveperson.infra.Command;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.auth.LPAuthenticationParams;
import com.liveperson.infra.controller.DBEncryptionHelper;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.network.http.requests.PushRequest;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.model.AmsAccount;
import com.liveperson.messaging.model.SynchronizedAuthenticationCompletedCallback;

import java.util.List;

import static com.liveperson.infra.errors.ErrorCode.*;
import static com.liveperson.infra.managers.PreferenceManager.APP_ID_PREFERENCE_KEY;
import static com.liveperson.infra.managers.PreferenceManager.CONSUMER_UNREGISTER_PUSHER_PREFERENCE_KEY;
import static com.liveperson.infra.managers.PreferenceManager.PUSHER_DEVICE_TOKEN_PREFERENCE_KEY;

/**
 * Created by Ilya Gazman on 11/26/2015.
 */
public class RegisterPusherCommand implements Command {

    private static final String TAG = "RegisterPusherCommand";
    private static final String PUSHER_URL = "https://%s/api/account/%s/device/register";
    private final Messaging mController;
    private final LPAuthenticationParams mLpAuthParams;
    private String mBrandId;
    /** The device token */
    private String mToken;
    private String mAppId;
    @Nullable
    private ICallback<Void, Exception> mRegistrationCompletedCallback;

    public RegisterPusherCommand(Messaging messagingController, String brandId, LPAuthenticationParams lpAuthParams) {
        this(messagingController, brandId, "", "", lpAuthParams, null);
    }

    public RegisterPusherCommand(Messaging messagingController, String brandId, String appId, String token, LPAuthenticationParams lpAuthParams, @Nullable final ICallback<Void, Exception> registrationCompletedCallback) {
        mController = messagingController;
        mBrandId = brandId;
        mToken = token;
        mAppId = appId;
        mLpAuthParams = lpAuthParams;
        mRegistrationCompletedCallback = registrationCompletedCallback;
    }

    @Override
    public void execute() {
        LPLog.INSTANCE.i(TAG, "execute with token " + LPLog.INSTANCE.mask(mToken));
        if (TextUtils.isEmpty(mAppId)) {
            mAppId = PreferenceManager.getInstance().getStringValue(APP_ID_PREFERENCE_KEY, mBrandId, "");
        } else {
            //Save appId in case it's needed for Messaging.liteLogout()
            PreferenceManager.getInstance().setStringValue(APP_ID_PREFERENCE_KEY, mBrandId, mAppId);
        }
        if (TextUtils.isEmpty(mToken)) {
            mToken = PreferenceManager.getInstance().getStringValue(PUSHER_DEVICE_TOKEN_PREFERENCE_KEY, mBrandId, "");
            if (TextUtils.isEmpty(mToken)) {
                LPLog.INSTANCE.w(TAG, "device token is empty. skip register pusher.");
                return;
            }
        } else {
            //Save device token. It's needed to register pusher after consumer switch.(lite logout)
            PreferenceManager.getInstance().setStringValue(PUSHER_DEVICE_TOKEN_PREFERENCE_KEY, mBrandId, mToken);
        }

        boolean isExecuting = new SynchronizedAuthenticationCompletedCallback(mController.mAccountsController, mBrandId, new ICallback<Void, Exception> (){
            @Override
            public void onSuccess(Void value) {
                List<String> certificates = mController.mAccountsController.getCertificatePinningKeys(mBrandId);
                String pusherDomain = mController.mAccountsController.getServiceUrl(mBrandId, ConnectionParamsCache.CSDS_PUSHER_DOMAIN_KEY);
                if (TextUtils.isEmpty(pusherDomain)) {
                    pusherDomain = PreferenceManager.getInstance().getStringValue(ConnectionParamsCache.CSDS_PUSHER_DOMAIN_KEY, mBrandId, null);
                    if (TextUtils.isEmpty(pusherDomain)) {
                        LPLog.INSTANCE.e(TAG, ERR_0000014C, "pusherDomain does not exists. Failed to register push");
                        if (mRegistrationCompletedCallback != null) {
                            mRegistrationCompletedCallback.onError(new Exception("Pusher domain is empty!, SDK may not initialized!"));
                        }
                        return;
                    }
                }

                String pusherURL = String.format(PUSHER_URL, pusherDomain, mBrandId);

                String consumerId = mController.amsUsers.getConsumerId(mBrandId);
                if (TextUtils.isEmpty(consumerId)) {
                    mController.amsUsers.getConsumerByBrandIDFromDB(mBrandId).setPostQueryOnBackground(returnedConsumerId -> {
                        if (returnedConsumerId != null && !TextUtils.isEmpty(returnedConsumerId)) {
                            LPLog.INSTANCE.d(TAG, "fetchConsumerIdAndSendRequest: got consumerId from DB (" + returnedConsumerId + "). get the unread message count with it...");
                            sendRequest(pusherURL, returnedConsumerId, certificates);
                        } else {
                            LPLog.INSTANCE.w(TAG, "fetchConsumerIdAndSendRequest: Cannot get user profile from DB. Quit get badge counter");
                            if (mRegistrationCompletedCallback != null) {
                                mRegistrationCompletedCallback.onError(new Exception("Failed to register pusher. Error: Missing consumerID"));
                            }
                        }
                    }).execute();
                } else {
                    sendRequest(pusherURL, consumerId, certificates);
                }
            }

            @Override
            public void onError(Exception exception) {
                if (mRegistrationCompletedCallback != null) {
                    mRegistrationCompletedCallback.onError(exception);
                }
            }

            private void sendRequest(String pusherURL, String consumerId, List<String> certificates) {
                LPLog.INSTANCE.i(TAG, "Running register pusher with token " + LPLog.INSTANCE.mask(mToken));
                String token = mController.mAccountsController.getToken(mBrandId);
                if (TextUtils.isEmpty(token)) {
                    token = getTokenFromSharedPreferences();
                }
                new PushRequest(pusherURL, consumerId, mAppId, mToken, token, certificates)
                        .setCallback(new ICallback<Void, Exception>() {
                            @Override
                            public void onSuccess(Void value) {
                                PreferenceManager.getInstance().setBooleanValue(PreferenceManager.IS_PUSHER_REGISTERED_PREFERENCE_KEY, mBrandId, true);
                                PreferenceManager.getInstance().remove(CONSUMER_UNREGISTER_PUSHER_PREFERENCE_KEY, mBrandId);
                                if (mRegistrationCompletedCallback != null) {
                                    mRegistrationCompletedCallback.onSuccess(value);
                                }
                            }

                            @Override
                            public void onError(Exception exception) {
                                if (mRegistrationCompletedCallback != null) {
                                    mRegistrationCompletedCallback.onError(exception);
                                }
                            }
                        })
                        .execute();
            }

            private String getTokenFromSharedPreferences() {
                String token;
                String decryptedToken = PreferenceManager.getInstance().getStringValue(AmsAccount.KEY_ACCOUNT_TOKEN_ENC, mBrandId, null);
                if (TextUtils.isEmpty(decryptedToken)) {
                    token = PreferenceManager.getInstance().getStringValue(AmsAccount.KEY_ACCOUNT_TOKEN, mBrandId, null);
                    PreferenceManager.getInstance().remove(AmsAccount.KEY_ACCOUNT_TOKEN, mBrandId);
                } else {
                    token = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1, decryptedToken);
                }
                return token;
            }

        }).executeWithReturnValue();
        //in case we have valid jwt we will execute the register request. otherwise, we'll try to connect.
        if (!isExecuting && mLpAuthParams != null){
            mController.connect(mBrandId, mLpAuthParams, null, true, false);
        }
    }
}
