package com.liveperson.messaging.utils;

import android.text.TextUtils;
import android.util.Base64;

import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.log.FlowTags;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.messaging.R;

import org.json.JSONObject;

import java.nio.charset.StandardCharsets;
import java.util.Date;

import static com.liveperson.infra.errors.ErrorCode.ERR_000000B9;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000BA;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000BB;

/**
 * Created by nirni on 11/7/17.
 */
public class TokenUtils {

	private static final String TAG = "TokenUtils";

	private static int jwtExpirationBuffer = -1;


	public static boolean isJwtExpired(String jwt) {

		// Get it once
		if(jwtExpirationBuffer != -1) {
			jwtExpirationBuffer = Configuration.getInteger(R.integer.lp_messaging_buffer_expiration_seconds);
		}

		if (TextUtils.isEmpty(jwt)) {
			//no jwt exists, we'll run normal flow
			return false;
		} else {
			LPLog.INSTANCE.d(TAG, FlowTags.LOGIN, "jwt exists: " + LPLog.INSTANCE.mask(jwt) + " checking if expired..");

			String [] segments = jwt.split("\\.");
			if (segments.length < 3){
				return false;
			}

			//take the second segment, decode it (base64), convert it to json object, and find "exp" value.
			try {
				byte[] secondSegmentsByte = Base64.decode(segments[1], Base64.DEFAULT);
				String secondSegments = new String(secondSegmentsByte, StandardCharsets.UTF_8);
				JSONObject secondSegmentsJson = new JSONObject(secondSegments);
				long expirationInSeconds = secondSegmentsJson.getLong("exp");

				long iat = secondSegmentsJson.getLong("iat");
				long expirationInMillis = expirationInSeconds * 1000;
				LPLog.INSTANCE.d(TAG, FlowTags.LOGIN, "expiration = " + new Date(expirationInMillis).toString());
				LPLog.INSTANCE.d(TAG, FlowTags.LOGIN, "iat = " + new Date(iat*1000).toString());

				if (isGoingToExpire(expirationInMillis)){
					LPLog.INSTANCE.d(TAG, FlowTags.LOGIN, "JWT is expired or about to.. ");
					return true;
				}else{
					LPLog.INSTANCE.d(TAG, FlowTags.LOGIN, "JWT is still valid ");
				}

			} catch (Exception e) {
				LPLog.INSTANCE.e(TAG, ERR_000000B9, "Exception while checking if JWT is expired.", e);
			}
			return false;

		}
	}

	private static boolean isGoingToExpire(long expirationInMillis) {
		long bufferTime = jwtExpirationBuffer*1000;
		long time = System.currentTimeMillis() + bufferTime;
		return time > expirationInMillis;
	}


	public static String getOriginalConsumerIdFromJWT(String jwt) {

		if (TextUtils.isEmpty(jwt)) {
			//no jwt exists, we'll run normal flow
			return "";
		} else {
			LPLog.INSTANCE.d(TAG, FlowTags.LOGIN, "Getting original consumerId from jwt: " + LPLog.INSTANCE.mask(jwt));

			String [] segments = jwt.split("\\.");
			if (segments.length < 3){
				return "";
			}
			//example:
			// 			{
			//				"sub": "b555ec39a12c560c2c76b43f5e218ccbd46c981318bd0df1c94db72aa48b9d95",
			//					"aud": "acc:le22225965",
			//					"acr": "0",
			//					"iss": "https://idp.liveperson.net",
			//					"lp.ext": {
			//							"sub": "1fcbeb5e-a245-4096-8beb-5ea24560964e",
			//							"iss": "https://idp.liveperson.net"
			//					},
			//					"exp": 1524988657,
			//					"iat": 1524988057
			//			}
			//take the second segment, decode it (base64), convert it to json object, and find "lp.ext"."sub" value.
			try {
				byte[] secondSegmentsByte = Base64.decode(segments[1], Base64.DEFAULT);
				String secondSegments = new String(secondSegmentsByte, StandardCharsets.UTF_8);
				JSONObject secondSegmentsJson = new JSONObject(secondSegments);

				JSONObject lpExt = secondSegmentsJson.optJSONObject("lp.ext");
				if (lpExt != null){
					return lpExt.optString("sub");
				}

			} catch (Exception e) {
				LPLog.INSTANCE.e(TAG, ERR_000000BA, "Exception while getting original consumer ID from JWT.", e);
			}
			return "";

		}
	}


	public static String getConsumerUserId(String jwt) {

		if (TextUtils.isEmpty(jwt)) {
			//no jwt exists, we'll run normal flow
			return "";
		} else {
			LPLog.INSTANCE.d(TAG, FlowTags.LOGIN, "Getting original consumerId from jwt: " + LPLog.INSTANCE.mask(jwt));

			String [] segments = jwt.split("\\.");
			if (segments.length < 3){
				return "";
			}
			//example:
			// 			{
			//				****"sub": "b555ec39a12c560c2c76b43f5e218ccbd46c981318bd0df1c94db72aa48b9d95",*****
			//					"aud": "acc:le22225965",
			//					....
			//					"exp": 1524988657,
			//					"iat": 1524988057
			//			}
			//take the second segment, decode it (base64), convert it to json object, and find "lp.ext"."sub" value.
			try {
				byte[] secondSegmentsByte = Base64.decode(segments[1], Base64.DEFAULT);
				String secondSegments = new String(secondSegmentsByte, StandardCharsets.UTF_8);
				JSONObject secondSegmentsJson = new JSONObject(secondSegments);
				return secondSegmentsJson.optString("sub");
			} catch (Exception e) {
				LPLog.INSTANCE.e(TAG, ERR_000000BB, "Exception while getting Consumer User ID.", e);
			}
			return "";

		}
	}
}
