package com.liveperson.messaging.background;

import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.util.LongSparseArray;
import android.text.TextUtils;
import android.util.SparseArray;
import android.widget.Toast;

import com.liveperson.api.response.types.ConversationState;
import com.liveperson.infra.Infra;
import com.liveperson.infra.InternetConnectionService;
import com.liveperson.infra.LocalBroadcastReceiver;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.messaging.R;
import com.liveperson.infra.network.socket.SocketManager;
import com.liveperson.infra.network.socket.SocketState;
import com.liveperson.infra.utils.DateUtils;
import com.liveperson.infra.utils.DispatchQueue;
import com.liveperson.infra.utils.LocalBroadcast;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.MessagingFactory;
import com.liveperson.messaging.background.filesharing.BaseUploadTask;
import com.liveperson.messaging.background.filesharing.DownloadFileTaskBundle;
import com.liveperson.messaging.background.filesharing.FileSharingType;
import com.liveperson.messaging.background.filesharing.ReUploadFileTaskBundle;
import com.liveperson.messaging.background.filesharing.UploadFileTaskBundle;
import com.liveperson.messaging.background.filesharing.document.DownloadDocumentTask;
import com.liveperson.messaging.background.filesharing.document.UploadDocumentTask;
import com.liveperson.messaging.background.filesharing.document.UploadDocumentTaskBundle;
import com.liveperson.messaging.background.filesharing.image.DownloadImageTask;
import com.liveperson.messaging.background.filesharing.image.ReUploadImageTaskBundle;
import com.liveperson.messaging.background.filesharing.image.UploadImageTaskBundle;
import com.liveperson.messaging.background.filesharing.voice.DownloadVoiceTask;
import com.liveperson.messaging.background.filesharing.voice.ReUploadVoiceTask;
import com.liveperson.messaging.background.filesharing.voice.UploadVoiceTask;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.exception.FileSharingException;
import com.liveperson.messaging.model.AmsConnection;
import com.liveperson.messaging.model.Conversation;
import com.liveperson.messaging.network.http.RestRequestParams;

import java.io.File;
import java.util.ArrayList;
import java.util.concurrent.CopyOnWriteArrayList;

import static com.liveperson.infra.errors.ErrorCode.ERR_00000090;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000091;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000092;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000093;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000094;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000095;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000096;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000097;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000098;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000099;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000009A;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000009B;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000009C;

/**
 * Created by nirni on 7/4/16.
 */
public class FileSharingManager extends DispatchQueue implements BackgroundActionsService.ServiceActioner {

	private static final String TAG = "FileSharingManager";
	private static final String BRAND_ID = "brandId";
	private static final String TARGET_ID = "targetId";
	private static final String FILE_URI = "uri";
	private static final String FILE_URI_LIST = "uriList";
	private static final String RELATIVE_PATH = "relativePath";
	private static final String FILE_SHARING_TYPE = "fileSharingType";
	private static final String FILE_ROW_ID = "fileRowId";
	private static final String MESSAGE_ROW_ID = "messageRowId";
	private static final String CAPTION = "caption";
	private static final String FILE_FROM_CAMERA = "fileFromCamera";
	private static final String EVENT_ID = "EVENT_ID";
	private static final String THUMBNAIL_LOCAL_URI_PATH = "THUMBNAIL_LOCAL_URI_PATH";
	private static final String ORIGINAL_LOCAL_URI_PATH = "ORIGINAL_LOCAL_URI_PATH";
	private static final String ORIGINAL_MESSAGE_TIME = "ORIGINAL_MESSAGE_TIME";
	private static final String ORIGINAL_CONVERSATION_ID = "ORIGINAL_CONVERSATION_ID";

	private static final int TYPE_UPLOAD_MSG_NEW_FILE = 1;
	private static final int TYPE_RE_UPLOAD_MSG_NEW_FILE = 2;
	private static final int TYPE_DOWNLOAD_MSG_NEW_FILE = 3;
    private static final int TYPE_MSG_CONNECTION_TIMEOUT = 4;
	private static final int TYPE_REMOVE_DOWNLOADED_LOCAL_FILES = 5;

	private static int UPLOAD_TASK_ID = 0;

	public static final String SERVICE_EXTRA_BRAND_ID = "service_extra_brand_id";
	public static final String SERVICE_EXTRA_TARGET_ID = "service_extra_target_id";
	public static final String SERVICE_EXTRA_FILE_URI = "service_extra_file_uri";
	public static final String SERVICE_EXTRA_FILE_CAPTION = "service_extra_file_caption";
	public static final String SERVICE_EXTRA_FILE_ROW_ID = "service_extra_file_row_id";
	public static final String SERVICE_EXTRA_MESSAGE_ROW_ID = "service_extra_message_row_id";
	public static final String SERVICE_EXTRA_IMAGE_FROM_CAMERA = "service_extra_image_from_camera";
	public static final String SERVICE_EXTRA_MESSAGE = "service_extra_message";
	public static final String SERVICE_EXTRA_EVENT_ID = "service_extra_event_id";
	public static final String SERVICE_EXTRA_ORIGINAL_MESSAGE_TIME = "extra_original_message_time";
	public static final String SERVICE_EXTRA_CONVERSATION_ID = "extra_conversation_id";

	public static final String BROADCAST_FILE_UPLOAD_FAILED = "BROADCAST_FILE_UPLOAD_FAILED";
	public static final String KEY_FILE_UPLOAD_ERROR = "KEY_FILE_UPLOAD_ERROR";

	private final Context mContext;
	private final Messaging mController;
	private String mSwiftDomain;
	private RestRequestParams mRestParams = new RestRequestParams();
	private final int mDownloadTimeoutMs;
	private final int mUploadImageTimeoutMs;
	private final int mUploadVoiceTimeoutMs;
	private final int mUploadDocumentTimeoutMs;
	private LocalBroadcastReceiver mLocalBroadcastReceiver = null;

	private final LongSparseArray<FileDownloadProgressListener> mFileDownloadProgressListener = new LongSparseArray<>();
	private final SparseArray<FileUploadProgressListener> mFileUploadProgressListener = new SparseArray<>();

	private final CopyOnWriteArrayList<BaseUploadTask> mQueuedUploadFiles = new CopyOnWriteArrayList<>();
	private final CopyOnWriteArrayList<DownloadFileTask> mQueuedDownloadFiles = new CopyOnWriteArrayList<>();

	public FileSharingManager(Messaging messaging, Context context) {
		super(TAG);
		mContext = context;
		mController = messaging;

		// Get the upload timeout from resources
		mDownloadTimeoutMs = Configuration.getInteger(R.integer.download_file_timeout_ms);
		mUploadImageTimeoutMs = Configuration.getInteger(R.integer.image_upload_timeout_ms);
		mUploadVoiceTimeoutMs = Configuration.getInteger(R.integer.voice_upload_timeout_ms);
		mUploadDocumentTimeoutMs = Configuration.getInteger(R.integer.document_upload_timeout_ms);

		//initializing the handler onHandleMessage
		setHandleMessageCallback(msg -> {
			LPLog.INSTANCE.d(TAG, "onHandleMessage");
			//handle timeout for uploading file.
			//aborting the connection and cancelling all the tasks.
			FileSharingType fileSharingType;
			if (msg.what == TYPE_MSG_CONNECTION_TIMEOUT) {
				if (!mQueuedUploadFiles.isEmpty() || !mQueuedDownloadFiles.isEmpty()) {
					LPLog.INSTANCE.d(TAG, "Timeout for sending files. aborting.");
					abortConnection();
				}
				return;
			}
			int fileType = msg.getData().getInt(FILE_SHARING_TYPE, -1);
			if (fileType != -1) {
				fileSharingType = FileSharingType.values()[fileType];
			} else {
				fileSharingType = FileSharingType.UNKNOWN;
			}
			//handle send new file message request. creating new task and running it.
			switch (msg.arg1) {
				case TYPE_UPLOAD_MSG_NEW_FILE:
					handleNewUploadRequest(fileSharingType, msg);
					break;
				case TYPE_RE_UPLOAD_MSG_NEW_FILE:
					handleReUploadRequest(fileSharingType, msg);
					break;
				//handle download image message request. creating new task and running it.
				case TYPE_DOWNLOAD_MSG_NEW_FILE:
					handleNewDownloadRequest(fileSharingType, msg);
					break;
				case TYPE_REMOVE_DOWNLOADED_LOCAL_FILES:
					Bundle data = msg.getData();
					ArrayList<String> filePathList = data.getStringArrayList(FILE_URI_LIST);

					if (filePathList != null) {
						removeLocalFilesFromDirectoryAndFilePathsFromDB(filePathList);
					}
					break;
				default:
					LPLog.INSTANCE.e(TAG, ERR_00000090, "Unknown message type " + msg.arg1  + " found");
					break;
			}
		});
	}

	public void downloadFile(FileSharingType fileSharingType, String brandId, String targetId, String swiftFileRelativePath, long messageRowId, long fileRowId, String conversationID, FileDownloadProgressListener fileProgressListener) {

		Message msg = new Message();
		Bundle data = new Bundle();
		msg.arg1 = TYPE_DOWNLOAD_MSG_NEW_FILE;
		data.putString(BRAND_ID, brandId);
		data.putString(TARGET_ID, targetId);
		data.putString(RELATIVE_PATH, swiftFileRelativePath);
		data.putLong(FILE_ROW_ID, fileRowId);
		data.putLong(MESSAGE_ROW_ID, messageRowId);
		data.putString(ORIGINAL_CONVERSATION_ID, conversationID);
		data.putInt(FILE_SHARING_TYPE, fileSharingType.ordinal());
		msg.setData(data);
		//fileRowId is the task id.
		if (mFileDownloadProgressListener.get(fileRowId) == null) {
			LPLog.INSTANCE.d(TAG, "Adding download image task");
			mFileDownloadProgressListener.put(fileRowId, fileProgressListener);
			sendMessage(msg);
		} else {
			//if this task already in progress:
			//update the listener, but don't need to add another task to download the same file.
			LPLog.INSTANCE.d(TAG, "Adding download file listener, task for this file is already exists.");
			mFileDownloadProgressListener.put(fileRowId, fileProgressListener);
		}
	}

	public void uploadFile(FileSharingType fileSharingType, String brandId, String targetId, String fileUri, String caption, boolean fileFromCamera, FileUploadProgressListener fileUploadProgressListener) {
		Message msg = Message.obtain();
		Bundle data = new Bundle();

		msg.arg1 = TYPE_UPLOAD_MSG_NEW_FILE;
		data.putString(BRAND_ID, brandId);
		data.putString(TARGET_ID, targetId);
		data.putString(FILE_URI, fileUri);
		data.putString(CAPTION, caption);
		data.putBoolean(FILE_FROM_CAMERA, fileFromCamera);
		data.putInt(FILE_SHARING_TYPE, fileSharingType.ordinal());
		msg.setData(data);

		int taskId = UPLOAD_TASK_ID++;
		msg.arg2 = taskId;
		mFileUploadProgressListener.put(taskId, fileUploadProgressListener);
		sendMessage(msg);
	}

	public void reUploadFile(final FileSharingType fileSharingType, final String brandId, final String targetId, final String message,
							  final String eventId, final long originalMessageTime,
							  final long fileRowId, final FileUploadProgressListener fileProgressListener) {

		mController.amsFiles.getFileByFileRowId(fileRowId).setPostQueryOnBackground(fileMessage -> {
			Message msg = Message.obtain();
			Bundle data = new Bundle();
			msg.arg1 = TYPE_RE_UPLOAD_MSG_NEW_FILE;
			data.putString(BRAND_ID, brandId);
			data.putString(TARGET_ID, targetId);
			data.putString(EVENT_ID, eventId);
			data.putLong(ORIGINAL_MESSAGE_TIME, originalMessageTime);
			data.putLong(FILE_ROW_ID, fileRowId);
			data.putString(ORIGINAL_LOCAL_URI_PATH, fileMessage.getLocalUrl());
			data.putString(THUMBNAIL_LOCAL_URI_PATH, fileMessage.getPreview());
			data.putInt(FILE_SHARING_TYPE, fileSharingType.ordinal());
			data.putString(CAPTION, message);

			msg.setData(data);
			int taskId = UPLOAD_TASK_ID++;
			msg.arg2 = taskId;
			mFileUploadProgressListener.put(taskId, fileProgressListener);
			sendMessage(msg);
		}).execute();
	}

	/**
	 * Check if we have more than the full size file limit and if so remove the oldest one
	 */
	public void removeMultipleOlderFiles(final String targetId, final int maxNumberOfStoredFiles, @Nullable final String fileTypesString) {

		LPLog.INSTANCE.d(TAG, "removeMultipleOlderFiles: removing older files if greater than: " + maxNumberOfStoredFiles + ". fileTypeString: " + fileTypesString);

		// Get number of full size files from DB
		MessagingFactory.getInstance().getController().amsFiles.getNumOfLocalPathFromDB(targetId, fileTypesString).setPostQueryOnBackground(count -> {

			LPLog.INSTANCE.d(TAG, "removeMultipleOlderFiles: number of localUrl exist in DB: " + count + " (fileTypeString = " + fileTypesString + ")");

			// If need to remove older file
			if (count > maxNumberOfStoredFiles) {
				final ArrayList<String> filesToRemove = MessagingFactory.getInstance().getController().amsFiles.getMultipleOldestLocalPathFromDB(targetId, (count - maxNumberOfStoredFiles), fileTypesString).executeSynchronously();

				if (filesToRemove != null) {

					LPLog.INSTANCE.d(TAG, "removeMultipleOlderFiles: going to remove older files: " + filesToRemove);

					// Add a message to remove a file
					Message msg = Message.obtain();
					Bundle data = new Bundle();
					msg.arg1 = TYPE_REMOVE_DOWNLOADED_LOCAL_FILES;
					data.putStringArrayList(FILE_URI_LIST, filesToRemove);
					data.putString(TARGET_ID, targetId);
					msg.setData(data);

					sendMessage(msg);
				} else {
					LPLog.INSTANCE.w(TAG, "onResult: received empty localUrl");
				}
			}

		}).execute();

	}

	/**
	 * Get a comma separated string with all currently uploading images' rowIds
	 * @return
	 */
	public String getInProgressUploadMessageRowIdsString() {

		StringBuilder result = new StringBuilder();

		for (BaseUploadTask uploadFileTask : mQueuedUploadFiles) {

			result.append(uploadFileTask.getMessageRowId());
			result.append(",");
		}

		return result.length() > 0 ? result.substring(0, result.length() - 1) : "";
	}

	private void handleNewDownloadRequest(FileSharingType fileType, Message msg) {
		Bundle data = msg.getData();
		String brandId = data.getString(BRAND_ID);
		String targetId = data.getString(TARGET_ID);
		String relativePath = data.getString(RELATIVE_PATH);
		long fileRowId = data.getLong(FILE_ROW_ID);
		long messageRowId = data.getLong(MESSAGE_ROW_ID);
		String conversationId = data.getString(ORIGINAL_CONVERSATION_ID);


		LPLog.INSTANCE.d(TAG, "runNewDownloadFileTask: data.getString(RELATIVE_PATH) = " + relativePath + " fileRowId = " + fileRowId
				+  " messageRowId = " + messageRowId + " conversationId = " + conversationId);

		updateServicesUrl(brandId);
		//setting new timeout timer.
		setTimeout(mDownloadTimeoutMs);
		runNewDownloadFileTask(fileType, brandId, targetId, relativePath, messageRowId, fileRowId, fileRowId, conversationId);
	}

	private void handleNewUploadRequest(FileSharingType fileType, Message msg) {
		Bundle data = msg.getData();
		String brandId = data.getString(BRAND_ID);
		String targetId = data.getString(TARGET_ID);
		Uri fileUri = Uri.parse(data.getString(FILE_URI));
		String caption = data.getString(CAPTION);
		boolean imageFromCamera = data.getBoolean(FILE_FROM_CAMERA, false);
		LPLog.INSTANCE.d(TAG, "runNewUploadFileTask: data.getString(FILE_URI) = " + data.getString(FILE_URI));
		updateServicesUrl(brandId);

		int taskId = msg.arg2;

		BaseUploadTask uploadFileTask = null;

		try{

			if (fileType.getCommonFileType() == FileSharingType.CommonFileType.IMAGE) {
				UploadImageTaskBundle params = new UploadImageTaskBundle();
				params.addMessage(mController.getMaskedMessage(brandId, caption)).addBrandId(brandId).addTargetId(targetId).
						addFileUri(fileUri).addSwiftDomain(mSwiftDomain).addRestDomain(mRestParams).addImageFromCamera(imageFromCamera).build(taskId, mContext);

				uploadFileTask = new UploadImageTask(params, mUploadImageTimeoutMs);

				//setting new timeout timer.
				setTimeout(mUploadImageTimeoutMs);
			}
			else if(fileType.getCommonFileType() == FileSharingType.CommonFileType.AUDIO){
				UploadFileTaskBundle params = new UploadFileTaskBundle();
				params.addMessage(mController.getMaskedMessage(brandId, caption)).addBrandId(brandId).addTargetId(targetId).
						addFileUri(fileUri).addSwiftDomain(mSwiftDomain).addRestDomain(mRestParams).build(taskId, mContext);
				uploadFileTask = new UploadVoiceTask(params, mUploadVoiceTimeoutMs);

				//setting new timeout timer.
				setTimeout(mUploadVoiceTimeoutMs);
			} else if (fileType.getCommonFileType() == FileSharingType.CommonFileType.DOCUMENT) {
				UploadDocumentTaskBundle params = new UploadDocumentTaskBundle();
				params.addMessage(mController.getMaskedMessage(brandId, caption)).addBrandId(brandId).addTargetId(targetId).
						addFileUri(fileUri).addSwiftDomain(mSwiftDomain).addRestDomain(mRestParams).build(taskId, mContext);
				uploadFileTask = new UploadDocumentTask(mContext, params, mUploadDocumentTimeoutMs, false);
				setTimeout(mUploadDocumentTimeoutMs);
			}

			if (uploadFileTask == null) {
				LPLog.INSTANCE.e(TAG, ERR_00000091, "handleNewUploadRequest: cannot crate UploadTask");
				return;
			}

			runNewUploadFileTask(uploadFileTask, brandId, taskId);

		} catch (FileSharingException e) {
			LPLog.INSTANCE.e(TAG, ERR_00000092, "Exception while handling new upload request.", e);
			sendFileUploadFailedStatus(e);
			mFileUploadProgressListener.get(taskId).onFailedUpload(e);
			mFileUploadProgressListener.remove(taskId);
			checkNoMoreDownloadTasks();
		}
	}


	private void handleReUploadRequest(FileSharingType fileType, final Message msg) {
		Bundle data = msg.getData();
		final String eventId = data.getString(EVENT_ID);
		int taskId = msg.arg2;

		for (BaseUploadTask task : mQueuedUploadFiles) {
			String eventId1 = task.getEventId();
			LPLog.INSTANCE.d(TAG, "createNewReUploadImageTask: event id: " + eventId + " taskEventId =" + eventId1);
			if (TextUtils.equals(eventId1, eventId)){
				LPLog.INSTANCE.d(TAG, "createNewReUploadImageTask: event id: " + eventId + " is already in progress. no need to resend.");
				return;
			}
		}

		final String brandId = data.getString(BRAND_ID);
		final String targetId = data.getString(TARGET_ID);
		String caption = data.getString(CAPTION);
		final String originalFilePath = data.getString(ORIGINAL_LOCAL_URI_PATH);
		final String previewFileUri = data.getString(THUMBNAIL_LOCAL_URI_PATH);
		final long originalMessageTime = data.getLong(ORIGINAL_MESSAGE_TIME);
		final long fileRowId = data.getLong(FILE_ROW_ID);
		boolean imageFromCamera = data.getBoolean(FILE_FROM_CAMERA, false);
		LPLog.INSTANCE.d(TAG, "createNewReUploadImageTask: thumbnailLocalUriPath = " + previewFileUri);
		updateServicesUrl(brandId);

		BaseUploadTask uploadFileTask = null;
		try {
			if (fileType.getCommonFileType() == FileSharingType.CommonFileType.IMAGE) {
				//ReUploadImageTaskBundle params = new ReUploadImageTaskBundle(mContext, brandId, targetId, taskId, mSwiftDomain, mRestParams, Uri.parse(originalFilePath), mController.getMaskedMessage(brandId, caption), imageFromCamera, previewFileUri, eventId, originalMessageTime, fileRowId);
				ReUploadImageTaskBundle params = new ReUploadImageTaskBundle();
				params.addFileRowId(fileRowId)
						.addOriginalLocalPath(originalFilePath)
						.addThumbnailLocalPath(previewFileUri)
						.addEventID(eventId)
						.addOriginalMessageTime(originalMessageTime)
						.addBrandId(brandId)
						.addTargetId(targetId)
						.addFileUri(Uri.parse(originalFilePath))
						.addSwiftDomain(mSwiftDomain)
						.addRestDomain(mRestParams)
						.addMessage(mController.getMaskedMessage(brandId, caption))
						.addImageFromCamera(imageFromCamera)
						.build(taskId, mContext);
				uploadFileTask = new ReUploadImageTask(params, mUploadImageTimeoutMs);

				//setting new timeout timer.
				setTimeout(mUploadImageTimeoutMs);

			} else if (fileType.getCommonFileType() == FileSharingType.CommonFileType.DOCUMENT) {
				UploadDocumentTaskBundle params = new UploadDocumentTaskBundle();
				params.addMessage(mController.getMaskedMessage(brandId, caption))
						.addBrandId(brandId)
						.addTargetId(targetId)
						.addFileUri(Uri.parse(originalFilePath))
						.addSwiftDomain(mSwiftDomain)
						.addRestDomain(mRestParams)
						.addEventID(eventId)
						.addFileRowId(fileRowId)
						.addOriginalMessageTime(originalMessageTime)
						.build(taskId, mContext);
				uploadFileTask = new UploadDocumentTask(mContext, params, mUploadDocumentTimeoutMs, true);
				setTimeout(mUploadDocumentTimeoutMs);
			} else if(fileType.getCommonFileType() == FileSharingType.CommonFileType.AUDIO){
				ReUploadFileTaskBundle params = new ReUploadFileTaskBundle();
				params.addFileRowId(fileRowId).addEventID(eventId).addOriginalMessageTime(originalMessageTime).addMessage(mController.getMaskedMessage(brandId, caption)).addBrandId(brandId).addTargetId(targetId).
						addFileUri(Uri.parse(originalFilePath)).addSwiftDomain(mSwiftDomain).addRestDomain(mRestParams).build(taskId, mContext);
				uploadFileTask = new ReUploadVoiceTask(params, mUploadVoiceTimeoutMs);

				//setting new timeout timer.
				setTimeout(mUploadVoiceTimeoutMs);

			}

			if (uploadFileTask == null) {
				LPLog.INSTANCE.e(TAG, ERR_00000093, "handleNewUploadRequest: uploadFileTask is null");
				return;
			}

			runNewUploadFileTask(uploadFileTask, brandId, taskId);
		} catch (FileSharingException e) {
			LPLog.INSTANCE.e(TAG, ERR_00000094, "Exception while handling upload request.", e);
			mFileUploadProgressListener.get(taskId).onFailedUpload(e);
			mFileUploadProgressListener.remove(taskId);
			checkNoMoreDownloadTasks();
		}
	}



	private void updateServicesUrl(String brandId) {
		updateSwiftDomain(brandId);
		updateRestParams(brandId);
	}

	/**
	 * On each new task we set a new timeout timer, and cancel the old one.
	 */
	private void setTimeout(int timeout) {
		//removing old timer
		removeTimer();

		//set new timer from this moment.
		Message msg = new Message();
		msg.what = TYPE_MSG_CONNECTION_TIMEOUT;
		sendMessage(msg, timeout);
	}

	private void removeTimer() {
		//remove old timeout
		removeMessage(TYPE_MSG_CONNECTION_TIMEOUT);
	}

	/**
	 * update swift domain from csds
	 */
	private void updateSwiftDomain(String brandId) {
		if (TextUtils.isEmpty(mSwiftDomain)) {
			mSwiftDomain = mController.mAccountsController.getServiceUrl(brandId, ConnectionParamsCache.CSDS_SWIFT_DOMAIN_KEY);
			if (TextUtils.isEmpty(mSwiftDomain)) {
				LPLog.INSTANCE.w(TAG, "No swift url from csds! can;t upload image.");// TODO: 8/9/16 Don't throw here. Return an error or catch at the caller
				abortConnection();

			}
		}

	}

	/**
	 * update swift domain from csds
	 */
	private void updateRestParams(String brandId) {
		if (mRestParams.isNotValid()) {
			mRestParams.setParams(brandId,
					mController.mAccountsController.getServiceUrl(brandId, ConnectionParamsCache.CSDS_UMS_DOMAIN_KEY),
					mController.mAccountsController.getToken(brandId), mController.mAccountsController.getCertificatePinningKeys(brandId));

			if (mRestParams.isNotValid()) {
				LPLog.INSTANCE.w(TAG, "No asyncMessagingEnt url from csds! can;t upload image.");// TODO: 8/9/16 Don't throw here. Return an error or catch at the caller
			}
		}
	}


	/**
	 * create new upload image task. adding it to list of current uploading tasks (mQueuedImages).
	 * after each image is done uploading, we check if they are the first in line to be send and
	 * calling  {@link UploadImageTask#sendPublishFile(boolean)}. if not the first in line, we wait til
	 * the earlier tasks will complete uploading.
	 */
	private void runNewUploadFileTask(BaseUploadTask uploadFileTask, final String brandId, final int taskId) {
		uploadFileTask.setCallBack(new UploadFileTaskCallback() {

			/**
			 * First step - save file in db. that does not require an available connection.
			 * when connection is available we will call {@link BaseUploadTask#onConnectionAvailable()}
			 * and it will start uploading.
			 */
			@Override
			public void onFileAddedToDB() {
				waitForConnection(brandId);
			}

			/**
			 * when upload completed check if publish message can be send.
			 */
			@Override
			public void onUploadFinishedSuccessfully(final BaseUploadTask currentUploadFileTask) {
				postRunnable(() -> {
					boolean sendViaRest = false;

					if (InternetConnectionService.isNetworkAvailable() && isSocketClose(brandId)) {
						LPLog.INSTANCE.w(TAG, "onUploadFinishedSuccessfully: Socket is closed, Failing Message. " + currentUploadFileTask.getEventId());

						onUploadFailed(currentUploadFileTask, new Exception("No open socket"));
						return;
					}
					for (BaseUploadTask uploadFileTask1 : mQueuedUploadFiles) {
						if (uploadFileTask1.isUploadCompleted()) {
							LPLog.INSTANCE.d(TAG, "onUploadFinishedSuccessfully: isUploadCompleted, sending message "+ uploadFileTask1.getEventId());
							uploadFileTask1.sendPublishFile(sendViaRest);

							mQueuedUploadFiles.remove(uploadFileTask1);
							int currentTaskId = uploadFileTask1.getTaskId();
							LPLog.INSTANCE.d(TAG, "sending message "+ uploadFileTask1.getEventId() + " currentTaskId = " + currentTaskId);
							mFileUploadProgressListener.get(currentTaskId).onDoneUpload();
							mFileUploadProgressListener.remove(currentTaskId);
						} else {
							LPLog.INSTANCE.d(TAG, "onUploadFinishedSuccessfully: isUploadCompleted, waiting for earlier messages... "+ uploadFileTask1.getEventId());
							return;
						}
					}
					checkNoMoreUploadTasks();
				});

			}

			/**
			 * upload failed. removing the failed task
			 * @param uploadFileTask
			 * @param exception
			 */
			@Override
			public void onUploadFailed(BaseUploadTask uploadFileTask, Throwable exception) {
				LPLog.INSTANCE.d(TAG, "get: "  + taskId );
				mQueuedUploadFiles.remove(uploadFileTask);
				mFileUploadProgressListener.get(taskId).onFailedUpload(exception);
				mFileUploadProgressListener.remove(taskId);
				checkNoMoreDownloadTasks();
				LPLog.INSTANCE.w(TAG, "onUploadFailed: Upload Failed!. exception = " + exception.getMessage() + uploadFileTask.getEventId());

			}
		});
		mQueuedUploadFiles.add(uploadFileTask);
		uploadFileTask.startUpload();
	}


	/**
	 * create new upload file task. adding it to list of current uploading tasks (mQueuedImages).
	 * after each image is done uploading, we check if they are the first in line to be send and
	 * calling  {@link UploadImageTask#sendPublishFile(boolean)}. if not the first in line, we wait til
	 * the earlier tasks will complete uploading.
	 * @param relativePath
	 * @param fileRowId
	 * @param taskId       - the file row id is the task unique id.
	 * @param conversationId
	 */
	private void runNewDownloadFileTask(FileSharingType fileType, final String brandId, String targetId, String relativePath, long messageRowId, long fileRowId, final long taskId, final String conversationId) {

		LPLog.INSTANCE.d(TAG, "runNewDownloadFileTask: relativePath = " + relativePath);

		final DownloadFileTask downloadFileTask;
		try {
			DownloadFileTaskBundle params = new DownloadFileTaskBundle();
			params.addRelativePath(relativePath)
					.addBrandId(brandId)
					.addMessageRowId(messageRowId)
					.addFileRowId(fileRowId)
					.addTargetId(targetId)
					.addSwiftDomain(mSwiftDomain)
					.addRestDomain(mRestParams)
					.addConversationId(conversationId)
					.build(mContext);

			downloadFileTask = createDownloadFileTask(fileType, params);
		} catch (FileSharingException e) {
			LPLog.INSTANCE.e(TAG, ERR_00000095, "runNewDownloadFileTask: cannot create downloadTask", e);
			return;
		}

		downloadFileTask.setCallBack(new DownloadFileTaskCallback() {

			@Override
			public void onReadyToGetUrl() {
				boolean getViaRest = false;

				if (InternetConnectionService.isNetworkAvailable() && isSocketClose(brandId)) {
					LPLog.INSTANCE.w(TAG, "onReadyToGetUrl: Socket is closed, running via rest");
					getViaRest = true;
				}
				LPLog.INSTANCE.w(TAG, "onReadyToGetUrl: running via rest = " + getViaRest);

				Conversation conversation = mController.amsConversations.getConversationById(brandId, conversationId).executeSynchronously();
				boolean isFromInca = conversation.getState() == ConversationState.CLOSE && !DateUtils.isInTheLast24hours(conversation.getEndTimestamp());

				downloadFileTask.startDownload(isFromInca);
			}

			@Override
			public void onDownloadFinishedSuccessfully(String fileLocalPath) {
				mQueuedDownloadFiles.remove(downloadFileTask);
				mFileDownloadProgressListener.get(taskId).onDoneDownload();
				mFileDownloadProgressListener.remove(taskId);
				LPLog.INSTANCE.d(TAG, "onDownloadFinishedSuccessfully: Download Completed. fullImageLocalPath = " + fileLocalPath);
				checkNoMoreDownloadTasks();
			}

			/**
			 * download failed. removing the failed task
			 * @param downloadFileTask
			 * @param exception
			 */
			@Override
			public void onDownloadFailed(DownloadFileTask downloadFileTask, Throwable exception) {
				mQueuedDownloadFiles.remove(downloadFileTask);
				mFileDownloadProgressListener.get(taskId).onFailedDownload(exception);
				mFileDownloadProgressListener.remove(taskId);
				checkNoMoreDownloadTasks();
				LPLog.INSTANCE.w(TAG, "onDownloadFailed: Download Failed!. exception = ", exception);
			}
		});
		mQueuedDownloadFiles.add(downloadFileTask);
		waitForConnection(brandId);
	}

	private void removeLocalFilesFromDirectoryAndFilePathsFromDB(final ArrayList<String> imageFilePathList) {

		for (String filePath : imageFilePathList) {

			File fileToDelete = new File(filePath);

			if (fileToDelete.isFile()) {

				LPLog.INSTANCE.d(TAG, "removeLocalFilesFromDirectoryAndFilePathsFromDB: deleting file: " + filePath);

				// Delete the image file from the file system
				if (fileToDelete.delete()) {

					LPLog.INSTANCE.d(TAG, "removeLocalFilesFromDirectoryAndFilePathsFromDB: file "+ filePath + "removed successfully");

					// Remove the full image url from the DB
					MessagingFactory.getInstance().getController().amsFiles.removeLocalPathFromDB(filePath).setPostQueryOnBackground(numOfRowsRemoved -> {
						if (numOfRowsRemoved == 1) {
							LPLog.INSTANCE.d(TAG, "onResult: Image LocalUrl " + filePath + " was removed from DB");
						} else if (numOfRowsRemoved == 0) {
							LPLog.INSTANCE.w(TAG, "onResult: no localUrl was removed");
						} else {
							LPLog.INSTANCE.w(TAG, "onResult: number of rows removed: " + numOfRowsRemoved);
						}
					}).execute();

				} else {
					LPLog.INSTANCE.w(TAG, "removeLocalFilesFromDirectoryAndFilePathsFromDB: file was not removed (" + filePath + ")");
				}

			} else {
				LPLog.INSTANCE.w(TAG, "removeLocalFilesFromDirectoryAndFilePathsFromDB: File to remove is not a file (" + filePath + ")");
			}
		}
	}


	private boolean isSocketClose(String brandId) {
		String connectionUrl = mController.mAccountsController.getConnectionUrl(brandId);
		SocketState state = SocketManager.getInstance().getSocketState(connectionUrl);
		LPLog.INSTANCE.i(TAG, "Current socket state: " + state);
		switch (state) {
			case CONNECTING:
			case OPEN:
				return false;
			default:
				return true;
		}
	}

	/**
	 * making sure there are not more tasks in mQueuedImages,if so - removing the timeout timer,
	 * unregister the relevant receivers and notify listeners {@link FileUploadProgressListener#onDoneUpload()}
	 */
	private void checkNoMoreUploadTasks() {
		if (mQueuedUploadFiles.isEmpty()) { // TODO: 8/9/16 do we need to synchronize here to be sure that in the meanwhile no new messages go in?
			LPLog.INSTANCE.d(TAG, "Finished handling all the messages");
			removeTimer();
			unregisterReceiver();
		}
	}


	/**
	 * making sure there are not more tasks in mQueuedImages,if so - removing the timeout timer,
	 * unregister the relevant receivers and notify listeners {@link FileDownloadProgressListener#onDoneDownload()}}
	 */
	private void checkNoMoreDownloadTasks() {
		if (mQueuedDownloadFiles.isEmpty()) { // TODO: 8/9/16 do we need to synchronize here to be sure that in the meanwhile no new messages go in?
			LPLog.INSTANCE.d(TAG, "Finished handling all the messages");
			removeTimer();
			unregisterReceiver();
		}
	}


	private boolean isFilesInProgress() {
		return (mQueuedUploadFiles.size() > 0 || mQueuedDownloadFiles.size() > 0);
	}

	/**
	 * create new download task
	 *
	 *
	 * @param fileType
	 * @param params
	 * @return
	 */
	@NonNull
	private DownloadFileTask createDownloadFileTask(FileSharingType fileType, DownloadFileTaskBundle params) throws FileSharingException {

		switch (fileType.getCommonFileType()) {

			case IMAGE: {
				return new DownloadImageTask(params);
			}
		    case AUDIO: {
				return new DownloadVoiceTask(params);
			}
		    case DOCUMENT: {
		    	return new DownloadDocumentTask(params,fileType);
			}
		}

		String msg = "createDownloadFileTask: cannot create DownloadFileTask. Received unknown file type: " + fileType;
		LPLog.INSTANCE.e(TAG, ERR_00000096, msg);
		throw new FileSharingException(msg);
	}

	/**
	 * Wait for connection to UMS to start the upload.
	 */
	private void waitForConnection(final String brandId) {
		LPLog.INSTANCE.d(TAG, "waiting for connection..................");

		if (MessagingFactory.getInstance().getController().mConnectionController.isSocketReady(brandId) &&
				MessagingFactory.getInstance().getController().mConnectionController.isUpdated(brandId)) {
			//when updated from server we need to update the message's conversation id
			updateConnectionAvailable();
			return;
		}
		if (!InternetConnectionService.isNetworkAvailable()) {
			abortConnection();
			return;
		}
		//register for future connections events
		if (mLocalBroadcastReceiver == null) {
			mLocalBroadcastReceiver = new LocalBroadcastReceiver.Builder()
					.addAction(AmsConnection.BROADCAST_AMS_CONNECTION_UPDATE_ACTION)
					.addAction(AmsConnection.BROADCAST_KEY_SOCKET_READY_ACTION)
					.build((context, intent) -> {

						if (intent.getAction().equals(AmsConnection.BROADCAST_KEY_SOCKET_READY_ACTION)) {
							String brand = intent.getStringExtra(AmsConnection.BROADCAST_KEY_BRAND_ID);
							if (brandId.equals(brand)) {
								boolean isConnected = intent.getBooleanExtra(AmsConnection.BROADCAST_KEY_SOCKET_READY_EXTRA, false);
								if (!isConnected) {
									//in case we lost connection we aborting waiting tasks.

									// if there is no network available - abort.
									// if there is network available - it will be sent via rest..
									if (!InternetConnectionService.isNetworkAvailable()) {
										abortConnection();
									}
								}
							}
						} else if (intent.getAction().equals(AmsConnection.BROADCAST_AMS_CONNECTION_UPDATE_ACTION)) {

							boolean isUpdated = intent.getBooleanExtra(AmsConnection.BROADCAST_AMS_CONNECTION_UPDATE_EXTRA, false);
							LPLog.INSTANCE.d(TAG, "waiting for connection - got update, connected = " + isUpdated);

							if (isUpdated) {
								//when updated from server we need to update the message's conversation id
								updateConnectionAvailable();
							} else {
								//in case we lost connection we aborting waiting tasks.

								// if there is no network available - abort.
								// if there is network available - it will be sent via rest..
								if (!InternetConnectionService.isNetworkAvailable()) {
									abortConnection();
								}
							}
						}
					});
		}
	}

	/**
	 * cancel all the waiting tasks in case of error (connection problem or timeout)
	 */
	private void abortConnection() {
		LPLog.INSTANCE.d(TAG, "Connection unavailable. aborting waiting tasks..");

		unregisterReceiver();
		for (BaseUploadTask uploadFileTask : mQueuedUploadFiles) {
			uploadFileTask.onConnectionUnavailable();
		}
		for (DownloadFileTask downloadFileTask : mQueuedDownloadFiles) {
			downloadFileTask.onConnectionUnavailable();
		}
	}

	private void unregisterReceiver() {
		if (mLocalBroadcastReceiver != null) {
			mLocalBroadcastReceiver.unregister();
			mLocalBroadcastReceiver = null;

		}
	}

	/**
	 * notify all waiting tasks that connection is available
	 */
	private void updateConnectionAvailable() {
		for (BaseUploadTask uploadFileTask : mQueuedUploadFiles) {
			uploadFileTask.onConnectionAvailable();
		}
		for (DownloadFileTask downloadFileTask : mQueuedDownloadFiles) {
			downloadFileTask.onConnectionAvailable();
		}
	}

	///////////////// ServiceActioner implementation //////////////////////////////////////

	@Override
	public void actionFromService(Intent intent, BackgroundActionsService.ServiceActionCallbackListener serviceActionCallbackListener) {

		int type = intent.getIntExtra(BackgroundActionsService.EXTRA_ACTION_TYPE, -1);
		int fileType = intent.getIntExtra(BackgroundActionsService.EXTRA_FILE_TYPE, -1);
		LPLog.INSTANCE.d(TAG, "actionFromService: new action for service. Type = " + type);

		if ((type == -1)|| (fileType == -1)){
			LPLog.INSTANCE.e(TAG, ERR_00000097, "actionFromService: received type -1. Cannot proceed with action");
			String brandId = intent.getStringExtra(SERVICE_EXTRA_BRAND_ID);
			serviceActionCallbackListener.onFail(brandId);
			return;
		}

		FileSharingType fileSharingType = FileSharingType.values()[fileType];

		switch (type) {
			// TODO NN: 5/8/18 support voice
			case BackgroundActionsService.EXTRA_ACTION_TYPE_UPLOAD:
				uploadFileFromService(fileSharingType, intent, serviceActionCallbackListener);
				break;
			case BackgroundActionsService.EXTRA_TYPE_ACTION_DOWNLOAD:
				downloadFileFromService(fileSharingType, intent, serviceActionCallbackListener);
				break;
			case BackgroundActionsService.EXTRA_TYPE_ACTION_REUPLOAD:
				reUploadFileFromService(fileSharingType, intent, serviceActionCallbackListener);
				break;
		}

	}

	@Override
	public boolean isPendingActions() {
		return isFilesInProgress();
	}

	/////////////////////// Private actions for ServiceActioner ////////////////////////////////////
	/**
	 * Download a file
	 * @param intent
	 */
	private void downloadFileFromService(FileSharingType fileSharingType, Intent intent, final BackgroundActionsService.ServiceActionCallbackListener serviceActionCallbackListener) {
		final String brandId = intent.getStringExtra(SERVICE_EXTRA_BRAND_ID);
		String targetId = intent.getStringExtra(SERVICE_EXTRA_TARGET_ID);
		String swiftUri = intent.getStringExtra(SERVICE_EXTRA_FILE_URI);
		long fileRowID = intent.getLongExtra(SERVICE_EXTRA_FILE_ROW_ID, -1);
		long messageRowID = intent.getLongExtra(SERVICE_EXTRA_MESSAGE_ROW_ID, -1);
		String conversationId = intent.getStringExtra(SERVICE_EXTRA_CONVERSATION_ID);
		if (TextUtils.isEmpty(swiftUri)){
			LPLog.INSTANCE.e(TAG, ERR_00000098, "downloadFile: Error getting one of the required params for uploading a file");
		}

		LPLog.INSTANCE.d(TAG, "downloadFile: starting a thread from the service. Download Params: swiftUri=" + swiftUri);

		downloadFile(fileSharingType, brandId, targetId, swiftUri, messageRowID, fileRowID, conversationId, new FileDownloadProgressListener() {
			@Override
			public void onDoneDownload() {
				serviceActionCallbackListener.onSuccess(brandId);
			}

			@Override
			public void onFailedDownload(Throwable e) {
				LPLog.INSTANCE.e(TAG, ERR_00000099, "onFailedDownload", e);
				showToastMessage(R.string.lp_failed_download_toast_message);
				serviceActionCallbackListener.onFail(brandId);
			}

		});
	}

	/**
	 * Upload a file
	 * @param intent
	 */
	private void uploadFileFromService(FileSharingType fileSharingType, Intent intent, final BackgroundActionsService.ServiceActionCallbackListener serviceActionCallbackListener) {
		final String brandId = intent.getStringExtra(SERVICE_EXTRA_BRAND_ID);
		String targetId = intent.getStringExtra(SERVICE_EXTRA_TARGET_ID);
		String imageUriString = intent.getStringExtra(SERVICE_EXTRA_FILE_URI);
		String captionText = intent.getStringExtra(SERVICE_EXTRA_FILE_CAPTION);
		boolean imageFromCamera = intent.getBooleanExtra(SERVICE_EXTRA_IMAGE_FROM_CAMERA, false);

		if (TextUtils.isEmpty(imageUriString)){
			LPLog.INSTANCE.e(TAG, ERR_0000009A, "uploadImage: Error getting one of the required params for uploading an image");
		}

		LPLog.INSTANCE.d(TAG, "uploadImage: starting a thread from the service. Upload Params: imageUri=" + imageUriString);

		uploadFile(fileSharingType, brandId, targetId, imageUriString, captionText, imageFromCamera, new FileUploadProgressListener() {
			@Override
			public void onDoneUpload() {
				serviceActionCallbackListener.onSuccess(brandId);
				LPLog.INSTANCE.i(TAG, "Consumer successfully sent file. Type: " + fileSharingType);
				if (Infra.instance.isInitialized()) {
					Infra.instance.getLoggos().reportFeatureStatistics();
				}
			}

			@Override
			public void onFailedUpload(Throwable e) {
				LPLog.INSTANCE.e(TAG, ERR_0000009B, "Failed to send file. Type: " + fileSharingType + ". Reason: ", e);
                if (e.getMessage().equals("This file type is not supported")) {
                    showToastMessage(R.string.lp_failed_file_type_not_supported);
                } else {
                    showToastMessage(R.string.lp_failed_upload_toast_message);
                }
				serviceActionCallbackListener.onFail(brandId);
			}
		});
	}

	/**
	 * Re-upload an image
	 * @param intent
	 */
	private void reUploadFileFromService(FileSharingType fileSharingType, Intent intent, final BackgroundActionsService.ServiceActionCallbackListener serviceActionCallbackListener) {
		final String brandId = intent.getStringExtra(SERVICE_EXTRA_BRAND_ID);
		String targetId = intent.getStringExtra(SERVICE_EXTRA_TARGET_ID);
		String message = intent.getStringExtra(SERVICE_EXTRA_MESSAGE);
		String eventId = intent.getStringExtra(SERVICE_EXTRA_EVENT_ID);
		long originalMessageTime = intent.getLongExtra(SERVICE_EXTRA_ORIGINAL_MESSAGE_TIME, -1);
		long fileRowId = intent.getLongExtra(SERVICE_EXTRA_FILE_ROW_ID, -1);

		LPLog.INSTANCE.d(TAG, "reUploadImage: starting a thread from the service. Upload Params: eventId = " + eventId + ", fileRowId = " + fileRowId + ", message = " + LPLog.INSTANCE.mask(message));

		reUploadFile(fileSharingType, brandId, targetId, message, eventId, originalMessageTime,
				fileRowId, new FileUploadProgressListener() {
					@Override
					public void onDoneUpload() {
						serviceActionCallbackListener.onSuccess(brandId);
						LPLog.INSTANCE.i(TAG, "Consumer successfully sent file. Type: " + fileSharingType);
						if (Infra.instance.isInitialized()) {
							Infra.instance.getLoggos().reportFeatureStatistics();
						}
					}
					@Override
					public void onFailedUpload(Throwable e) {
						LPLog.INSTANCE.e(TAG, ERR_0000009C, "Failed to send file. Type: " + fileSharingType + ". Reason: ", e);
						showToastMessage(R.string.lp_failed_upload_toast_message);
						serviceActionCallbackListener.onFail(brandId);
					}
				});

	}

	/**
	 * Show a toast message on the main thread
	 * @param messageId
	 */
	private void showToastMessage(final int messageId) {
		Handler handler = new Handler(Looper.getMainLooper());
		handler.post(() -> Toast.makeText(mContext, messageId, Toast.LENGTH_LONG).show());
	}

	/**
	 * Send Broadcast intent when failed to upload file
	 * @param e exception
	 */
	private void sendFileUploadFailedStatus(Exception e) {
		if (e.getMessage() != null) {
			Bundle extras = new Bundle();
			extras.putString(KEY_FILE_UPLOAD_ERROR, e.getMessage());
			LocalBroadcast.sendBroadcast(BROADCAST_FILE_UPLOAD_FAILED, extras);
		}
	}


	///////////////////////// Interfaces /////////////////////////////////////////////

	public interface FileUploadProgressListener {

		void onDoneUpload();

		void onFailedUpload(Throwable exception);
	}

	public interface FileDownloadProgressListener {

		void onDoneDownload();

		void onFailedDownload(Throwable exception);
	}
}
