package com.liveperson.messaging.controller.connection;

import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.utils.VersionUtils;
import com.liveperson.monitoring.MonitoringFactory;

import java.util.HashMap;

import static com.liveperson.infra.errors.ErrorCode.ERR_0000000F;

/**
 * Created by nirni on 6/6/17.
 */

public class ConnectionParamsCache implements IConnectionParamsCache {

	private static final String TAG = "ConnectionParamsCache";


	public static final String CSDS_UMS_DOMAIN_KEY = "asyncMessagingEnt";
	public static final String CSDS_AMS_TOKENIZER_DOMAIN_KEY = "tokenizer";
	public static final String CSDS_IDP_DOMAIN_KEY = "idp";
	public static final String CSDS_PUSHER_DOMAIN_KEY = "pusher";
	public static final String CSDS_AC_CDN_DOMAIN_KEY = "acCdnDomain";
	public static final String CSDS_LE_CDN_DOMAIN_KEY = "leCdnDomain";
	public static final String CSDS_LOGGOS_DOMAIN_KEY = "loggos";
	public static final String CSDS_SWIFT_DOMAIN_KEY = "swift";
	public static final String CSDS_INCA_KEY = "msgHist";

	public static final String AC_CDN_VERSION_KEY = "ac_cdn_version_key";
	public static final String LE_CDN_VERSION_KEY = "le_cdn_version_key";

	public static final String FULL_CONNECTION_FLOW_REQUIRED_KEY = "full_connection_flow_required_key";

	public static final String AUTO_MESSAGES_ENABLED_KEY = "auto_messages_enabled_key";

	private final String mBrandId;

	// CSDS domains
	private String mCsdsUmsDomain;
	private String mCsdsTokenizerDomain;
	private String mCsdsIdpDomain;
	private String mCsdsPusherDomain;
	private String mCsdsAcCdnDomain;
	private String mCsdsLeCdnDomain;
	private String mCsdsLoggosDomain;
	private String mCsdsSwiftDomain;
	private String mIncaDomain;

	// Version from ac_cdn
	private String mAcCdnSdkMinVersion;

	// Version from le_cdn
	private String mLeCdnSdkMinVersion;

	private boolean mFullConnectionFlowRequired;

	private boolean mAutoMessagesFeatureEnabled;

	public ConnectionParamsCache(String brandId) {
		mBrandId = brandId;

		// Update memory from Shared Preferences
		mCsdsUmsDomain = PreferenceManager.getInstance().getStringValue(CSDS_UMS_DOMAIN_KEY, mBrandId, null);
		mCsdsTokenizerDomain = PreferenceManager.getInstance().getStringValue(CSDS_AMS_TOKENIZER_DOMAIN_KEY, mBrandId, null);
		mCsdsIdpDomain = PreferenceManager.getInstance().getStringValue(CSDS_IDP_DOMAIN_KEY, mBrandId, null);
		mCsdsPusherDomain = PreferenceManager.getInstance().getStringValue(CSDS_PUSHER_DOMAIN_KEY, mBrandId, null);
		mCsdsAcCdnDomain = PreferenceManager.getInstance().getStringValue(CSDS_AC_CDN_DOMAIN_KEY, mBrandId, null);
		mCsdsLeCdnDomain = PreferenceManager.getInstance().getStringValue(CSDS_LE_CDN_DOMAIN_KEY, mBrandId, null);
		mCsdsLoggosDomain = PreferenceManager.getInstance().getStringValue(CSDS_LOGGOS_DOMAIN_KEY, mBrandId, null);
		mCsdsSwiftDomain = PreferenceManager.getInstance().getStringValue(CSDS_SWIFT_DOMAIN_KEY, mBrandId, null);
		mIncaDomain = PreferenceManager.getInstance().getStringValue(CSDS_INCA_KEY, mBrandId, null);

		mAcCdnSdkMinVersion = PreferenceManager.getInstance().getStringValue(AC_CDN_VERSION_KEY, mBrandId, null);
		mLeCdnSdkMinVersion = PreferenceManager.getInstance().getStringValue(LE_CDN_VERSION_KEY, mBrandId, null);

		mFullConnectionFlowRequired = PreferenceManager.getInstance().getBooleanValue(FULL_CONNECTION_FLOW_REQUIRED_KEY, mBrandId, true);

		mAutoMessagesFeatureEnabled = PreferenceManager.getInstance().getBooleanValue(AUTO_MESSAGES_ENABLED_KEY, mBrandId, false);
	}


	/**
	 * Update CSDS domains in SharedPreferences and in memory.
	 * Return true if one of the domains that are relevant to the SDK are changed, false if non changed.
	 * If the SharedPreferences was empty before updating - return true
	 * @param csdsDomains
	 * @return
	 */
	@Override
	public boolean updateCsdsDomains(HashMap<String, String> csdsDomains) {

		boolean updated = false;

		if (csdsDomains == null) {
			LPLog.INSTANCE.e(TAG, ERR_0000000F, "updateCsdsDomains: domains received are null");
			return false;
		}


		// Set UMS domain
		if (updateDomain(csdsDomains, CSDS_UMS_DOMAIN_KEY, mCsdsUmsDomain)) {
			updated = true;
		}

		// Set Tokenizer domain
		if (updateDomain(csdsDomains, CSDS_AMS_TOKENIZER_DOMAIN_KEY, mCsdsTokenizerDomain)) {
			updated = true;
		}

		// Set IDP domain
		if (updateDomain(csdsDomains, CSDS_IDP_DOMAIN_KEY, mCsdsIdpDomain)) {
			updated = true;
		}

		// Set Pusher domain
		if (updateDomain(csdsDomains, CSDS_PUSHER_DOMAIN_KEY, mCsdsPusherDomain)) {
			updated = true;
		}

		// Set acCdn domain
		if (updateDomain(csdsDomains, CSDS_AC_CDN_DOMAIN_KEY, mCsdsAcCdnDomain)) {
			updated = true;
		}

		// Set leCdn domain
		if (updateDomain(csdsDomains, CSDS_LE_CDN_DOMAIN_KEY, mCsdsLeCdnDomain)) {
			updated = true;
		}

		// Set Loggos domain
		if (updateDomain(csdsDomains, CSDS_LOGGOS_DOMAIN_KEY, mCsdsLoggosDomain)) {
			updated = true;
		}

		// Set Swift domain
		if (updateDomain(csdsDomains, CSDS_SWIFT_DOMAIN_KEY, mCsdsSwiftDomain)) {
			updated = true;
		}

		// Set INCA domain
		if (updateDomain(csdsDomains, CSDS_INCA_KEY, mIncaDomain)) {
			updated = true;
		}

		//set the domains also for monitoring, in case we need to send request using monitoring
		if (MonitoringFactory.INSTANCE.isInitialized()) {
			MonitoringFactory.INSTANCE.getMonitoring().getParamsCache().updateCsdsDomains(csdsDomains);
		}
		return updated;
	}

	@Override
	public void updateAcCdnVersion(String newVersion) {
		if (newVersion != null){
			PreferenceManager.getInstance().setStringValue(AC_CDN_VERSION_KEY, mBrandId, newVersion);
			mAcCdnSdkMinVersion = newVersion;
		}
	}

	@Override
	public void updateLeCdnVersion(String newVersion) {
		if (newVersion != null){
			PreferenceManager.getInstance().setStringValue(LE_CDN_VERSION_KEY, mBrandId, newVersion);
			mLeCdnSdkMinVersion = newVersion;
		}
	}

	@Override
	public boolean isCsdsFilled(){
		return mCsdsUmsDomain != null &&
				mCsdsTokenizerDomain != null &&
				mCsdsIdpDomain != null &&
				mCsdsPusherDomain != null &&
				mCsdsAcCdnDomain != null &&
				mCsdsLeCdnDomain != null &&
				mCsdsLoggosDomain != null &&
				mIncaDomain != null;
	}



	@Override
	public String getServiceDomain(String serviceName) {

		String domainToReturn = null;

		switch (serviceName) {
			case CSDS_UMS_DOMAIN_KEY:
				domainToReturn = mCsdsUmsDomain;
				break;
			case CSDS_AMS_TOKENIZER_DOMAIN_KEY:
				domainToReturn = mCsdsTokenizerDomain;
				break;
			case CSDS_IDP_DOMAIN_KEY:
				domainToReturn = mCsdsIdpDomain;
				break;
			case CSDS_PUSHER_DOMAIN_KEY:
				domainToReturn = mCsdsPusherDomain;
				break;
			case CSDS_AC_CDN_DOMAIN_KEY:
				domainToReturn = mCsdsAcCdnDomain;
				break;
			case CSDS_LE_CDN_DOMAIN_KEY:
				domainToReturn = mCsdsLeCdnDomain;
				break;
			case CSDS_LOGGOS_DOMAIN_KEY:
				domainToReturn = mCsdsLoggosDomain;
				break;
			case CSDS_SWIFT_DOMAIN_KEY:
				domainToReturn = mCsdsSwiftDomain;
				break;
			case CSDS_INCA_KEY:
				domainToReturn = mIncaDomain;
				break;
		}

		return domainToReturn;
	}

	@Override
	public boolean isVersionsCompatible(){

		return !(mLeCdnSdkMinVersion != null && mAcCdnSdkMinVersion != null &&
				(!VersionUtils.isValidSdkVersion(mAcCdnSdkMinVersion) || !VersionUtils.isValidSdkVersion(mLeCdnSdkMinVersion)));

	}

	public boolean isFullConnectionFlowRequired() {
		return mFullConnectionFlowRequired;
	}

	public void setFullConnectionFlowRequired(boolean fullConnectionFlowRequired) {
		mFullConnectionFlowRequired = fullConnectionFlowRequired;
		PreferenceManager.getInstance().setBooleanValue(FULL_CONNECTION_FLOW_REQUIRED_KEY, mBrandId, fullConnectionFlowRequired);

	}

	public boolean isAutoMessagesFeatureEnabled() {
		return mAutoMessagesFeatureEnabled;
	}

	public void setAutoMessagesFeatureEnabled(boolean autoMessagesFeatureEnabled) {
		mAutoMessagesFeatureEnabled = autoMessagesFeatureEnabled;
		PreferenceManager.getInstance().setBooleanValue(AUTO_MESSAGES_ENABLED_KEY, mBrandId, autoMessagesFeatureEnabled);

	}

	/**
	 * Update specific domain
	 * @param csdsDomains
	 * @param csdsDomainKey
	 * @param oldDomainValue
	 * @return
	 */
	private boolean updateDomain(HashMap<String, String> csdsDomains, final String csdsDomainKey, String oldDomainValue) {
		boolean updated = false;

		String domain = csdsDomains.get(csdsDomainKey);
		if (domain != null) {
			PreferenceManager.getInstance().setStringValue(csdsDomainKey, mBrandId, domain);
			if (oldDomainValue != null && !domain.equals(oldDomainValue)) {
				updated = true;
			}
			switch (csdsDomainKey) {
				case CSDS_UMS_DOMAIN_KEY:
					mCsdsUmsDomain = domain;
					break;
				case CSDS_AMS_TOKENIZER_DOMAIN_KEY:
					mCsdsTokenizerDomain = domain;
					break;
				case CSDS_IDP_DOMAIN_KEY:
					mCsdsIdpDomain = domain;
					break;
				case CSDS_PUSHER_DOMAIN_KEY:
					mCsdsPusherDomain = domain;
					break;
				case CSDS_AC_CDN_DOMAIN_KEY:
					mCsdsAcCdnDomain = domain;
					break;
				case CSDS_LE_CDN_DOMAIN_KEY:
					mCsdsLeCdnDomain = domain;
					break;
				case CSDS_LOGGOS_DOMAIN_KEY:
					mCsdsLoggosDomain = domain;
					break;
				case CSDS_SWIFT_DOMAIN_KEY:
					mCsdsSwiftDomain = domain;
					break;
				case CSDS_INCA_KEY:
					mIncaDomain = domain;
					break;
			}
		}

		return updated;
	}
}
