package com.liveperson.messaging.controller.connection;

import com.liveperson.infra.Infra;
import com.liveperson.infra.statemachine.interfaces.IState;
import com.liveperson.infra.statemachine.interfaces.IStateMachine;
import com.liveperson.messaging.commands.tasks.BaseAmsAccountConnectionTask;
import com.liveperson.messaging.commands.tasks.CloseConnectionTask;
import com.liveperson.messaging.commands.tasks.CollectSDKConfigurationDataTask;
import com.liveperson.messaging.commands.tasks.ConfigurationFetcherTask;
import com.liveperson.messaging.commands.tasks.CsdsTask;
import com.liveperson.messaging.commands.tasks.DataBaseTask;
import com.liveperson.messaging.commands.tasks.IdpTask;
import com.liveperson.messaging.commands.tasks.LptagTask;
import com.liveperson.messaging.commands.tasks.OpenSocketTask;
import com.liveperson.messaging.commands.tasks.RemoveOlderImagesTask;
import com.liveperson.messaging.commands.tasks.SiteSettingsFetcherTask;
import com.liveperson.messaging.commands.tasks.UnAuthGetEngagementTask;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by nirni on 6/5/17.
 */

public class ConnectionTasksHolder {

	private final BaseAmsAccountConnectionTask mCsdsTask;
	private final BaseAmsAccountConnectionTask mLptagTask;
	private final BaseAmsAccountConnectionTask mConfigurationFetcherTask;
	private final BaseAmsAccountConnectionTask mSiteSettingsFetcherTask;
	private final BaseAmsAccountConnectionTask mIdpTask;
	private final BaseAmsAccountConnectionTask mUnAuthGetEngagementTask;
	private final BaseAmsAccountConnectionTask mRemoveOlderImagesTask;
	private final BaseAmsAccountConnectionTask mDataBaseTask;
	private final BaseAmsAccountConnectionTask mOpenSocketTask;
	private final BaseAmsAccountConnectionTask mCollectSDKConfigurationDataTask;
	private final BaseAmsAccountConnectionTask mCloseConnectionTask;

	private final MessagingStateMachineInterface mController;

	private List<BaseAmsAccountConnectionTask> mPrimaryFullConnectingTasks;
	private List<BaseAmsAccountConnectionTask> mPrimaryShortConnectingTasks;
	private List<BaseAmsAccountConnectionTask> mSecondaryConnectingTasks;

	private List<BaseAmsAccountConnectionTask> mDisconnectingTasks;

	public ConnectionTasksHolder(IStateMachine<IState> stateMachine, MessagingStateMachineInterface messagingController, String brandId) {

		mController = messagingController;

		// Get the host version
		String hostVersion = Infra.instance.getHostVersion();

		// Connecting tasks
		mCsdsTask = new CsdsTask(Infra.instance.getApplicationContext(), mController.getAccountsController(), brandId);
		mCsdsTask.setCallback(new DefaultTaskCallbackAmsAccount(stateMachine, mCsdsTask.getClass().getSimpleName()));

		mLptagTask = new LptagTask(mController.getAccountsController());
		mLptagTask.setCallback(new DefaultTaskCallbackAmsAccount(stateMachine, mLptagTask.getClass().getSimpleName()));

		mConfigurationFetcherTask = new ConfigurationFetcherTask(mController.getAccountsController());
		mConfigurationFetcherTask.setCallback(new DefaultTaskCallbackAmsAccount(stateMachine, mConfigurationFetcherTask.getClass().getSimpleName()));

		mSiteSettingsFetcherTask = new SiteSettingsFetcherTask(mController.getAccountsController());
		mSiteSettingsFetcherTask.setCallback(new DefaultTaskCallbackAmsAccount(stateMachine, mSiteSettingsFetcherTask.getClass().getSimpleName()));

		mIdpTask = new IdpTask(mController.getAccountsController(), mController.getAmsUsers(), hostVersion);
		mIdpTask.setCallback(new DefaultTaskCallbackAmsAccount(stateMachine, mIdpTask.getClass().getSimpleName()));

		mUnAuthGetEngagementTask = new UnAuthGetEngagementTask(mController.getAccountsController());
		mUnAuthGetEngagementTask.setCallback(new DefaultTaskCallbackAmsAccount(stateMachine, mUnAuthGetEngagementTask.getClass().getSimpleName()));

		mRemoveOlderImagesTask = new RemoveOlderImagesTask(brandId);
		mRemoveOlderImagesTask.setCallback(new DefaultTaskCallbackAmsAccount(stateMachine, mRemoveOlderImagesTask.getClass().getSimpleName()));


		mCollectSDKConfigurationDataTask = new CollectSDKConfigurationDataTask();
		mCollectSDKConfigurationDataTask.setCallback(new DefaultTaskCallbackAmsAccount(stateMachine, mCollectSDKConfigurationDataTask.getClass().getSimpleName()));

		mDataBaseTask = new DataBaseTask(mController.getAmsUsers(), mController.getAmsConversations(), mController.getAmsDialogs());
		mDataBaseTask.setCallback(new DefaultTaskCallbackAmsAccount(stateMachine, mDataBaseTask.getClass().getSimpleName()));

		mOpenSocketTask = new OpenSocketTask(mController.getAccountsController(), mController.getConnectionController());
		mOpenSocketTask.setCallback(new DefaultTaskCallbackAmsAccount(stateMachine, mOpenSocketTask.getClass().getSimpleName()));

		// Disconnecting tasks
		mCloseConnectionTask = new CloseConnectionTask(mController.getAccountsController());
		mCloseConnectionTask.setCallback(new DefaultTaskCallbackAmsAccount(stateMachine, mCloseConnectionTask.getClass().getSimpleName()));


		mPrimaryFullConnectingTasks = constructPrimaryFullConnectionTasks();
		mPrimaryShortConnectingTasks = constructPrimaryShortConnectionTasks();
		mSecondaryConnectingTasks = constructSecondaryConnectionTasks();

		mDisconnectingTasks = constructDisconnectingTasks();
	}

	// primary full happens on CSDS empty or CSDS data change or in connection failure
	public List<BaseAmsAccountConnectionTask> constructPrimaryFullConnectionTasks() {

		List<BaseAmsAccountConnectionTask> connectingTasks = new ArrayList<>(6);

		connectingTasks.add(mCsdsTask);
		connectingTasks.add(mLptagTask);
		connectingTasks.add(mConfigurationFetcherTask);
		connectingTasks.add(mSiteSettingsFetcherTask);
		connectingTasks.add(mDataBaseTask);
		connectingTasks.add(mUnAuthGetEngagementTask);
		connectingTasks.add(mIdpTask);
		connectingTasks.add(mRemoveOlderImagesTask);
		connectingTasks.add(mOpenSocketTask);
		//TODO need to update Loggos class before perform mCollectSDKConfigurationDataTask
//		connectingTasks.add(mCollectSDKConfigurationDataTask);

		return connectingTasks;
	}

	// primary short happens whenever primary long doesn't happen , this is different from the secondary cause those are tasks that we must do to open socket
	public List<BaseAmsAccountConnectionTask> constructPrimaryShortConnectionTasks() {

		List<BaseAmsAccountConnectionTask> connectingTasks = new ArrayList<>(3);

		connectingTasks.add(mDataBaseTask);
		connectingTasks.add(mUnAuthGetEngagementTask);
		connectingTasks.add(mIdpTask);
		connectingTasks.add(mOpenSocketTask);

		return connectingTasks;
	}

	// after primary short
	public List<BaseAmsAccountConnectionTask> constructSecondaryConnectionTasks() {

		List<BaseAmsAccountConnectionTask> connectingTasks = new ArrayList<>(6);

		connectingTasks.add(mConfigurationFetcherTask);
		connectingTasks.add(mSiteSettingsFetcherTask);
		connectingTasks.add(mCsdsTask);
		//TODO need to update Loggos class before perform mCollectSDKConfigurationDataTask
//		connectingTasks.add(mCollectSDKConfigurationDataTask);
		connectingTasks.add(mRemoveOlderImagesTask);
		connectingTasks.add(mLptagTask);

		return connectingTasks;
	}

	private List<BaseAmsAccountConnectionTask> constructDisconnectingTasks() {
		List<BaseAmsAccountConnectionTask> disconnectingTasks = new ArrayList<>();
		disconnectingTasks.add(mCloseConnectionTask);
		return disconnectingTasks;
	}

	public List<BaseAmsAccountConnectionTask> getPrimaryFullConnectionTasks() {
		return mPrimaryFullConnectingTasks;
	}

	public List<BaseAmsAccountConnectionTask> getPrimaryShortConnectionTasks() {
		return mPrimaryShortConnectingTasks;
	}

	public List<BaseAmsAccountConnectionTask> getSecondaryConnectionTasks() {
		return mSecondaryConnectingTasks;
	}

	public List<BaseAmsAccountConnectionTask> getDisconnectionTasks() {
		return mDisconnectingTasks;
	}
}
