package com.liveperson.messaging.model;

import android.content.ContentValues;
import android.database.Cursor;
import android.os.Bundle;
import android.text.TextUtils;

import com.liveperson.api.response.model.UserProfile;
import com.liveperson.infra.Clearable;
import com.liveperson.infra.controller.DBEncryptionHelper;
import com.liveperson.infra.database.BaseDBRepository;
import com.liveperson.infra.database.DataBaseCommand;
import com.liveperson.infra.database.DataBaseExecutor;
import com.liveperson.infra.database.tables.UsersTable;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.infra.utils.LocalBroadcast;

import java.util.HashMap;

/**
 * Created by Ilya Gazman on 11/10/2015.
 * <p/>
 * Users model
 */
public class AmsUsers extends BaseDBRepository implements Clearable {

    private static final String TAG = "AmsUsers";

    public static final String BROADCAST_CONSUMER_ID_ACTION = "BROADCAST_CONSUMER_ID_ACTION";
    public static final String BRAND_ID_EXTRA = "BRAND_ID_EXTRA";
    public static final String BROADCAST_AGENT_CHANGED = "BROADCAST_AGENT_CHANGED";
    private static final String EXTRA_KEY_AGENT_SERVER_ID = "EXTRA_KEY_AGENT_SERVER_ID";
    private static final String EXTRA_KEY_AGENT_FULL_NAME = "EXTRA_KEY_AGENT_FULL_NAME";
    private static final String EXTRA_KEY_AGENT_NICKNAME = "EXTRA_KEY_AGENT_NICKNAME";

    private static final String BROADCAST_CONSUMER_CHANGED = "BROADCAST_CONSUMER_CHANGED";
    private static final String EXTRA_KEY_ID = "EXTRA_KEY_ID";
    private static final String EXTRA_KEY_FULL_NAME = "EXTRA_KEY_FULL_NAME";


    //<brandID, consumerID>
    private HashMap<String, String> mBrandIDConsumerIdMap = new HashMap<>();

    public AmsUsers() {
        super(UsersTable.USERS_TABLE);
    }

    /**
     * @return current consumer id
     */
    public String getConsumerId(String brandID) {
        LPLog.INSTANCE.d(TAG, "getConsumerId for brand " + brandID);
        String consumerID = mBrandIDConsumerIdMap.get(brandID);
        return TextUtils.isEmpty(consumerID) ? null : consumerID;
    }

    /**
     * set the current consumer id
     *
     * @param consumerId
     */
    private void setConsumerId(String brandID, String consumerId) {
        String oldConsumerID = mBrandIDConsumerIdMap.get(brandID);
        LPLog.INSTANCE.d(TAG, "Adding consumer Id " + LPLog.INSTANCE.mask(consumerId) + " for brand " + brandID);

        if (!TextUtils.equals(oldConsumerID, consumerId)) {
            mBrandIDConsumerIdMap.put(brandID, consumerId);
            Bundle extra = new Bundle();
            extra.putString(BRAND_ID_EXTRA, brandID);
            LocalBroadcast.sendBroadcast(BROADCAST_CONSUMER_ID_ACTION, extra);
        }
    }


    /**
     * update the current consumer id - empty consumer id from IDP JWT
     *
     * @param consumerId
     */
    public void updateConsumerId(final String brandID, final String consumerId) {
        String oldConsumerID = mBrandIDConsumerIdMap.get(brandID);

        if (TextUtils.isEmpty(oldConsumerID)) {
            LPLog.INSTANCE.d(TAG, "No Consumer ID - Adding consumer Id " + LPLog.INSTANCE.mask(consumerId) + " for brand " + brandID);
            //only if we don't have consumer id - update in DB and in map.
            mBrandIDConsumerIdMap.put(brandID, consumerId);
            DataBaseExecutor.execute(() -> {
                MessagingUserProfile userProfile = new MessagingUserProfile("", "", UserProfile.UserType.CONSUMER);
                userProfile.setOriginatorID(consumerId);
                userProfile.setBrandID(brandID);
                userProfile.setUserEncryptionVersion(EncryptionVersion.VERSION_1);
                ContentValues messageValues = getContentValues(userProfile);
                getDB().replace(messageValues);
            });

        }
    }

    /**
     * Fetch last known consumer id
     *
     * @param mBrandId
     */
    public void loadConsumerForBrand(String mBrandId) {
        loadConsumer(mBrandId);
    }

    /**
     * Update consumer
     *
     * @param userProfile a bundle of properties to update
     */
    public void updateUserProfile(final MessagingUserProfile userProfile) {
        DataBaseExecutor.execute(() -> {

	        LPLog.INSTANCE.d(TAG, "updateUserProfile type:" + userProfile.getUserType() + ", id = " + userProfile.getOriginatorId());
            userProfile.setUserEncryptionVersion(EncryptionVersion.VERSION_1);

            ContentValues messageValues = getContentValues(userProfile);
            getDB().replace(messageValues);
            if (userProfile.getUserType() == UserProfile.UserType.CONSUMER) {
                loadConsumer(userProfile.getBrandId());
                Bundle extras = new Bundle();
                extras.putString(EXTRA_KEY_ID, userProfile.getOriginatorId());
                extras.putString(EXTRA_KEY_FULL_NAME, userProfile.getFullName());
                LocalBroadcast.sendBroadcast(BROADCAST_CONSUMER_CHANGED, extras);
            } else {
                Bundle extras = new Bundle();
                extras.putString(EXTRA_KEY_AGENT_SERVER_ID, userProfile.getOriginatorId());
                extras.putString(EXTRA_KEY_AGENT_FULL_NAME, userProfile.getFullName());
                extras.putString(EXTRA_KEY_AGENT_NICKNAME, userProfile.getNickname());
                LocalBroadcast.sendBroadcast(BROADCAST_AGENT_CHANGED, extras);
            }
        });
    }

    public void loadProfile(String assignedAgentServerId) {
        if (TextUtils.isEmpty(assignedAgentServerId)) return;

        getUserById(assignedAgentServerId).setPostQueryOnBackground(profile -> {
            if (profile != null) {
                if (profile.getUserType() == UserProfile.UserType.AGENT) {
                    Bundle extras = new Bundle();
                    extras.putString(EXTRA_KEY_AGENT_SERVER_ID, profile.getOriginatorId());
                    LocalBroadcast.sendBroadcast(BROADCAST_AGENT_CHANGED, extras);
                } else {
                    Bundle extras = new Bundle();
                    extras.putString(EXTRA_KEY_ID, profile.getOriginatorId());
                    extras.putString(EXTRA_KEY_FULL_NAME, profile.getFullName());
                    LocalBroadcast.sendBroadcast(BROADCAST_CONSUMER_CHANGED, extras);
                }
            }
        }).execute();
    }

    /**
     * Fetch consumer form db
     */
    private void loadConsumer(final String mBrandId) {
        DataBaseExecutor.execute(() -> {

            String consumerId = getConsumerIdFromDB(mBrandId);
            setConsumerId(mBrandId, consumerId);

        });
    }

	/**
	 * Query DB for the given brandId's consumer and return the consumer ID
	 * @param brandId
	 * @return
	 */
	private String getConsumerIdFromDB(final String brandId){
		Cursor cursor = getDB().query(new String[]{UsersTable.KEY_ORIGINATOR_ID}, UsersTable.KEY_USER_TYPE + " =? AND " + UsersTable.KEY_BRAND_ID + " =? ",
				new String[]{String.valueOf(UserProfile.UserType.CONSUMER.ordinal()), brandId}, null, null, null);
		if (cursor != null) {
			try {
				String consumerId = "";
				if (cursor.moveToFirst()) {
					int columnIndex = cursor.getColumnIndex(UsersTable.KEY_ORIGINATOR_ID);
					consumerId = cursor.getString(columnIndex);
				}
				return consumerId;

			} finally {
				cursor.close();
			}
		}
		return null;
	}

	/**
	 * Get the ID of a consumer for the given brandId
	 * @param brandId
	 * @return
	 */
	public DataBaseCommand<String> getConsumerByBrandIDFromDB(final String brandId) {

		return new DataBaseCommand<>(() -> getConsumerIdFromDB(brandId));
    }

    public MessagingUserProfile getProfileFromCursor(Cursor cursor) {

        String firstName = cursor.getString(cursor.getColumnIndex(UsersTable.KEY_FIRST_NAME));
        String lastName = cursor.getString(cursor.getColumnIndex(UsersTable.KEY_LAST_NAME));
        String nickName = cursor.getString(cursor.getColumnIndex(UsersTable.KEY_NICKNAME));
        String profileImage = cursor.getString(cursor.getColumnIndex(UsersTable.KEY_PROFILE_IMAGE));
        String description = cursor.getString(cursor.getColumnIndex(UsersTable.KEY_DESCRIPTION));
        String email = cursor.getString(cursor.getColumnIndex(UsersTable.KEY_EMAIL));
        String mobileNumber = cursor.getString(cursor.getColumnIndex(UsersTable.KEY_PHONE_NUMBER));

        final int userEncryptionVersion = cursor.getInt(cursor.getColumnIndex(UsersTable.KEY_ENCRYPTION_VERSION));
        final EncryptionVersion currentEncryptionVersion = EncryptionVersion.fromInt(userEncryptionVersion);

        firstName = DBEncryptionHelper.decrypt(currentEncryptionVersion, firstName);
        lastName = DBEncryptionHelper.decrypt(currentEncryptionVersion, lastName);
        nickName = DBEncryptionHelper.decrypt(currentEncryptionVersion, nickName);
        profileImage = DBEncryptionHelper.decrypt(currentEncryptionVersion, profileImage);
        description = DBEncryptionHelper.decrypt(currentEncryptionVersion, description);
        email = DBEncryptionHelper.decrypt(currentEncryptionVersion, email);
        mobileNumber = DBEncryptionHelper.decrypt(currentEncryptionVersion, mobileNumber);

        MessagingUserProfile profile = new MessagingUserProfile(firstName, lastName,
                UserProfile.UserType.values()[cursor.getInt(cursor.getColumnIndex(UsersTable.KEY_USER_TYPE))]);

        profile.setNickname(nickName);
        profile.setLocalId(cursor.getLong(cursor.getColumnIndex(UsersTable.KEY_ID)));
        profile.setOriginatorID(cursor.getString(cursor.getColumnIndex(UsersTable.KEY_ORIGINATOR_ID)));
        profile.setAvatarUrl(profileImage);
        profile.setDescription(description);
        profile.setRequestId(cursor.getInt(cursor.getColumnIndex(UsersTable.KEY_REQUEST_ID)));

        //private Data
        profile.setEmail(email);
        profile.setMobileNumber(mobileNumber);

        return profile;
    }


    /**
     * Get user with type AGENT and specific originator id if exists, or null otherwise
     *
     * @param originatorId - the originatorID of the agent
     * @return
     */
    public DataBaseCommand<MessagingUserProfile> getUserById(final String originatorId) {
        return new DataBaseCommand<>(() -> {
            Cursor cursor = getDB().query(null, UsersTable.KEY_ORIGINATOR_ID + "=?",
                    new String[]{String.valueOf(originatorId)}, null, null, null);
            if (cursor != null) {
                try {
                    if (cursor.moveToFirst()) {
                        return getProfileFromCursor(cursor);
                    }
                } finally {
                    cursor.close();
                }
            }
            return null;
        });
    }


    /**
     * Basic set user profile to DB
     *
     * @param profile
     * @return
     */
    private ContentValues getContentValues(MessagingUserProfile profile) {
        ContentValues contentValues = new ContentValues();

        final EncryptionVersion userEncryptionVersion = profile.getUserEncryptionVersion();
        contentValues.put(UsersTable.KEY_ENCRYPTION_VERSION, userEncryptionVersion.ordinal());

        String encryptedFirstName = DBEncryptionHelper.encrypt(userEncryptionVersion, profile.getFirstName());
        String encryptedLastName = DBEncryptionHelper.encrypt(userEncryptionVersion, profile.getLastName());
        String encryptedNickName = DBEncryptionHelper.encrypt(userEncryptionVersion, profile.getNickname());
        String encryptedAvatarUrl = DBEncryptionHelper.encrypt(userEncryptionVersion, profile.getAvatarUrl());
        String encryptedDescription = DBEncryptionHelper.encrypt(userEncryptionVersion, profile.getDescription());

        contentValues.put(UsersTable.KEY_FIRST_NAME, encryptedFirstName);
        contentValues.put(UsersTable.KEY_LAST_NAME, encryptedLastName);
        contentValues.put(UsersTable.KEY_NICKNAME, encryptedNickName);
        contentValues.put(UsersTable.KEY_PROFILE_IMAGE, encryptedAvatarUrl);
        contentValues.put(UsersTable.KEY_DESCRIPTION, encryptedDescription);

        if (profile.getPrivateData() != null) {
            String encryptedEmail = DBEncryptionHelper.encrypt(userEncryptionVersion, profile.getPrivateData().mail);
            String encryptedMobileNumber = DBEncryptionHelper.encrypt(userEncryptionVersion, profile.getPrivateData().mobileNum);

            contentValues.put(UsersTable.KEY_EMAIL, encryptedEmail);
            contentValues.put(UsersTable.KEY_PHONE_NUMBER, encryptedMobileNumber);
        }

        contentValues.put(UsersTable.KEY_REQUEST_ID, profile.getRequestId());
        contentValues.put(UsersTable.KEY_ORIGINATOR_ID, profile.getOriginatorId());
        contentValues.put(UsersTable.KEY_BRAND_ID, profile.getBrandId());

        if (profile.getUserType() != null) {
            contentValues.put(UsersTable.KEY_USER_TYPE, profile.getUserType().ordinal());
        }

        return contentValues;
    }

    /**
     * Removing the current user
     * @param mBrandId
     */
    public void clearConsumerFromDB(final String mBrandId){
        final String consumerId = getConsumerIdFromDB(mBrandId);
        clear();

        DataBaseExecutor.execute(() -> {
            int rowIdDeleted = getDB().removeAll(UsersTable.KEY_ORIGINATOR_ID + "=?", new String[]{String.valueOf(consumerId)});
            if (rowIdDeleted == -1) {
	            LPLog.INSTANCE.d(TAG, "Could not find old consumer user");
            } else {
	            LPLog.INSTANCE.d(TAG, "Deleted old consumer user");
            }
        });
    }

    @Override
    public void clear() {
        mBrandIDConsumerIdMap.clear();
    }
}
