package com.liveperson.messaging.model;

import android.database.Cursor;

import com.liveperson.api.response.model.DialogData;
import com.liveperson.api.response.model.MultiDialog;
import com.liveperson.api.response.model.Participants;
import com.liveperson.api.response.types.CSAT;
import com.liveperson.api.response.types.CloseReason;
import com.liveperson.api.response.types.DialogState;
import com.liveperson.api.response.types.DialogType;
import com.liveperson.api.response.types.TTRType;
import com.liveperson.infra.database.tables.DialogsTable;
import com.liveperson.infra.log.FlowTags;
import com.liveperson.infra.log.LPLog;

import static com.liveperson.infra.errors.ErrorCode.ERR_00000063;

/**
 * Created by oferd on 8/10/15.
 */
public class Dialog {
    private static final String TAG = "Dialog";

    public static final String TEMP_DIALOG_ID = "TEMP_DIALOG";

    private String mConversationId;
    private String mDialogId;

    private DialogType mDialogType;
    private MultiDialog.ChannelType mChannelType;

    private Participants mParticipants = new Participants();
    private String mTargetId;
    private String mBrandId;
    private DialogState mState;
    private TTRType mConversationTTRType = TTRType.NORMAL;
    private String mAssignedAgentId;
    private long mRequestId;
    private int mLastServerSequence = -1;
    private long mStartTimestamp;
    private long mEndTimestamp = -1;
    private int mUnreadMessages;
    private CSAT.CSAT_SHOW_STATUS mShowedCSAT = CSAT.CSAT_SHOW_STATUS.NO_VALUE;
    private int mUpdateInProgress = 0;
    private CloseReason mCloseReason = null;
    private TTRManager mTTRManager;
    private PendingDialogData mPendingDialogData = new PendingDialogData();

    public Dialog(String targetId, String brandId) {
        mTargetId = targetId;
        mBrandId = brandId;
        mTTRManager = new TTRManager(mTargetId);
        mChannelType = MultiDialog.ChannelType.MESSAGING;
    }

    // TODO Perry, this is a fallback in case the old architecture is in use. Delete this when the new UMS is available!
    @Deprecated
    public Dialog(ConversationData data) {
        this(data.targetId, data.brandId);

        LPLog.INSTANCE.e(TAG, ERR_00000063, "The new UMS is not with us, we're communicating with a legacy UMS");

        mConversationId = data.conversationId;
        mDialogId = DialogData.extractDialogId(data);
        mDialogType = DialogType.MAIN;
        mChannelType = MultiDialog.ChannelType.MESSAGING;
        mRequestId = data.requestId;
        mState = DialogState.parse(data.state);
        mConversationTTRType = data.conversationTTRType;
        mAssignedAgentId = data.getAssignedAgentId();
        mStartTimestamp = data.startTs;
        mUnreadMessages = data.unreadMessages;
        mCloseReason = data.closeReason;
        mEndTimestamp = data.endTs;
    }

    public Dialog(Cursor cursor) {
        this(cursor.getString(cursor.getColumnIndex(DialogsTable.Key.TARGET_ID)), cursor.getString(cursor.getColumnIndex(DialogsTable.Key.BRAND_ID)));
        mConversationId = cursor.getString(cursor.getColumnIndex(DialogsTable.Key.CONVERSATION_ID));
        mDialogId = cursor.getString(cursor.getColumnIndex(DialogsTable.Key.DIALOG_ID));
        mDialogType = DialogType.parse(cursor.getString(cursor.getColumnIndex(DialogsTable.Key.DIALOG_TYPE)));
        mChannelType = MultiDialog.ChannelType.parse(cursor.getString(cursor.getColumnIndex(DialogsTable.Key.CHANNEL_TYPE)));
        int active = cursor.getInt(cursor.getColumnIndex(DialogsTable.Key.STATE));
        mState = DialogState.values()[active];
        mRequestId = cursor.getLong(cursor.getColumnIndex(DialogsTable.Key.REQUEST_ID));
        mLastServerSequence = cursor.getInt(cursor.getColumnIndex(DialogsTable.Key.LAST_SERVER_SEQUENCE));
        mAssignedAgentId = cursor.getString(cursor.getColumnIndex(DialogsTable.Key.ASSIGNED_AGENT_ID));
        int ttrType = cursor.getInt(cursor.getColumnIndex(DialogsTable.Key.TTR_TYPE));
        if (ttrType != -1){
            mConversationTTRType = TTRType.values()[ttrType];
        }
        int csatShownStatus = cursor.getInt(cursor.getColumnIndex(DialogsTable.Key.CSAT_STATUS));
        mShowedCSAT = CSAT.CSAT_SHOW_STATUS.parse(csatShownStatus);
        mEndTimestamp = cursor.getLong(cursor.getColumnIndex(DialogsTable.Key.END_TIMESTAMP));
        int closeReasonInt = cursor.getInt(cursor.getColumnIndex(DialogsTable.Key.CLOSE_REASON));
        if (closeReasonInt != -1){
            mCloseReason = CloseReason.values()[closeReasonInt];
        }
        mUnreadMessages = cursor.getInt(cursor.getColumnIndex(DialogsTable.Key.UNREAD_MESSAGES_COUNT));
        mStartTimestamp = cursor.getLong(cursor.getColumnIndex(DialogsTable.Key.START_TIMESTAMP));
        mUpdateInProgress = cursor.getInt(cursor.getColumnIndex(DialogsTable.Key.CONCURRENT_REQUESTS_COUNTER));
    }

    public Dialog(DialogData dialogData, ConversationData conversationData) {
        this(conversationData.targetId, conversationData.brandId);

        if (dialogData.isOpen) {
            mState = DialogState.OPEN;
        } else {
            mState = DialogState.CLOSE;
        }

        mParticipants = dialogData.participantsDetails;
        mConversationId = dialogData.conversationId;
        mDialogId = dialogData.dialogId;
        mDialogType = dialogData.dialogType;
        mChannelType = dialogData.channelType;
        mCloseReason = dialogData.closeReason;
        mState = dialogData.state;
        mRequestId = conversationData.requestId;
        mConversationTTRType = conversationData.conversationTTRType;
        mAssignedAgentId = conversationData.getAssignedAgentId();
        mStartTimestamp = dialogData.creationTs;
        mEndTimestamp = dialogData.endTs;
        mUnreadMessages = conversationData.unreadMessages;
    }

    public Participants getParticipants() {
        return mParticipants;
    }

    public String getConversationId() {
        return mConversationId;
    }

    public String getTargetId() {
        return mTargetId;
    }

    public String getBrandId() {
        return mBrandId ;
    }

    protected void setBrandId(String brandId) {
        mBrandId = brandId;
    }

    public DialogState getState() {
        return mState;
    }

    public void setState(DialogState state) {
        if (mState != state) {
            // Changing dialog's state
            LPLog.INSTANCE.d(TAG, FlowTags.DIALOGS, "Changing state from '"+ mState +"' to '" + state + "' of dialog: " + mDialogId);
        }
        mState = state;
    }

    public long getRequestId() {
        return mRequestId;
    }

    public void setRequestId(long requestId) {
        this.mRequestId = requestId;
    }

    public String getAssignedAgentId() {
        return mAssignedAgentId;
    }

    public void setConversationId(String conversationId) {
        this.mConversationId = conversationId;
    }

    public void setDialogId(String dialogId) {
        this.mDialogId = dialogId;
    }

    public void setAssignedAgentId(String assignedAgentServerId) {
        mAssignedAgentId = assignedAgentServerId;
    }

    public int getLastServerSequence() {
        return mLastServerSequence;
    }

    public void setLastServerSequence(int lastServerSequence) {
        if (lastServerSequence > mLastServerSequence) {
            this.mLastServerSequence = lastServerSequence;
        }
    }

    public TTRType getConversationTTRType() {
        return mConversationTTRType;
    }

    public void setConversationTTRType(TTRType conversationTTRType) {
        LPLog.INSTANCE.d(TAG, "Setting conversation ttr type: " + conversationTTRType);
        mConversationTTRType = conversationTTRType;
    }

    public TTRManager getTTRManager() {
        return mTTRManager;
    }

    public CSAT.CSAT_SHOW_STATUS isShowedCSAT() {
        LPLog.INSTANCE.d(TAG, "isShowedCSAT:" + mShowedCSAT);
        return mShowedCSAT;
    }

    public void setShowedCSAT(CSAT.CSAT_SHOW_STATUS showedCSAT) {
        LPLog.INSTANCE.d(TAG, "setShowedCSAT:" + showedCSAT);
        mShowedCSAT = showedCSAT;
    }

    public CloseReason getCloseReason() {
        return mCloseReason;
    }

    public void setCloseReason(CloseReason mCloseReason) {
        this.mCloseReason = mCloseReason;
    }

    public long getEndTimestamp() {
        return mEndTimestamp;
    }

    public void setEndTimestamp(long endTimestamp) {
        this.mEndTimestamp = endTimestamp;
    }

    public long getStartTimestamp() {
        return mStartTimestamp;
    }

    public int getUnreadMessages() {
        return mUnreadMessages;
    }

    public void setUnreadMessages(int unreadMessages) {
        mUnreadMessages = unreadMessages;
    }

    public void setStartTimestamp(long startTimestamp) {
        mStartTimestamp = startTimestamp;
    }

    public int getUpdateInProgress() {
        return mUpdateInProgress;
    }

    public void setUpdateInProgress(int mUpdateInProgress) {
        this.mUpdateInProgress = mUpdateInProgress;
    }

    public PendingDialogData getPendingData() {
        return mPendingDialogData;
    }

    public void setDialogType(DialogType dialogType) {
        this.mDialogType = dialogType;
    }

    public MultiDialog.ChannelType getChannelType() {
        return mChannelType;
    }

    public void setChannelType(MultiDialog.ChannelType channelType) {
        this.mChannelType = channelType;
    }

    public DialogType getDialogType() {
        return mDialogType;
    }

    public String getDialogId() {
        return mDialogId;
    }

    public boolean isOpen() {
        return mState == DialogState.OPEN;
    }

    public boolean isClosed() {
        return mState == DialogState.CLOSE;
    }

    @Override
    public String toString() {
        return "Dialog: {conversationId: " + mConversationId + ", dialogId: " + mDialogId + ", state: " + mState + ", type: " + mDialogType + "}";
    }

    @Override
    public boolean equals(Object obj) {
        return (obj instanceof Dialog && ((Dialog) obj).getDialogId().equals(getDialogId())) || super.equals(obj);
    }

    public boolean isOpenOrPending() {
        return mState == DialogState.OPEN || mState == DialogState.PENDING;
    }
}
