package com.liveperson.messaging.network.http;

import android.net.Uri;
import android.text.TextUtils;
import android.text.format.DateUtils;

import com.liveperson.api.response.model.ConversationINCADetails;
import com.liveperson.infra.Command;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.network.http.HttpHandler;
import com.liveperson.infra.network.http.request.HttpGetRequest;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.model.IncaGetConversationsListResponse;

import java.util.ArrayList;

import static com.liveperson.infra.errors.ErrorCode.ERR_000000C4;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000C5;

/**
 * Created by ofira on 3/17/16.
 */
public class IncaGetConversationsListRequest implements Command {


    //https://<domain>/messaging_history/api/account/{accountID}/conversations/consumer/metadata/
    // search?state=close&startFrom=1502628503961&startTo=1502628758614&offset=0&limit=50
    private static final String TAG = "IncaGetConversationsListRequest";

    private static final String GET_CONVERSATION_LIST_API_URL = "https://%s/messaging_history/api/account/%s/conversations/consumer/metadata/%s";
    private static final String SEARCH_QUERY = "search";
    private static final int REQUEST_TIMEOUT = 30000;
    private static final int MAX_LIMIT = 100;
    private static final long MAX_TIME_LIMIT = (DateUtils.YEAR_IN_MILLIS / 12) * 13;

    protected final Messaging mController;
    private final long startFrom;
    private String mRequestUrl;
    private long startTo;

    private String mBrand;

    private ICallback<ArrayList<ConversationINCADetails>, Exception> callback;
    private ArrayList<ConversationINCADetails> fullConversationList = new ArrayList<>();

    public IncaGetConversationsListRequest(Messaging controller, String brand, long from, long to, long offset, ICallback<ArrayList<ConversationINCADetails>, Exception> callback) {
        mBrand = brand;
        mController = controller;
        this.callback = callback;

        long limitTime = System.currentTimeMillis() - MAX_TIME_LIMIT;
        startFrom = (from < limitTime ? limitTime : from);
        this.startTo = to;
        if (this.startTo < limitTime){
            //mean we are checking data more than 13 months ago.. no need to.
            this.startTo = limitTime;
            LPLog.INSTANCE.w(TAG, "Got startTo that is older than 13 months ago. aborting..." );
            callback.onSuccess(new ArrayList<ConversationINCADetails>(){});
            return;
        }

        if (startTo == startFrom) {
            //abort call
            LPLog.INSTANCE.w(TAG, "start to == start from , no need to bring data"  );
            callback.onSuccess(new ArrayList<ConversationINCADetails>(){});
            return;
        }

        String incaDomain = mController.mAccountsController.getServiceUrl(mBrand, ConnectionParamsCache.CSDS_INCA_KEY);

        if (!TextUtils.isEmpty(incaDomain)) {
            String historyUrl = String.format(GET_CONVERSATION_LIST_API_URL, incaDomain, mBrand, SEARCH_QUERY);

            Uri fullUri = Uri.parse(historyUrl).buildUpon()
                    .appendQueryParameter("state", "close")
                    .appendQueryParameter("startFrom", String.valueOf(startFrom))
                    .appendQueryParameter("startTo", String.valueOf(startTo))
                    .appendQueryParameter("offset", String.valueOf(offset))
                    .appendQueryParameter("limit", String.valueOf(MAX_LIMIT))
                    .appendQueryParameter("sort", "start:desc")
                    .build();

            mRequestUrl = fullUri.toString();
        }
    }


    @Override
    public void execute() {

        if (TextUtils.isEmpty(mRequestUrl)){
            LPLog.INSTANCE.d(TAG, "mRequestUrl is empty");
            return;
        }

        if (mController.mAccountsController.getAccount(mBrand) == null) {
            LPLog.INSTANCE.d(TAG, "mAccountsController is empty");
            return;
        }

        mController.setStillBusyFetching(true);

        LPLog.INSTANCE.d(TAG, "Getting inca conversation list url " + mRequestUrl  );
        HttpGetRequest httpGetRequest = new HttpGetRequest(mRequestUrl);

        //["Content-Type": "application/json", "Authorization": "Bearer eyJraWQiOiIwMDAwMSIsInR5cCI6IkpXVCIsImFsZyI6IlJTMjU2In0.eyJzdWIiOiJhNDRiOGZiNWU1NjVkZWE5MTZlMWYzYmQ1MGNkMGU1ZmMzY2Q5OTJmMzhjZjc3ZGEwYTM5NjZkMzk5YTVkMTFiIiwiYXVkIjoiYWNjOjM2MzY2Nzk2IiwiaXNzIjoiaHR0cHM6XC9cL2lkcC5saXZlcGVyc29uLm5ldCIsImxwLmV4dCI6eyJzdWIiOiJscFRlc3R3b3ciLCJpc3MiOiJodHRwczpcL1wvY3VzdG9tZXJXZWJTaXRlLmNvbSIsImxwX3NkZXMiOiJlMmY3OTU4YmI2NTdkYWM4YTJjY2IwYTU4Y2I0ZGRiNDVkZmJiZmYzODQzMzVkYmRiYzAwY2Y1YzZiYWVmY2UyNTMyYjViY2M1MTU1N2IxZThlNWUxNzExNTg0NmY4Zjk3YWFhNmM0ZTM0ZDQwNjE4ZjJmMTgzZTg5M2EzZTFlN2YwYmU5N2VhNjA3MzI1YWE2NzY0NzZmMmYzNDhjYjVkZWNmZTIyYzMzYzQ0Y2JjZDFhYTBkMTU4ZTViZTFkYjIwNDU3MDQ5NWE5ZmIwNjQ0NmU0Zjk1MThmZDBiYTg4MCJ9LCJleHAiOjE1MTg2ODI4NTgsImlhdCI6MTUxNjA5MDg1OH0.OacsjIdHThLrsn6df1OdlSGqeSCDEtVOwFQOXHs-aqfDuSOQfjhPK95h33rKKDZaxBSqXJi23arCNaoVNRgF9PZna4oIrjqtRRqPPzTVm0sg0JNJKLtSRhgFGPEJOa2BnYaZlRROn5xLtglqAFWt5rTNZ9G4D4r3ew94ZXuhZ8ILEjzAH3_bG4rY1pRrkdhM9o_5m045Oslb4Tj11viY6l3U07O9qfPa3DUdyax4-ghCnZy5fmb0AFumPOubFRTRYf9WxsVeIA-llwXxpHX22et-o-riAtT5ENBFS72LzHwcExU4gf7ipp3BR4S1Lm6J3NCFrmscmqAb3PhLIPvKUw"]
        httpGetRequest.addHeader("Authorization", "Bearer "+ mController.mAccountsController.getAccount(mBrand).getToken());
        httpGetRequest.setCertificatePinningKeys(mController.mAccountsController.getCertificatePinningKeys(mBrand));
        httpGetRequest.setTimeout(REQUEST_TIMEOUT);
        httpGetRequest.setCallback( new ICallback<String, Exception>() {
            @Override
            public void onSuccess(String response) {
                try {
                    LPLog.INSTANCE.d(TAG, "onSuccess with INCA history response details " + response);
                    if (TextUtils.isEmpty(response)) {
                        mController.setStillBusyFetching(false);
                        callback.onError(new Exception("Empty response"));
                        return;
                    }
                    IncaGetConversationsListResponse historyResponse = new IncaGetConversationsListResponse(response);
                    if (historyResponse.getListOfConversations() == null) {
                        mController.setStillBusyFetching(false);
                        callback.onError(new Exception("Empty conversation list from INCA"));
                        return;
                    }
                    LPLog.INSTANCE.d(TAG, "onSuccess with INCA history: got " + historyResponse.getListOfConversations().size() +
                            " num of total conversations: " + historyResponse.getNumOfResults());
                    fullConversationList.addAll(historyResponse.getListOfConversations());

                    if (!TextUtils.isEmpty(historyResponse.mNextUrl)) {
                        //Recursion - could be there are more results, mNextUrl contains the next set of results
                        mRequestUrl = historyResponse.mNextUrl;
                        LPLog.INSTANCE.d(TAG, "More results available, sending next request...");
                        execute();
                    } else {
                        //finished collecting all the conversations, we can now return the response
                        mController.setStillBusyFetching(false);
                        callback.onSuccess(fullConversationList);
                    }
                } catch (Exception e) {
                    mController.setStillBusyFetching(false);
                    LPLog.INSTANCE.e(TAG, ERR_000000C4, "Error parsing inca conversation list.", e);
                    callback.onError(e);
                }
            }

            @Override
            public void onError(Exception exception) {
                LPLog.INSTANCE.e(TAG, ERR_000000C5, "Exception running inca conversation list.", exception);
                mController.setStillBusyFetching(false);
                if (fullConversationList.isEmpty()){
                    callback.onError(exception);
                }else{
                    callback.onSuccess(fullConversationList);
                }
            }
        });

        HttpHandler.execute(httpGetRequest);
    }
}


/*

{
    "_responseMetadata": {
        "count": 1,
        "self": {
            "rel": "self",
            "href": "http://localhost:8080/messaging_history/api/account/le69322492/conversations/consumer/metadata/search?limit=50&offset=0&sort=start:desc"
        },
        "shardsStatusResult": {
            "partialResult": false
        }
    },
    "conversationHistoryMetadataRecords": [
        {
            "convId": "2ba1f774-0455-4759-9e5c-42d9b67db3f1",
            "participants": [
                {
                    "id": "d74812871d96945af08f77b95b16b39cc2f29438bbe48b0109e2719575787332",
                    "role": "CONSUMER"
                },
                {
                    "id": "eea399b9-fed0-5cb0-8d38-00e3bc50d79b",
                    "role": "ASSIGNED_AGENT"
                }
            ],
            "state": "CLOSE",
            "startTs": 1501047777465,
            "endTs": 1501047800467,
            "csat": {
                "csatRate": 3,
                "csatResolutionConfirmation": false,
                "status": "FILLED"
            }
        }
    ]
}

 */
