package com.liveperson.messaging.background.filesharing.document;

import android.content.ContentResolver;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.net.Uri;
import android.webkit.MimeTypeMap;

import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.messaging.R;
import com.liveperson.infra.utils.DocumentUtils;
import com.liveperson.infra.utils.ImageUtils;
import com.liveperson.messaging.MessagingFactory;
import com.liveperson.messaging.background.filesharing.BaseUploadTask;
import com.liveperson.messaging.background.filesharing.UploadFileTaskBundle;
import com.liveperson.messaging.commands.ReSendDocumentCommand;
import com.liveperson.messaging.commands.SendDocumentCommand;
import com.liveperson.messaging.exception.FileSharingException;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

import static com.liveperson.infra.errors.ErrorCode.ERR_000000B6;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000B7;
import static com.liveperson.infra.utils.Utils.getResources;

public class UploadDocumentTask extends BaseUploadTask {

    private static final String TAG = "UploadDocumentTask";
    private static final long MAX_FILE_SIZE = 5000000L; // 5MB

    private UploadDocumentTaskBundle uploadDocumentTaskBundle;
    private String previewBase64;
    protected String previewContentType;
    private Context mContext;

    public UploadDocumentTask(Context context, UploadDocumentTaskBundle params, Integer uploadTimeout, boolean isReUploadRequest) throws FileSharingException {
        super(uploadTimeout);

        if (params == null) {
            throw new FileSharingException("Params is null");
        }
        mContext = context;
        uploadDocumentTaskBundle = params;
        generateFileByteArray(uploadDocumentTaskBundle.getFileUri());
        String filePath = null;
        try {
            filePath = DocumentUtils.saveOriginalDocument(uploadDocumentTaskBundle.getFileUri(), mContext, uploadDocumentTaskBundle.getBrandId(), mOriginalFileByteArray);
        } catch (Exception e) {
            LPLog.INSTANCE.e(TAG, ERR_000000B6, "Failed to copy file into app's local directory.", e);
        }
        if (filePath == null) {
            filePath = params.getFilePath();
        }
        generatePreviewImage();

        if (isReUploadRequest) {
            createReSendMessageCommand(filePath);
        } else {
            createSendMessageCommand(filePath);
        }
    }

    /**
     * Get byte array of the document file
     * @param uri File uri from which we generate byte array
     * @return
     */
    private void generateFileByteArray(Uri uri) throws FileSharingException {
        try {
            int fileSize = DocumentUtils.getDocumentSize(uri, mContext);
            if (fileSize > MAX_FILE_SIZE) {
                throw new FileSharingException("Unsupported file size");
            }
            byte[] bytes = new byte[fileSize];
            InputStream inputStream;
            String fileScheme = uri.getScheme();
            if (fileScheme != null && fileScheme.equals(ContentResolver.SCHEME_CONTENT)) {
                inputStream = mContext.getContentResolver().openInputStream(uri);
            } else {
                File file = new File(uri.toString());
                inputStream = new FileInputStream(file);
            }

            BufferedInputStream buf = new BufferedInputStream(inputStream);
            //noinspection ResultOfMethodCallIgnored
            buf.read(bytes, 0, bytes.length);
            buf.close();
            mOriginalFileByteArray = bytes;
        } catch (IOException e) {
            LPLog.INSTANCE.e(TAG, ERR_000000B7, "Failed to generate document's byte array.", e);
        }
    }

    private void generatePreviewImage() {

        Bitmap bitmap = BitmapFactory.decodeResource(getResources(), R.drawable.lp_messaging_dummy_file_thumbnal);
        previewBase64 = ImageUtils.bitmapToBase64(bitmap);

        previewContentType = MimeTypeMap.getSingleton().getMimeTypeFromExtension("png");
    }

    private void createSendMessageCommand(String filePath) {
        sendMessageCommand = new SendDocumentCommand(MessagingFactory.getInstance().getController(),
                uploadDocumentTaskBundle.getTargetId(),
                uploadDocumentTaskBundle.getBrandId(),
                uploadDocumentTaskBundle.getFileContentType(),
                null,
                filePath,
                uploadDocumentTaskBundle.getFileTypeExtension(),
                uploadDocumentTaskBundle.getMessage(),
                previewContentType);

        setSendMessageCommandCallback();
    }

    private void createReSendMessageCommand(String filePath) {
        sendMessageCommand = new ReSendDocumentCommand(MessagingFactory.getInstance().getController(),
                uploadDocumentTaskBundle.getTargetId(),
                uploadDocumentTaskBundle.getBrandId(),
                uploadDocumentTaskBundle.getFileContentType(),
                null,
                filePath,
                uploadDocumentTaskBundle.getFileTypeExtension(),
                uploadDocumentTaskBundle.getMessage(),
                previewContentType,
                uploadDocumentTaskBundle.getEventId(),
                uploadDocumentTaskBundle.getFileRowId());

        setSendMessageCommandCallback();
    }

    @Override
    protected UploadFileTaskBundle getUploadTaskBundle() {
        return uploadDocumentTaskBundle;
    }

    @Override
    protected byte[] getFileByteArray() {
        return mOriginalFileByteArray;
    }

    @Override
    public int getTaskId() {
        return uploadDocumentTaskBundle.getTaskId();
    }

    @Override
    public String getThumbnailBase64() {
        return previewBase64;
    }

}
