package com.liveperson.messaging.model;

import android.text.TextUtils;

import com.liveperson.api.response.model.UserProfile;
import com.liveperson.api.response.types.TTRType;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.messaging.R;
import com.liveperson.infra.model.LPWelcomeMessage;
import com.liveperson.infra.preferences.PushMessagePreferences;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.network.http.AgentProfileRequest;

/**
 * Created by shiranr on 28/01/2016.
 */
public class ConversationUtils {

    private static final String TAG = "ConversationUtils";
    protected final Messaging mController;

    public ConversationUtils(Messaging controller) {
        mController = controller;
    }

    public void addWelcomeMessage(final String brandId, final String message) {
        mController.amsMessages.isLastMessageWelcomeMessage().setPostQueryOnBackground(isLastMessageWelcomeMessage -> {
            long messageTimestamp = System.currentTimeMillis();
            MessagingChatMessage messagingChatMessage = buildWelcomeMessagingChatMessage(brandId, messageTimestamp, message,  AmsMessages.WELCOME_MSG_SEQUENCE_NUMBER);
            if (!isLastMessageWelcomeMessage) {
                mController.amsMessages.getLatestOutboundMessage().setPostQueryOnBackground(outboundMessage -> {
                    // Add welcome message only If last message is not outbound campaign message
                    if (TextUtils.isEmpty(outboundMessage)) {
                        LPLog.INSTANCE.d(TAG, "addWelcomeMessage: Adding welcome message");
                        mController.amsMessages.addMessage(messagingChatMessage, true).execute();
                    }
                }).execute();
            } else {
                LPLog.INSTANCE.d(TAG, "addWelcomeMessage: Loading existing welcome message");
                mController.amsMessages.loadExistingWelcomeMessage(messagingChatMessage).execute();
            }
        }).execute();
    }

    /**
     * If show welcome message for every new conversation, we need to update welcome every time enter conversation screen when the conversation is not started.
     * If show welcome message for only first conversation, we remove welcome message if there is already an welcome message and it's not first message or last message.
     */
    public void updateWelcomeMessage(String brandId, LPWelcomeMessage message) {
        mController.amsMessages.isLastMessageWelcomeMessage().setPostQueryOnBackground(
                isLastMessageWelcomeMessage -> {
                    if (isLastMessageWelcomeMessage) {
                        if (message.getMessageFrequency() == LPWelcomeMessage.MessageFrequency.FIRST_TIME_CONVERSATION) {
                            mController.amsMessages.removeLastWelcomeMessage(false).execute();
                        } else {
                            long messageTimestamp = System.currentTimeMillis();
                            MessagingChatMessage messagingChatMessage = buildWelcomeMessagingChatMessage(brandId,
                                    messageTimestamp, message.getWelcomeMessage(), AmsMessages.WELCOME_MSG_SEQUENCE_NUMBER);
                            mController.amsMessages.updateLastWelcomeMessage(messagingChatMessage).execute();
                        }
                    }
                }
        ).execute();
    }

    /**
     * Add or update outbound welcome message into the database If necessary.
     * @param brandId respective account id
     */
    public void updateOutboundCampaignMessage(String brandId) {
        if (PushMessagePreferences.INSTANCE.isPushNotificationClicked()) {
            PushMessagePreferences.INSTANCE.setPushMessageClicked(null, false);
            String id = PushMessagePreferences.INSTANCE.getClickedNotificationId();
            displayOutboundCampaignMessage(brandId, id);
        } else if (Configuration.getBoolean(R.bool.show_outbound_in_app_message)) {
            String id = PushMessagePreferences.INSTANCE.getLatestNotificationId();
            displayOutboundCampaignMessage(brandId, id);
        } else {
            // Clean up existing outbound campaign message if required
            LPLog.INSTANCE.d(TAG, "updateOutboundCampaignMessage: Removing existing outbound welcome message");
            mController.amsMessages.removeLastOutboundMessage().execute();
        }
    }

    /**
     * Add outbound welcome message into database and notify UI
     * @param brandId respective account id
     * @param messageId id of message to be fetched from cache
     */
    private void displayOutboundCampaignMessage(String brandId, String messageId) {
        String newOutboundMessage = PushMessagePreferences.INSTANCE.getCachedPushWelcomeMessage(messageId, brandId);
        if (!TextUtils.isEmpty(newOutboundMessage)) {
            mController.amsMessages.getLatestOutboundMessage().setPostQueryOnBackground(storedOutboundMessage -> {
                if (TextUtils.isEmpty(storedOutboundMessage)) {
                    long messageTimestamp = System.currentTimeMillis();
                    MessagingChatMessage messagingChatMessage = buildWelcomeMessagingChatMessage(brandId,
                            messageTimestamp, newOutboundMessage, AmsMessages.OUTBOUND_CAMPAIGN_MSG_SEQUENCE_NUMBER);

                    // Remove any existing welcome message or an already added/old outbound campaign message
                    mController.amsMessages.removeLastWelcomeMessage(true).execute();
                    mController.amsMessages.removeLastOutboundMessage().execute();

                    LPLog.INSTANCE.d(TAG, "displayOutboundCampaignMessage: Adding new outbound campaign welcome message.");
                    mController.amsMessages.addMessage(messagingChatMessage, true).execute();
                } else if (!newOutboundMessage.equals(storedOutboundMessage)) {
                    // Update outbound message if we got new one
                    long messageTimestamp = System.currentTimeMillis();
                    MessagingChatMessage messagingChatMessage = buildWelcomeMessagingChatMessage(brandId,
                            messageTimestamp, newOutboundMessage, AmsMessages.OUTBOUND_CAMPAIGN_MSG_SEQUENCE_NUMBER);

                    LPLog.INSTANCE.d(TAG, "displayOutboundCampaignMessage: Updating existing outbound campaign welcome message with new one.");
                    mController.amsMessages.updateLastOutboundMessage(messagingChatMessage).execute();
                }
            }).execute();
        } else {
            // Clear existing outbound campaign message if we have one stored in database but nothing
            // in shared preferences (may be expired one)
            LPLog.INSTANCE.d(TAG, "displayOutboundCampaignMessage: Removing existing outbound welcome message");
            mController.amsMessages.removeLastOutboundMessage().execute();
        }
    }

    /**
     * Build a new temporary welcome message/Campaign for consumer. This could be Brand set welcome message or
     * outbound campaign message.
     * @param brandId respective brand's account id
     * @param messageTimestamp current time
     * @param message message text
     * @param messageSequence message sequence number (Currently -4 or -5)
     * @return chat message objects
     */
    private MessagingChatMessage buildWelcomeMessagingChatMessage(String brandId, long messageTimestamp, String message, int messageSequence) {
        String conversationId = AmsConversations.KEY_WELCOME_CONVERSATION_ID;
        String dialogId = AmsDialogs.KEY_WELCOME_DIALOG_ID;
        String originatorId = "";
        String eventId = messageTimestamp + "_" + messageSequence;


        mController.amsConversations.createDummyConversationForWelcomeMessage(brandId, conversationId, messageTimestamp);
        mController.amsDialogs.createDummyDialogForWelcomeMessage(brandId, conversationId, dialogId, messageSequence, messageTimestamp);

        MessagingChatMessage messagingChatMessage = new MessagingChatMessage(originatorId, message, messageTimestamp, dialogId, eventId,
                MessagingChatMessage.MessageType.BRAND, MessagingChatMessage.MessageState.RECEIVED, EncryptionVersion.NONE);
        messagingChatMessage.setServerSequence(messageSequence);
        return messagingChatMessage;
    }

    public void updateParticipants(final String targetId, String[] participants, final UserProfile.UserType userType,
                                   final String conversationId, final boolean updateUI, final boolean forceUpdate) {
        updateParticipants(targetId, participants, userType, conversationId, updateUI, forceUpdate, null);
    }

    public void updateParticipants(final String targetId, String[] participants, final UserProfile.UserType userType,
                                   final String conversationId, final boolean updateUI, final boolean forceUpdate, final ICallback<MessagingUserProfile, Exception> callback) {
        for (final String userId : participants) {
            if (!TextUtils.isEmpty(userId)) {
                //foreach participant - if does not exist in db - send getUserProfile request
                mController.amsUsers.getUserById(userId).setPostQueryOnBackground(userProfile -> {
                    if (userProfile == null) {
                        userProfile = new MessagingUserProfile("", "", userType);
                        userProfile.setOriginatorID(userId);
                        mController.amsUsers.updateUserProfile(userProfile);

	                    LPLog.INSTANCE.i(TAG, "First time bringing information for another participant that joined conversation " + conversationId);
                        sendUpdateUserRequest(targetId, userId, conversationId, updateUI, callback);
                    } else if (userProfile.isEmptyUser() || forceUpdate) {
                        sendUpdateUserRequest(targetId, userId, conversationId, updateUI, callback);
                    } else if (callback != null) {
                        callback.onSuccess(userProfile);
                    }
                }).execute();
            }
        }
    }

    /**
     * @param targetId
     * @param userId
     * @param conversationId
     * @param shouldUpdateUi
     * @param callback
     */
    private void sendUpdateUserRequest(String targetId, String userId, String conversationId, boolean shouldUpdateUi, ICallback<MessagingUserProfile, Exception> callback) {
        if (!TextUtils.isEmpty(userId)) {
            // There is an assigned agent, get his details and update
            new AgentProfileRequest(mController, targetId, userId, conversationId, shouldUpdateUi).setCallback(callback).execute();
        } else {
            if (TextUtils.isEmpty(conversationId)){
                LPLog.INSTANCE.d(TAG, "sendUpdateUserRequest: no dialog id");
                return;
            }

            // There is no assigned agent, get the conversation from the DB and update the callback
            mController.amsConversations.queryConversationById(conversationId)
                    .setPostQueryOnBackground(data -> {
	                    LPLog.INSTANCE.d(TAG, "onResult: Calling agent details callback with null agent");
                        mController.onAgentDetailsChanged(null, data.isConversationOpen());
                    }).execute();
        }

    }

    public void updateTTR(TTRType type, long effectiveTTR, String targetId) {
	    LPLog.INSTANCE.d(TAG, "update TTR type to - " + type + ". EffectiveTTR = " + effectiveTTR);
        mController.amsConversations.updateTTRType(targetId, type, effectiveTTR);

    }

	public long calculateEffectiveTTR(final String targetId, final long ttrValue, final long manualTTR, final long delayTTR) {
		long clockDiff = mController.mConnectionController.getClockDiff(targetId);

		return mController.amsConversations.calculateEffectiveTTR(targetId, ttrValue, manualTTR, delayTTR, clockDiff);
	}

	public static void showTTR(final Messaging controller, String targetId) {
    	controller.amsConversations.showTTR(targetId);
	}
}
