package com.liveperson.messaging.network.socket.requests;

import android.text.TextUtils;

import com.liveperson.api.request.BaseAMSSocketRequest;
import com.liveperson.api.request.ConsumerRequestConversation;
import com.liveperson.api.request.ReqBody;
import com.liveperson.api.request.RequestConversation;
import com.liveperson.api.response.types.TTRType;
import com.liveperson.infra.CampaignInfo;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.network.socket.BaseResponseHandler;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.model.ConversationData;
import com.liveperson.messaging.model.MessagingChatMessage;

import org.json.JSONException;
import org.json.JSONObject;

import static com.liveperson.infra.errors.ErrorCode.ERR_000000AB;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000AC;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000AD;

/**
 * Created by Ilya Gazman on 11/11/2015.
 * <p/>
 * Request for creating new conversation
 */
public class NewConversationRequest extends BaseAMSSocketRequest<RequestConversation.Response, NewConversationRequest> {
    private static final String TAG = "NewConversationRequest";
    private final Messaging mController;
    private final String mBrandId;
    private final String mTargetId;
    private String appId = "";
    private TTRType ttrType = TTRType.NORMAL;
    private String mTempConversationId;
    private String mTempDialogId;

    public NewConversationRequest(Messaging controller, String targetId, String brandId, String tempConversationId, String tempDialogId){
        super(controller.mAccountsController.getConnectionUrl(brandId));
        mController = controller;
        mBrandId = brandId;
        mTargetId = targetId;
        mTempConversationId = tempConversationId;
        mTempDialogId = tempDialogId;
    }

    /**
     * Sets app id
     *
     * @param appId
     * @return
     */
    public NewConversationRequest setAppId(String appId) {
        this.appId = appId;
        return this;
    }

    /**
     * sets time to response type
     *
     * @param ttrType
     * @return
     */
    public NewConversationRequest setTtrType(TTRType ttrType) {
        this.ttrType = ttrType;
        return this;
    }

    @Override
    protected String getData() {
    	// Try to get CampaignInfo from the AccountController
		CampaignInfo campaignInfo = mController.getConversationViewParams().getCampaignInfo();
		LPLog.INSTANCE.d(TAG, "getData: campaignInfo = " + campaignInfo);

		return new ConsumerRequestConversation(appId, ttrType, mBrandId, "", campaignInfo).toJsonString(getRequestId());
    }

    @Override
    protected String getRequestName() {
        return TAG;
    }

    @Override
    protected BaseResponseHandler<RequestConversation.Response, NewConversationRequest> getResponseHandler() {
        return new BaseResponseHandler<RequestConversation.Response, NewConversationRequest>() {

            @Override
            public String getAPIResponseType() {
                return RequestConversation.Response.REQUEST_CONVERSATION_RESPONSE_TYPE;
            }

            @Override
            protected RequestConversation.Response parse(JSONObject jsonObject) throws JSONException {
                return new RequestConversation.Response(jsonObject);
            }

            @Override
            protected boolean handle(final RequestConversation.Response data) {
                // This will be handled only at the event, unless we have error response
                if (data.code >= 200 && data.code < 300) {
                    LPLog.INSTANCE.d(TAG, "Succeeded, create conversation response code: " + data.code);
                } else {
                    LPLog.INSTANCE.e(TAG, ERR_000000AB, "Error, create conversation response code: " + data.code);
                }

                return true;
            }

            @Override
            protected void giveUp() {
                super.giveUp();
                if (isConversationWaitingForResponse()){

                    LPLog.INSTANCE.e(TAG, ERR_000000AC, getRequestId() + ": Request lost (socket closed) for newConversationRequest, Changing all messages of this conversation to error state");
                    failConversationAnsMessages();
                }
            }

            @Override
            public BaseResponseHandler getResponseByExpectedType(String expectedType) {
                BaseResponseHandler supportedResponseHandler = getSupportedResponseHandler(expectedType);
                if (supportedResponseHandler != null){
                    return supportedResponseHandler;
                }
                return this;
            }
            //http://qtvr-wng113/api/account/le52821091/app/<ACR0 connectorId (generated via houston)>/authenticate?v=3.0



            @Override
            protected BaseResponseHandler getSupportedResponseHandler(String messageType) {
                if (TextUtils.equals(messageType, ReqBody.StringResp.REQ_BODY_RESPONSE_TYPE)){
                    return new BaseResponseHandler<ReqBody.StringResp,NewConversationRequest>() {

                        @Override
                        protected ReqBody.StringResp parse(JSONObject jsonObject) throws JSONException {
                            return new ReqBody.StringResp(jsonObject);
                        }

                        @Override
                        public String getAPIResponseType() {
                            return ReqBody.StringResp.REQ_BODY_RESPONSE_TYPE;
                        }

                        @Override
                        protected boolean handle(ReqBody.StringResp response) {
                            LPLog.INSTANCE.i(TAG, "Received String response (" + response.code + ").");

                            if (response.code >= 400) {
                                LPLog.INSTANCE.e(TAG, ERR_000000AD, "Bad response (" + response.code + ") for newConversationRequest, Changing all messages of this conversation to error state\nString response: " + LPLog.INSTANCE.mask(response));
                                failConversationAnsMessages();
                                return true;
                            }
                            return false;
                        }
                    };
                }
                return super.getSupportedResponseHandler(messageType);
            }
        };
    }

    private void failConversationAnsMessages() {
        final ConversationData conversationData = new ConversationData();
        conversationData.requestId = getRequestId();
        conversationData.conversationId = mTempConversationId;
        conversationData.brandId = mBrandId;
        conversationData.targetId = mTargetId;

        mController.amsConversations.updateClosedConversation(conversationData, false).execute();
        mController.amsDialogs.updateClosedDialog(conversationData, false).execute();
        mController.amsMessages.updateAllMessagesStateByDialogId(mTempDialogId, MessagingChatMessage.MessageState.ERROR);
    }

    private boolean isConversationWaitingForResponse() {
        return mController.amsConversations.getConversation(mTargetId) != null &&
                mController.amsConversations.getConversation(mTargetId).getConversationId().equals(mTempConversationId);
    }
}
