package com.liveperson.api.response.model;

import android.webkit.URLUtil;

import com.liveperson.api.request.message.FilePublishMessage;
import com.liveperson.api.request.message.JsonPublishMessage;
import com.liveperson.api.request.message.BasePublishMessage;
import com.liveperson.api.request.message.FormPublishMessage;
import com.liveperson.api.request.message.FormSubmissionPublishMessage;
import com.liveperson.api.request.message.StructuredContentPublishMessage;
import com.liveperson.api.request.message.TextPublishMessage;
import com.liveperson.api.response.types.DeliveryStatus;
import com.liveperson.api.exception.BadMessageException;

import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.model.types.ChatState;
import com.liveperson.messaging.MessagingFactory;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by oferd on 8/10/15.
 */
public class Event {

	private static final String TAG = "Event";
	private static final String CONTENT = "content";

	public Types type;

	/**
	 * type : ContentEvent
	 */
	public String contentType;

	// The message can contain either a simple string of a json object
	public BasePublishMessage message;

	/**
	 * type : ChatStateEvent
	 */
	public ChatState chatState;

	/**
	 * type : AcceptStatusEvent
	 */
	public DeliveryStatus status;
	public int[] sequenceList;

	/**
	 * Holds QuickReplies JSON string (if exist in the event)
	 */
	public String quickRepliesJsonString;

	public enum Types {
		AcceptStatusEvent,
		ChatStateEvent,
		ContentEvent,
		RichContentEvent
	}

	/**
	 * extractLinks - a method to extract only the url from the whole message
	 * for later parse
	 *
	 * @param text - the original msg
	 * @return String[] of all the links
	 */
	private String[] extractLinks(String text) {
		text = text.replaceAll("\u200B", "");
		text = text.replaceAll("\u200C", "");
		text = text.replaceAll("\u200D", "");
		text = text.replaceAll("\uFEFF", "");
		text = text.trim();
		text = text.replaceAll("\n", "");

		List<String> links = new ArrayList<>();
		String[] l = text.split("\\s+");
		for (int i = 0; i < l.length; i++) {
			if (URLUtil.isValidUrl(l[i])) {
				links.add(l[i]);
			}
		}
		return links.toArray(new String[links.size()]);
	}

	public Event(JSONObject jsonObject) throws JSONException,BadMessageException {
		if (jsonObject == null) {
			LPLog.INSTANCE.w(TAG, "No EVENT content!");
			return;
		}

		type = Types.valueOf(jsonObject.getString("type"));
		switch (type) {
			case AcceptStatusEvent: {
				String statusString = jsonObject.optString("status");
				try {
					this.status = DeliveryStatus.valueOf(statusString);

				}catch (IllegalArgumentException e){
					//could be type 'unknown', if so we ignore the status event
					throw new BadMessageException("Bad Accept Status: "+ statusString);
				}

				JSONArray list = jsonObject.optJSONArray("sequenceList");
				if (list == null) {
					list = new JSONArray();
				}

				sequenceList = new int[list.length()];
				for (int i = 0; i < list.length(); i++) {
					sequenceList[i] = list.getInt(i);
				}
				break;
			}
			case ChatStateEvent: {
				String chatStateString = jsonObject.optString("chatState", "GONE");
				chatState = ChatState.valueOf(chatStateString);
				break;
			}
			case ContentEvent: {
				contentType = jsonObject.optString("contentType");

				ContentType contentTypeEnum = ContentType.fromString(contentType);

				if (contentTypeEnum.isFile()) {
					JSONObject messageObject = jsonObject.optJSONObject("message");
					if (messageObject != null) {
						LPLog.INSTANCE.d(TAG, "Event: parsing file message");
						message = new FilePublishMessage(messageObject);
					}
				} else if (contentTypeEnum.isSimpleText()) {
					String str = jsonObject.optString("message");
					String[] urls = extractLinks(str);
					if (urls.length <= 0) {
						message = new TextPublishMessage(jsonObject.optString("message"));
					} else {
						LPLog.INSTANCE.d(TAG, "Event: parsing url message");
						message = new JsonPublishMessage(str, urls[0]);
					}
				} else if (contentTypeEnum.isFormInvitation()) {
					JSONObject messageObject = jsonObject.optJSONObject("message");
					if (messageObject != null) {
						LPLog.INSTANCE.d(TAG, "Event: parsing form invitation message");
						message = new FormPublishMessage(messageObject);
					}
				} else if (contentTypeEnum.isFormSubmission()) {
					JSONObject messageObject = jsonObject.optJSONObject("message");
					if (messageObject != null) {
						LPLog.INSTANCE.d(TAG, "Event: parsing form Submission message");
						message = new FormSubmissionPublishMessage(messageObject);
					}
				} else {
					LPLog.INSTANCE.w(TAG, "Event: received an unsupported message type");
					message = new TextPublishMessage("Message Type Not supported");
				}

				// Get the QuickReplies string from the JSON
				parseQuickReplies(jsonObject);

				break;
			}
			case RichContentEvent:{
				// Parse rich content event only if the Structured Content is enabled in branding
				if(MessagingFactory.getInstance().getController().isEnableStructuredContent()) {
					LPLog.INSTANCE.d(TAG, "Event: parsing RichContentEvent");
					message = new StructuredContentPublishMessage(jsonObject.optString(CONTENT));

					parseQuickReplies(jsonObject);
				}
				else{
					LPLog.INSTANCE.d(TAG, "Event: got RichContentEvent but was not parsed since the feature is disabled in branding");
				}
				break;
			}
		}
	}

	/**
	 * Get the QuickReplies JSON string from the given JSON
	 * @param jsonObject
	 */
	private void parseQuickReplies(JSONObject jsonObject) {
		JSONObject quickRepliesJsonObject = jsonObject.optJSONObject("quickReplies");
		if (quickRepliesJsonObject != null) {
			quickRepliesJsonString = quickRepliesJsonObject.toString();
		}
	}
}
