package com.liveperson.messaging.structuredcontent.model.elements;


import android.support.annotation.Nullable;

import com.liveperson.messaging.structuredcontent.model.actions.BaseAction;
import com.liveperson.messaging.structuredcontent.model.actions.PublishTextAction;
import com.liveperson.messaging.structuredcontent.parsers.ActionParser;
import com.liveperson.messaging.structuredcontent.parsers.ElementType;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;

/**
 * Created by nirni on 2/13/17.
 * This is the abstract base class for all actionable elements (e.g. ButtonElement)
 */
public abstract class ActionableElement extends SimpleElement {

	/**
	 * Metadata json array that needs to be sent to UMS upon click action
 	 */
	private JSONArray mMetadata;

	/**
	 * The action for this element
	 */
	private List<BaseAction> mActionList;

	// Comparator to order the actionable element according to importance order (PublishText be first)
	private static Comparator<BaseAction> comparator = (left, right) -> left instanceof PublishTextAction ? -1 : 0;


	public ActionableElement(String type) {
		super(type);
	}

	public ActionableElement(JSONObject jsonObject) throws JSONException {
		super(jsonObject);

		// Parse the whole click json object
		parseClick(jsonObject);
	}

	private void parseClick(JSONObject jsonObject) {

		JSONObject clickObject = jsonObject.optJSONObject(ElementType.CLICK);
		if (clickObject == null) {
			return;
		}

		// Get the metadata object
		mMetadata = clickObject.optJSONArray(ElementType.METADATA);

		// Get the actions array
		parseActions(clickObject);
	}

	/**
	 * Parse the Actions json
	 * @param jsonObject
	 */
	private void parseActions(JSONObject jsonObject) {

		// Parse and set the action. If the JSONObject is null, the parse will deal with it
		JSONArray actionArray = jsonObject.optJSONArray(ElementType.ACTIONS);

		if (actionArray != null) {
			mActionList = ActionParser.parse(actionArray, mMetadata);

			// If the action list has more than one item we sort it to make the actions that need to be
			// executed first
			if (mActionList != null && mActionList.size() > 1) {
				Collections.sort(mActionList, comparator);
			}
		}
	}

	public List<BaseAction> getActions() {
		return mActionList;
	}

	/**
	 * If link or navigate action found, return link as action otherwise return button as action.
	 * @return Action tobe announced by accessibility
	 */
	public String getAccessibilityActionString() {
		if (mActionList != null && mActionList.size() > 0) {
			if (containsElementType(ElementType.LINK) || containsElementType(ElementType.NAVIGATE)) {
				return ElementType.LINK;
			} else if (containsElementType(ElementType.PUBLISH_TEXT)) {
				return ElementType.BUTTON;
			}
		}
		return "";
	}

	private boolean containsElementType(String elementType) {
		for (int i = 0; i < mActionList.size(); i++) {
			String actionType = mActionList.get(i).getType();
			if (actionType.equals(elementType)) {
				return true;
			}
		}
		return false;
	}

	@Nullable
	public JSONArray getMetadata() {
		return mMetadata;
	}
}
