package com.liveperson.messaging.background;

import android.graphics.Bitmap;
import android.media.ExifInterface;
import androidx.annotation.NonNull;

import com.liveperson.infra.Infra;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.messaging.R;
import com.liveperson.infra.utils.ImageUtils;
import com.liveperson.messaging.MessagingFactory;
import com.liveperson.messaging.background.filesharing.BaseUploadTask;
import com.liveperson.messaging.background.filesharing.UploadFileTaskBundle;
import com.liveperson.messaging.background.filesharing.image.UploadImageTaskBundle;
import com.liveperson.messaging.commands.SendImageCommand;
import com.liveperson.messaging.exception.FileSharingException;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;

/**
 * Created by shiranr on 7/24/16.
 */
public class UploadImageTask  extends BaseUploadTask {

    private static final String TAG = "UploadImageTask";
    protected String mBase64 = null;

    protected UploadImageTaskBundle uploadImageTaskParams;
	private byte[] mOriginalFileByteArray = null;

    /**
     * creating bitmaps (thumbnail and full size)
     * @param params
     */
    public UploadImageTask(UploadImageTaskBundle params, Integer uploadTimeout) throws FileSharingException {
    	super(uploadTimeout);

		int orientation = 0;
		ExifInterface exif;

		if(params == null){
			throw new FileSharingException("Params is null");
		}

        uploadImageTaskParams = params;

        LPMobileLog.d(TAG + Thread.currentThread(), "imageTypeExtension = " + uploadImageTaskParams.getFileTypeExtension() + ", imageContentType = " + uploadImageTaskParams.getFileContentType() );
        if(uploadImageTaskParams.getFileTypeExtension().equals("GIF")){
            throw new FileSharingException("This file type is not supported");
        }

        try {
            // Save the original orientation because picasso losing this information during image resize
			try {
				exif = new ExifInterface(uploadImageTaskParams.getFilePath());
				orientation = exif.getAttributeInt(ExifInterface.TAG_ORIENTATION, ExifInterface.ORIENTATION_NORMAL);
			} catch (FileNotFoundException e) {
				LPMobileLog.w(TAG, "UploadImageTask: cannot get orientation from exif. Using default (0)");
			}

            final Bitmap thumbnailBitmap = processThumbnailImage(orientation, uploadImageTaskParams.isImageFromCamera());
            final Bitmap originalBitmap = processOriginalImage(orientation, uploadImageTaskParams.isImageFromCamera());


            // We support only JPG and PNG image types, so if the imageTypeExtension is not one of those we change it to jpg
            if (!uploadImageTaskParams.getFileTypeExtension().equalsIgnoreCase(ImageUtils.JPG) && !uploadImageTaskParams.getFileTypeExtension().equalsIgnoreCase(ImageUtils.PNG)) {
                uploadImageTaskParams.setFileTypeExtension(ImageUtils.JPG.toUpperCase());
            }

            final String thumbnailPath = saveThumbnailImage(thumbnailBitmap);
            final String fullImagePath =  saveOriginalImage(originalBitmap);

            LPMobileLog.d(TAG, "UploadImageTask: Image and thumbnail created. Continue to send image to server");

            createSendMessageCommand(thumbnailPath, fullImagePath);


        } catch (IOException e) {
            LPMobileLog.e(TAG + Thread.currentThread(), "error resizing or saving image");
            onUploadFailed(e);
            throw new FileSharingException("error resizing or saving image");
        }

    }

	@Override
	protected UploadFileTaskBundle getUploadTaskBundle() {
		return uploadImageTaskParams;
	}

	@Override
	protected byte[] getFileByteArray() {
		return mOriginalFileByteArray;
	}

	@Override
	public int getTaskId() {
		return uploadImageTaskParams.getTaskId();
	}

	@Override
	public String getThumbnailBase64() {
		return mBase64;
	}

	@NonNull
    protected Bitmap processOriginalImage(int orientation, boolean fromCamera) throws IOException, FileSharingException {
        Bitmap originalBitmap;// Resize the full image
        int fullImageLongerDimensionResize = Configuration.getInteger(R.integer.full_image_longer_dimension_resize);
        originalBitmap = ImageUtils.createResizedBitmap(uploadImageTaskParams.getFileUri(), fullImageLongerDimensionResize, orientation, fromCamera);
        LPMobileLog.d(TAG + Thread.currentThread(), "originalBitmap size (w, h): " + originalBitmap.getWidth() + ", " + originalBitmap.getHeight());
        return originalBitmap;
    }

    @NonNull
    protected String saveOriginalImage(Bitmap originalBitmap) throws FileSharingException {
        // Create a byte array of the image. If the image is NOT from the camera, save it to local folder
        final String fullImagePath = convertAndSaveFullImage(originalBitmap, uploadImageTaskParams.getFilePath(), uploadImageTaskParams.isImageFromCamera(), uploadImageTaskParams.getFileTypeExtension());
        if (fullImagePath == null) { // File is too large (after compression)
            LPMobileLog.d(TAG, "UploadImageTask: file is too large after compression");
            throw new FileSharingException("File is too large after compression");
        }
        return fullImagePath;
    }

    @NonNull
    protected Bitmap processThumbnailImage(int orientation, boolean fromCamera) throws IOException, FileSharingException {
        Bitmap thumbnailBitmap;// Create a thumbnailBitmap
        int thumbnailLongerDimensionResize = Configuration.getInteger(R.integer.thumbnail_longer_dimension_resize);
        thumbnailBitmap = ImageUtils.createResizedBitmap(uploadImageTaskParams.getFileUri(), thumbnailLongerDimensionResize, orientation, fromCamera);
        LPMobileLog.d(TAG + Thread.currentThread(), "thumbnailBitmap size: " + thumbnailBitmap.getWidth() + ", " + thumbnailBitmap.getHeight());
        return thumbnailBitmap;
    }

    @NonNull
    protected String saveThumbnailImage(Bitmap thumbnailBitmap) throws FileSharingException {
        // Create and save a thumbnail to the local folder
        final String thumbnailPath = saveThumbnail(thumbnailBitmap, uploadImageTaskParams.getFileTypeExtension());
        if (thumbnailPath == null) {
            LPMobileLog.e(TAG + Thread.currentThread(), "run: Could not save image thumbnail to disk");
            throw new FileSharingException("Could not save thumbnailBitmap image to disk");
        }
        return thumbnailPath;
    }

    private String saveThumbnail(Bitmap mThumbnail, String imageTypeExtension) {

		// We support only JPG and PNG image types, so if the imageTypeExtension is not one of those we change it to jpg
		if (!imageTypeExtension.equals(ImageUtils.JPG) && !imageTypeExtension.equals(ImageUtils.PNG)) {
			imageTypeExtension = ImageUtils.JPG;
		}

		byte [] thumbnailByteArray = ImageUtils.getOutputStreamFromBitmap(mThumbnail, 100, imageTypeExtension);

		// If thumbnail was not created successfully return null
		if (thumbnailByteArray == null) {
			return null;
		}

		// Encode thumbnail's base64 string
		mBase64 = ImageUtils.bitmapToBase64(thumbnailByteArray);
        LPMobileLog.d(TAG + Thread.currentThread(), "run: Thumbnail Base64: " + mBase64);

        // Save the bitmap to disk and get its path
        return ImageUtils.saveBitmapToDisk(Infra.instance.getApplicationContext(), thumbnailByteArray, uploadImageTaskParams.getBrandId(), ImageUtils.ImageFolderType.PREVIEW, imageTypeExtension);
    }

	/**
	 * Create an output stream from the given bitmap.
	 * @param mOriginalBitmap
	 * @param originalImagePath
	 * @param saveOnOriginalPath  @return
	 * @param imageTypeExtension
	 */
	private String convertAndSaveFullImage(Bitmap mOriginalBitmap, String originalImagePath, boolean saveOnOriginalPath, String imageTypeExtension) {

        createOriginalImageByteArray(mOriginalBitmap, imageTypeExtension);


        // If the full image was copied successfully to private folder return null
		if (mOriginalFileByteArray == null) {
			return null;
		}

		LPMobileLog.d(TAG, "convertAndSaveFullImage: size of full image file (after compression): " + mOriginalFileByteArray.length);

		// Check if the image after compression is larger than allowed
		int maxImageSizeKb = Configuration.getInteger(R.integer.max_image_size_kb);
		if (mOriginalFileByteArray.length > maxImageSizeKb * 1000) {
			return null;
		}

		if(saveOnOriginalPath) {
			return ImageUtils.saveBitmapToDisk(mOriginalFileByteArray, new File(originalImagePath));
		}
		else{
			return ImageUtils.saveBitmapToDisk(Infra.instance.getApplicationContext(), mOriginalFileByteArray, uploadImageTaskParams.getBrandId(), ImageUtils.ImageFolderType.FULL, imageTypeExtension);
		}
	}

    protected void createOriginalImageByteArray(Bitmap mOriginalBitmap, String imageTypeExtension) {
        // Get the compression rate for the full image from the resources
        int compressionRate = Configuration.getInteger(R.integer.full_image_compression_rate);

        LPMobileLog.d(TAG, "convertAndSaveFullImage: compression rate for full image: " + compressionRate);

        // Get byte array of the full image bitmap
        mOriginalFileByteArray = ImageUtils.getOutputStreamFromBitmap(mOriginalBitmap, compressionRate, imageTypeExtension);
    }

    protected void createSendMessageCommand(String thumbnailPath, String fullImagePath) {
        sendMessageCommand = new SendImageCommand(MessagingFactory.getInstance().getController(), uploadImageTaskParams.getTargetId(), uploadImageTaskParams.getBrandId(),
                uploadImageTaskParams.getFileContentType(), thumbnailPath, fullImagePath,uploadImageTaskParams.getFileTypeExtension(), uploadImageTaskParams.getMessage());
        setSendMessageCommandCallback();
    }

/*
    protected void setSendMessageCommandCallback() {
        sendMessageCommand.setCallback(new SendFileCommand.SendFileCommandListener() {

            */
/**
             * Callback after calling sendMessageCommand.addMessageToDB();
             * @param messageRowId
             * @param fileRowId - image file rowId
             *//*

            @Override
            public void onFileAddedToDB(long messageRowId, long fileRowId) {
                mFileRowId = fileRowId;
                mMessageRowId = messageRowId;
                if (mUploadFileTaskCallback != null) {
                    mUploadFileTaskCallback.onFileAddedToDB();
                }
            }

            */
/**
             * after connecting to server and getting updated- we update the message in DB.
             * this callback will run after updating the message in DB.
             *//*

            @Override
            public void onMessageUpdatedInDB() {
                //after updating message (with new conversation id) we can upload the image to swift server.
                //starting by generate url from UMS.
                generateUrl();
            }

        });
    }
*/

/*
    */
/**
     * Adding message and file to db, when completed- will run the callback's method- onFileAddedToDB.
     *//*

    public void startUpload() {
		if (sendMessageCommand != null) {
			sendMessageCommand.addMessageToDB();
		}
	}

    public void onConnectionAvailable() {
        LPMobileLog.d(TAG + Thread.currentThread(), "onConnectionAvailable");
        if (mStatus == FilesTable.LoadStatus.NOT_STARTED){
            setStatus(FilesTable.LoadStatus.PROCESSING);
            sendMessageCommand.updateMessageConversationId();
        }
    }
*/

/*
    public void setCallBack(UploadFileTaskCallback uploadFileTaskCallback) {
        mUploadFileTaskCallback = uploadFileTaskCallback;
    }
*/

/*
    private void generateUrl() {
        setStatus(FilesTable.LoadStatus.REQUESTING_URL);

        LPMobileLog.d(TAG + Thread.currentThread(), "generateUrlRunnable");
        SocketManager.getInstance().send(new GetUrlForUploadRequest(MessagingFactory.getInstance().getController(),
                uploadImageTaskParams.getBrandId(), mOriginalFileByteArray.length, uploadImageTaskParams.getFileTypeExtension(),
                new IOnUrlReady() {
                    @Override
                    public void onUrlReady(final BaseGenerateURLResponse body) {
                        LPMobileLog.d(TAG + Thread.currentThread(), "URL ready!!" + body.relativePath );
                        //updating file relativePath on DB\
                        mRelativePath = body.relativePath;
                        MessagingFactory.getInstance().getController().amsFiles.updateRelativePath(mFileRowId, mRelativePath);

                        if (isUploadFailed()){
                            return;
                        }
                        uploadToSwift(body.queryParams);
                    }

                    @Override
                    public void onUrlError(String errorMessage) {
                        if (isUploadFailed()){
                            return;
                        }
                        onUploadFailed(new Exception("failed to generate url." + errorMessage));
                    }
                }));
    }
*/

/*
    private void uploadToSwift(QueryParams queryParams) {
        LPMobileLog.d(TAG + Thread.currentThread(), "uploading to swift..");

        setStatus(FilesTable.LoadStatus.UPLOADING);
        // TODO: 7/24/16 extract "swift" to constant

        UploadFileRequest mUploadImageRequest = new UploadFileRequest(uploadImageTaskParams.getSwiftDomain(), mRelativePath, queryParams, uploadImageTaskParams.getRestParams().mCertificates, new ICallback() {
            @Override
            public void onSuccess(Object value) {
                if (isUploadFailed()) {
                    return;
                }
                LPMobileLog.d(TAG + Thread.currentThread(), "uploading to swift succeeded!");
                setStatus(FilesTable.LoadStatus.COMPLETED);
                if (mUploadFileTaskCallback != null) {
                    mUploadFileTaskCallback.onUploadFinishedSuccessfully(UploadImageTask.this);
                }
                //sendPublishFile will be called by the manager after old waiting image-messages
            }

            @Override
            public void onError(Throwable exception) {
                if (isUploadFailed()) {
                    return;
                }
                onUploadFailed(new Exception("failed to upload to swift " + exception.getMessage()));

            }
        }, mOriginalFileByteArray);
        mUploadImageRequest.execute();
    }
*/

/*
    public void onConnectionUnavailable() {
        onUploadFailed(new Exception("Failed to upload. connection unavailable"));
    }
    private void onUploadFailed(Throwable exception) {
        LPMobileLog.d(TAG , "onUploadFailed. "+ exception.getMessage());
        if (sendMessageCommand != null){
            sendMessageCommand.failMessage();
        }
        setStatus(FilesTable.LoadStatus.FAILED);

        if (mUploadFileTaskCallback != null) {
            mUploadFileTaskCallback.onUploadFailed(this, exception);
        }

    }
*/

 /*   private void notifyChange() {
        MessagingFactory.getInstance().getController().amsMessages.updateFileMessageByRowId(mMessageRowId, mFileRowId).execute();
    }

    protected void sendPublishFile(boolean sendViaRest) {
        LPMobileLog.d(TAG + Thread.currentThread(), "sending PublishImage request..");
        sendMessageCommand.setFileDetails(mRelativePath, uploadImageTaskParams.getFileTypeExtension(),mBase64);
        sendMessageCommand.setSendViaRest(sendViaRest, uploadImageTaskParams.getRestParams());
        sendMessageCommand.execute();
    }

    public void setStatus(FilesTable.LoadStatus status) {
        this.mStatus = status;
        LPMobileLog.d(TAG + Thread.currentThread(), "set file status: "+ status +" mFileRowId = " + mFileRowId);
        if (mFileRowId != -1){
            MessagingFactory.getInstance().getController().amsFiles.updateStatus(mFileRowId, mStatus);
        }
        notifyChange();
    }

	public long getMessageRowId() {
		return mMessageRowId;
	}

	public boolean isUploadCompleted() {
        return mStatus == FilesTable.LoadStatus.COMPLETED;
    }
    public boolean isUploadFailed() {
        return mStatus == FilesTable.LoadStatus.FAILED;
    }

    public String getEventId() {
        return (sendMessageCommand != null ? sendMessageCommand.getEventId() : null );

    }
*/
	public UploadImageTaskBundle getUploadImageTaskParams() {
		return uploadImageTaskParams;
	}
}
